<?php

namespace Intucart\Services;

use Intufind\AI\Client as IntufindAIClient;
use Intufind\AI\Config\Configuration as IntufindAIConfig;
use Intucart\Services\Constants;
use Intucart\Services\Cache\CacheService;
use Intufind\AI\Config\Configuration;

/**
 * Centralized AI Client Manager
 * 
 * Manages the creation and configuration of the Intufind AI SDK client
 * to eliminate duplicate initialization code across the application.
 */
class AIClientManager
{
    private ?IntufindAIClient $client = null;
    private ?IntufindAIConfig $config = null;
    private CacheService $cacheService;
    
    // Cache configuration
    private const CACHE_GROUP = 'intucart_ai_client';
    private const LICENSE_CACHE_KEY = 'license_validation';
    private const CONNECTION_CACHE_KEY = 'connection_test';
    private const CACHE_EXPIRATION = 300; // 5 minutes for license validation
    private const CONNECTION_CACHE_EXPIRATION = 600; // 10 minutes for connection test
    
    // Request-level cache to prevent multiple API calls during same request
    private ?array $licenseValidationCache = null;
    private ?bool $connectionTestCache = null;

    /**
     * Constructor
     *
     * @param CacheService $cacheService Cache service for persistent caching
     */
    public function __construct(CacheService $cacheService)
    {
        $this->cacheService = $cacheService;
    }

    /**
     * Get the AI client instance (singleton pattern)
     *
     * @return IntufindAIClient
     */
    public function getClient(): IntufindAIClient
    {
        if ($this->client === null) {
            $this->client = new IntufindAIClient($this->getConfig());
        }

        return $this->client;
    }

    /**
     * Get the AI configuration (singleton pattern)
     *
     * @return IntufindAIConfig
     */
    public function getConfig(): IntufindAIConfig
    {
        if ($this->config === null) {
            $this->config = new IntufindAIConfig([
                'api_endpoint' => $this->getApiEndpoint(),
                'streaming_endpoint' => $this->getStreamingEndpoint(),
                'api_key' => $this->getApiKey(),
                'workspace_id' => $this->getWorkspaceId(),
                'debug' => $this->isDebugMode(),
                'timeout' => $this->getTimeout(),
                'retry_attempts' => $this->getRetryAttempts(),
            ]);
        }

        return $this->config;
    }

    /**
     * Create a new client with updated configuration
     *
     * @param array $configUpdates Configuration updates
     * @return IntufindAIClient
     */
    public function createClientWithConfig(array $configUpdates): IntufindAIClient
    {
        $config = $this->getConfig()->withUpdates($configUpdates);
        return new IntufindAIClient($config);
    }

    /**
     * Reset the client instance (useful for testing or config changes)
     *
     * @return void
     */
    public function resetClient(): void
    {
        $this->client = null;
        $this->config = null;
        $this->licenseValidationCache = null;
        $this->connectionTestCache = null;
        
        // Clear persistent cache as well
        $this->clearCache();
    }

    /**
     * Clear all AI client related caches
     *
     * @return void
     */
    public function clearCache(): void
    {
        // Clear persistent cache
        $this->cacheService->delete(self::LICENSE_CACHE_KEY, self::CACHE_GROUP);
        $this->cacheService->delete(self::CONNECTION_CACHE_KEY, self::CACHE_GROUP);
        
        // Clear request-level cache
        $this->licenseValidationCache = null;
        $this->connectionTestCache = null;
    }

    /**
     * Test the connection to the API (with caching)
     *
     * @return bool
     */
    public function testConnection(): bool
    {
        // Check request-level cache first to prevent multiple API calls during same request
        if ($this->connectionTestCache !== null) {
            return $this->connectionTestCache;
        }

        // Check persistent cache
        $cached_result = $this->cacheService->get(self::CONNECTION_CACHE_KEY, self::CACHE_GROUP);
        if ($cached_result !== false) {
            $this->connectionTestCache = (bool)$cached_result;
            return $this->connectionTestCache;
        }

        try {
            $result = $this->getClient()->testConnection();
            
            // Cache the result in both request-level and persistent cache
            $this->connectionTestCache = $result;
            $this->cacheService->set(
                self::CONNECTION_CACHE_KEY,
                $result,
                self::CACHE_GROUP,
                self::CONNECTION_CACHE_EXPIRATION
            );
            
            return $result;
        } catch (\Exception $e) {
            $result = false;
            $this->connectionTestCache = $result;
            
            // Cache negative results for a shorter time to avoid repeated failures
            $this->cacheService->set(
                self::CONNECTION_CACHE_KEY,
                $result,
                self::CACHE_GROUP,
                60 // 1 minute for failed connections
            );
            
            return $result;
        }
    }

    /**
     * Validate the license/API key
     * 
     * With the new API key auth model, validation is simply checking
     * if we have a valid API key configured. The API key is obtained
     * after successful license activation.
     *
     * @return bool
     */
    public function validateLicense(): bool
    {
        // Check class-level cache first
        if ($this->licenseValidationCache !== null) {
            return $this->licenseValidationCache['valid'] ?? false;
        }

        // With API key auth, validation = having a valid API key
        $hasApiKey = $this->hasApiKey();
        $this->licenseValidationCache = ['valid' => $hasApiKey];
        return $hasApiKey;
    }

    /**
     * Get license validation information
     * 
     * With the new API key auth model, this returns basic validation status
     * based on whether we have a valid API key configured.
     *
     * @return array
     */
    public function getLicenseValidation(): array
    {
        // Check request-level cache first
        if ($this->licenseValidationCache !== null) {
            return $this->licenseValidationCache;
        }

        // Check persistent cache
        $cached_result = $this->cacheService->get(self::LICENSE_CACHE_KEY, self::CACHE_GROUP);
        if ($cached_result !== false && is_array($cached_result)) {
            $this->licenseValidationCache = $cached_result;
            return $cached_result;
        }

        // With API key auth, validation = having a valid API key
        $hasApiKey = $this->hasApiKey();
        $result = [
            'valid' => $hasApiKey,
            'timestamp' => time(),
        ];

        if ($hasApiKey) {
            // Get tier from stored option (set during activation)
            $tier = get_option(Constants::TIER_OPTION, 'starter');
            $result['tier'] = $tier;
        }

        // Cache the result
        $this->licenseValidationCache = $result;
        
        if ($hasApiKey) {
            $this->cacheService->set(
                self::LICENSE_CACHE_KEY,
                $result,
                self::CACHE_GROUP,
                self::CACHE_EXPIRATION
            );
        }
        
        return $result;
    }

    /**
     * Check if a specific feature is available
     * 
     * With the new API key auth model, all features are available
     * if the user has a valid API key. Usage limits are enforced
     * server-side by the cloud.
     *
     * @param string $feature Feature name (e.g., 'chat', 'search', 'recommendations')
     * @return bool
     */
    public function hasFeature(string $feature): bool
    {
        // If we have a valid API key, features are available
        // The cloud enforces actual limits server-side
        return $this->hasApiKey();
    }

    /**
     * Check if within quota for a specific operation
     * 
     * With the new API key auth model, quota is enforced server-side.
     * The cloud will return 429 if quota is exceeded.
     *
     * @param string $operation Operation name
     * @param int $count Number of operations to check
     * @return bool
     */
    public function isWithinQuota(string $operation, int $count = 1): bool
    {
        // If we have a valid API key, assume within quota
        // The cloud enforces limits server-side and returns 429 if exceeded
        return $this->hasApiKey();
    }

    /**
     * Get version information
     *
     * @return array
     */
    public function getVersionInfo(): array
    {
        return $this->getClient()->getVersionInfo();
    }

    /**
     * Get API endpoint from wp-config.php constant (defined in intucart.php with default)
     *
     * @return string
     */
    private function getApiEndpoint(): string
    {
        return INTUFIND_API_URL;
    }

    /**
     * Get streaming endpoint from wp-config.php constant (defined in intucart.php with default)
     *
     * @return string
     */
    private function getStreamingEndpoint(): string
    {
        return INTUFIND_STREAMING_API_URL;
    }

    /**
     * Get cloud API key from WordPress options
     * This is the preferred authentication method for all API calls
     *
     * @return string
     */
    private function getApiKey(): string
    {
        $apiKey = trim(get_option(Constants::API_KEY_OPTION, ''));
        
        // Debug logging
        if ($this->isDebugMode() && empty($apiKey)) {
            error_log('AIClientManager: API key is empty. Will fall back to license key if available.');
        }
        
        return $apiKey;
    }

    /**
     * Get EDD license key from WordPress options
     * Used only for license activation/deactivation operations
     *
     * @return string
     */
    public function getLicenseKey(): string
    {
        return trim(get_option(Constants::LICENSE_KEY_OPTION, ''));
    }

    /**
     * Store API key in WordPress options
     * Called after successful license activation
     *
     * @param string $apiKey The API key to store
     * @return bool True on success
     */
    public function storeApiKey(string $apiKey): bool
    {
        return update_option(Constants::API_KEY_OPTION, $apiKey);
    }

    /**
     * Store tenant ID in WordPress options
     * Called after successful license activation
     *
     * @param string $tenantId The tenant ID to store
     * @return bool True on success
     */
    public function storeTenantId(string $tenantId): bool
    {
        return update_option(Constants::TENANT_ID_OPTION, $tenantId);
    }

    /**
     * Clear stored API key, tenant ID, and tier
     * Called during license deactivation
     *
     * @return void
     */
    public function clearCredentials(): void
    {
        delete_option(Constants::API_KEY_OPTION);
        delete_option(Constants::TENANT_ID_OPTION);
        delete_option(Constants::TIER_OPTION);
        
        if ($this->isDebugMode()) {
            error_log('AIClientManager: Cleared stored API key, tenant ID, and tier');
        }
    }

    /**
     * Check if we have a valid API key for cloud authentication
     * 
     * Note: License keys are NOT valid for API authentication.
     * You must activate the license first to get an API key.
     *
     * @return bool True if we have an API key
     */
    public function hasApiKey(): bool
    {
        return !empty($this->getApiKey());
    }

    /**
     * Check if we have an EDD license key (for activation)
     *
     * @return bool True if we have a license key
     */
    public function hasLicenseKey(): bool
    {
        return !empty($this->getLicenseKey());
    }

    /**
     * Check if we can make authenticated API calls
     * Requires an API key (obtained after license activation)
     *
     * @return bool True if we have an API key
     */
    public function canMakeApiCalls(): bool
    {
        return $this->hasApiKey();
    }

    /**
     * Get normalized workspace ID from WordPress site URL
     *
     * Converts URL to normalized ID (e.g., "example.com" → "example-com")
     *
     * @return string Normalized workspace ID
     */
    public function getWorkspaceId(): string
    {
        $site_url = get_site_url();
        return Configuration::workspaceIdFromUrl($site_url);
    }

    /**
     * Check if debug mode is enabled
     *
     * @return bool
     */
    private function isDebugMode(): bool
    {
        return defined('WP_DEBUG') && WP_DEBUG;
    }

    /**
     * Get timeout setting
     *
     * @return int
     */
    private function getTimeout(): int
    {
        return 30;
    }

    /**
     * Get retry attempts setting
     *
     * @return int
     */
    private function getRetryAttempts(): int
    {
        return 3;
    }

    /**
     * Delete live agent credentials from AWS SSM for a given provider
     * 
     * @param string $provider Provider name (e.g., 'slack', 'zendesk')
     * @return array Response array with 'success' and optional 'error' keys
     */
    public function deleteLiveAgentCredentials(string $provider): array
    {
        try {
            $client = $this->getClient();
            $response = $client->tenant()->deleteLiveAgentCredentials($provider);
            return $response;
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Delete all live agent credentials for the current tenant
     * Used during plugin uninstall
     * 
     * @return bool True if deletion was successful or not needed, false on error
     */
    public function deleteAllLiveAgentCredentials(): bool
    {
        try {
            $provider = get_option(Constants::OPTION_LIVE_AGENT_PROVIDER, Constants::LIVE_AGENT_PROVIDER_NONE);
            
            // No credentials to delete if provider is 'none'
            if ($provider === Constants::LIVE_AGENT_PROVIDER_NONE) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('AIClientManager: No live agent provider configured, skipping credential deletion');
                }
                return true;
            }

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log("AIClientManager: Attempting to delete {$provider} credentials from AWS SSM");
            }

            $response = $this->deleteLiveAgentCredentials($provider);

            if ($response['success'] ?? false) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log("AIClientManager: Successfully deleted {$provider} credentials from AWS SSM");
                }
                return true;
            } else {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log("AIClientManager: Failed to delete {$provider} credentials: " . 
                        ($response['error'] ?? 'Unknown error'));
                }
                return false;
            }
        } catch (\Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('AIClientManager: Error deleting live agent credentials: ' . $e->getMessage());
            }
            return false;
        }
    }

    /**
     * Generate an AI-powered color scheme for the chat widget
     * 
     * @param string $primary_color Primary brand color (hex format, e.g., '#0084ff')
     * @param string|null $secondary_color Optional secondary brand color (hex format)
     * @return array Response array with 'success', 'data', and optional 'error' keys
     */
    public function generateColorScheme(string $primary_color, ?string $secondary_color = null): array
    {
        try {
            // Validate hex color format
            if (!preg_match('/^#[0-9A-Fa-f]{6}$/', $primary_color)) {
                return [
                    'success' => false,
                    'error' => 'Primary color must be a valid hex color (e.g., #0084ff)'
                ];
            }

            if ($secondary_color !== null && !preg_match('/^#[0-9A-Fa-f]{6}$/', $secondary_color)) {
                return [
                    'success' => false,
                    'error' => 'Secondary color must be a valid hex color (e.g., #0084ff)'
                ];
            }

            $client = $this->getClient();
            $colorScheme = $client->config()->generateColorScheme($primary_color, $secondary_color);

            return [
                'success' => true,
                'data' => $colorScheme->toArray()
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    // ========================================
    // MCP CONFIGURATION
    // ========================================

    /**
     * Sync MCP configuration to the cloud based on current WordPress settings
     * 
     * Called when MCP-related settings change to keep cloud in sync.
     * 
     * @return array Response array with 'success' and optional 'error' keys
     */
    public function syncMcpConfig(): array
    {
        try {
            $client = $this->getClient();
            $mcpConfig = $this->buildMcpConfig();
            
            // If no MCP features are enabled, delete the config from cloud
            if (empty($mcpConfig)) {
                $response = $client->tenant()->deleteMcpConfig();
                return [
                    'success' => true,
                    'action' => 'deleted',
                    'message' => 'MCP configuration removed from cloud'
                ];
            }
            
            // Store the MCP config in cloud
            $response = $client->tenant()->storeMcpConfig($mcpConfig);
            
            return [
                'success' => true,
                'action' => 'stored',
                'message' => 'MCP configuration synced to cloud',
                'config' => $mcpConfig
            ];
        } catch (\Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('AIClientManager: Error syncing MCP config: ' . $e->getMessage());
            }
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Get current MCP configuration from the cloud
     * 
     * @return array Response array with 'success', 'configured', 'mcp' keys
     */
    public function getMcpConfig(): array
    {
        try {
            $client = $this->getClient();
            $response = $client->tenant()->getMcpConfig();
            
            return [
                'success' => true,
                'configured' => $response['configured'] ?? false,
                'mcp' => $response['mcp'] ?? []
            ];
        } catch (\Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('AIClientManager: Error getting MCP config: ' . $e->getMessage());
            }
            return [
                'success' => false,
                'configured' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Check if MCP is configured in the cloud
     * 
     * @return bool
     */
    public function isMcpConfigured(): bool
    {
        try {
            $client = $this->getClient();
            return $client->tenant()->isMcpConfigured();
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Delete MCP configuration from the cloud
     * 
     * @return array Response array with 'success' and optional 'error' keys
     */
    public function deleteMcpConfig(): array
    {
        try {
            $client = $this->getClient();
            $client->tenant()->deleteMcpConfig();
            
            return [
                'success' => true,
                'message' => 'MCP configuration deleted from cloud'
            ];
        } catch (\Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('AIClientManager: Error deleting MCP config: ' . $e->getMessage());
            }
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Build MCP configuration array based on current WordPress settings
     * 
     * Uses domain-based routing (/mcp/domain/{domain}) which is the new standard.
     * Each domain maps to a specific MCP endpoint that provides tools for that domain.
     * 
     * @return array MCP configuration (empty if no features enabled)
     */
    private function buildMcpConfig(): array
    {
        $config = [];
        
        // Check if WooCommerce is active
        $woocommerceActive = class_exists('WooCommerce');
        
        // Order inquiries (requires WooCommerce + setting enabled)
        // Uses domain-based routing: /mcp/domain/orders
        $enableOrdersMcp = get_option(Constants::OPTION_ENABLE_MCP_ORDERS, true);
        if ($woocommerceActive && $enableOrdersMcp) {
            $config['ordersUri'] = home_url('/wp-json/intufind/v1/mcp/domain/orders');
        }
        
        // Future MCP domains can be added here as they are enabled:
        // $enableInventoryMcp = get_option(Constants::OPTION_ENABLE_MCP_INVENTORY, false);
        // if ($woocommerceActive && $enableInventoryMcp) {
        //     $config['inventoryUri'] = home_url('/wp-json/intufind/v1/mcp/domain/inventory');
        // }
        //
        // $enableAnalyticsMcp = get_option(Constants::OPTION_ENABLE_MCP_ANALYTICS, false);
        // if ($woocommerceActive && $enableAnalyticsMcp) {
        //     $config['analyticsUri'] = home_url('/wp-json/intufind/v1/mcp/domain/analytics');
        // }
        
        return $config;
    }
}
