<?php

namespace Intucart\Services;

use Intucart\Utils\StringUtils;

/**
 * Advanced Custom Fields Auto-Mapper
 *
 * Automatically detects and maps ACF fields to searchable formats
 * with zero configuration required. Provides intelligent type mapping
 * and content extraction for semantic search integration.
 *
 * Features:
 * - Automatic field detection
 * - Smart type mapping (text, number, select, boolean, etc.)
 * - Content extraction for searchable text
 * - Filter generation for faceted search
 * - Support for complex field types (repeater, flexible content)
 */
class AcfAutoMapper
{
    private Logger $logger;

    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
    }

    /**
     * Check if ACF is available and active
     */
    public function isAcfAvailable(): bool
    {
        return function_exists('get_fields') && function_exists('get_field_object');
    }

    /**
     * Extract all ACF fields for a post and map them to search format
     *
     * @param int $postId Post ID
     * @param string $postType Post type for context
     * @return array Mapped ACF fields ready for search indexing
     */
    public function extractAcfFields(int $postId, string $postType = 'post'): array
    {
        if (!$this->isAcfAvailable()) {
            return [];
        }

        try {
            $acfFields = get_fields($postId) ?: [];

            if (empty($acfFields)) {
                return [];
            }

            $mappedFields = [];
            $extractedText = [];

            foreach ($acfFields as $fieldName => $fieldValue) {
                // Skip empty values
                if ($this->isEmpty($fieldValue)) {
                    continue;
                }

                try {
                    $fieldObject = get_field_object($fieldName, $postId);
                    
                    // Skip malformed field objects
                    if (!is_array($fieldObject)) {
                        $this->logger->debug('Skipping non-array field object', [
                            'field' => $fieldName,
                            'post_id' => $postId,
                            'type' => gettype($fieldObject)
                        ]);
                        continue;
                    }
                    
                    $mappedField = $this->mapFieldToSearchFormat($fieldName, $fieldValue, $fieldObject);

                    if ($mappedField) {
                        $mappedFields[] = $mappedField;

                        // Collect searchable text
                        if ($mappedField['searchable'] ?? false) {
                            $extractedText[] = $mappedField['search_text'] ?? '';
                        }
                    }
                } catch (\Throwable $e) {
                    $this->logger->warning('Failed to process ACF field', [
                        'field' => $fieldName,
                        'post_id' => $postId,
                        'error' => $e->getMessage()
                    ]);
                    continue;
                }
            }

            $this->logger->info('ACF fields extracted', [
                'post_id' => $postId,
                'post_type' => $postType,
                'field_count' => count($mappedFields),
                'search_content_length' => strlen($this->buildSearchContent($extractedText))
            ]);

            return [
                'fields' => $mappedFields,
                'search_content' => $this->buildSearchContent($extractedText),
                'field_count' => count($mappedFields)
            ];
        } catch (\Exception $e) {
            $this->logger->error('ACF field extraction failed', [
                'post_id' => $postId,
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    /**
     * Map individual ACF field to search format based on field type
     */
    private function mapFieldToSearchFormat(string $fieldName, $fieldValue, ?array $fieldObject): ?array
    {
        if (!$fieldObject) {
            return null;
        }

        $fieldType = $fieldObject['type'] ?? 'text';
        $fieldLabel = $fieldObject['label'] ?? $fieldName;

        switch ($fieldType) {
            case 'text':
            case 'email':
            case 'url':
                return $this->mapTextField($fieldName, $fieldValue, $fieldLabel);

            case 'textarea':
                return $this->mapTextareaField($fieldName, $fieldValue, $fieldLabel);

            case 'wysiwyg':
                return $this->mapWysiwygField($fieldName, $fieldValue, $fieldLabel);

            case 'number':
            case 'range':
                return $this->mapNumberField($fieldName, $fieldValue, $fieldLabel, $fieldObject);

            case 'select':
            case 'radio':
            case 'button_group':
                return $this->mapSelectField($fieldName, $fieldValue, $fieldLabel, $fieldObject);

            case 'checkbox':
                return $this->mapCheckboxField($fieldName, $fieldValue, $fieldLabel, $fieldObject);

            case 'true_false':
                return $this->mapBooleanField($fieldName, $fieldValue, $fieldLabel);

            case 'date_picker':
            case 'date_time_picker':
            case 'time_picker':
                return $this->mapDateField($fieldName, $fieldValue, $fieldLabel, $fieldType);

            case 'image':
                return $this->mapImageField($fieldName, $fieldValue, $fieldLabel);

            case 'gallery':
                return $this->mapGalleryField($fieldName, $fieldValue, $fieldLabel);

            case 'file':
                return $this->mapFileField($fieldName, $fieldValue, $fieldLabel);

            case 'repeater':
                return $this->mapRepeaterField($fieldName, $fieldValue, $fieldLabel);

            case 'flexible_content':
                return $this->mapFlexibleContentField($fieldName, $fieldValue, $fieldLabel);

            case 'group':
                return $this->mapGroupField($fieldName, $fieldValue, $fieldLabel);

            case 'post_object':
            case 'page_link':
            case 'relationship':
                return $this->mapRelationshipField($fieldName, $fieldValue, $fieldLabel, $fieldType);

            case 'taxonomy':
                return $this->mapTaxonomyField($fieldName, $fieldValue, $fieldLabel);

            case 'user':
                return $this->mapUserField($fieldName, $fieldValue, $fieldLabel);

            case 'google_map':
                return $this->mapMapField($fieldName, $fieldValue, $fieldLabel);

            case 'color_picker':
                return $this->mapColorField($fieldName, $fieldValue, $fieldLabel);

            default:
                // Fallback for unknown field types
                return $this->mapGenericField($fieldName, $fieldValue, $fieldLabel, $fieldType);
        }
    }

    /**
     * Map text field
     */
    private function mapTextField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        return [
            'key' => "acf_{$fieldName}",
            'value' => [(string) $fieldValue],
            'type' => 'text',
            'label' => $fieldLabel,
            'searchable' => true,
            'filterable' => true,
            'search_text' => (string) $fieldValue
        ];
    }

    /**
     * Map textarea field
     */
    private function mapTextareaField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        $textContent = $this->extractTextContent($fieldValue);

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$textContent],
            'type' => 'textarea',
            'label' => $fieldLabel,
            'searchable' => true,
            'filterable' => false, // Too long for filtering
            'search_text' => $textContent
        ];
    }

    /**
     * Map WYSIWYG field
     */
    private function mapWysiwygField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        $textContent = $this->extractTextContent($fieldValue);

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$textContent],
            'type' => 'wysiwyg',
            'label' => $fieldLabel,
            'searchable' => true,
            'filterable' => false, // Too long for filtering
            'search_text' => $textContent
        ];
    }

    /**
     * Map number/range field
     */
    private function mapNumberField(string $fieldName, $fieldValue, string $fieldLabel, array $fieldObject): array
    {
        $numericValue = (float) $fieldValue;

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$numericValue],
            'type' => 'number',
            'label' => $fieldLabel,
            'searchable' => false,
            'filterable' => true,
            'filter_type' => 'range',
            'min' => $fieldObject['min'] ?? null,
            'max' => $fieldObject['max'] ?? null,
            'step' => $fieldObject['step'] ?? null
        ];
    }

    /**
     * Map select/radio field
     */
    private function mapSelectField(string $fieldName, $fieldValue, string $fieldLabel, array $fieldObject): array
    {
        $value = is_array($fieldValue) ? $fieldValue : [$fieldValue];
        $choices = $fieldObject['choices'] ?? [];

        return [
            'key' => "acf_{$fieldName}",
            'value' => array_map('strval', $value),
            'type' => 'select',
            'label' => $fieldLabel,
            'searchable' => true,
            'filterable' => true,
            'filter_type' => 'select',
            'options' => array_values($choices),
            'search_text' => implode(' ', $value)
        ];
    }

    /**
     * Map checkbox field
     */
    private function mapCheckboxField(string $fieldName, $fieldValue, string $fieldLabel, array $fieldObject): array
    {
        $values = is_array($fieldValue) ? $fieldValue : [$fieldValue];
        $choices = $fieldObject['choices'] ?? [];

        return [
            'key' => "acf_{$fieldName}",
            'value' => array_map('strval', $values),
            'type' => 'checkbox',
            'label' => $fieldLabel,
            'searchable' => true,
            'filterable' => true,
            'filter_type' => 'multi_select',
            'options' => array_values($choices),
            'search_text' => implode(' ', $values)
        ];
    }

    /**
     * Map true/false field
     */
    private function mapBooleanField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        $boolValue = (bool) $fieldValue;

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$boolValue ? 'true' : 'false'],
            'type' => 'boolean',
            'label' => $fieldLabel,
            'searchable' => false,
            'filterable' => true,
            'filter_type' => 'boolean'
        ];
    }

    /**
     * Map date field
     */
    private function mapDateField(string $fieldName, $fieldValue, string $fieldLabel, string $fieldType): array
    {
        $formattedDate = $this->formatDate($fieldValue, $fieldType);

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$formattedDate],
            'type' => 'date',
            'label' => $fieldLabel,
            'searchable' => false,
            'filterable' => true,
            'filter_type' => 'date_range',
            'date_type' => $fieldType
        ];
    }

    /**
     * Map image field
     */
    private function mapImageField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        $imageData = $this->extractImageData($fieldValue);

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$imageData['alt_text'] ?? ''],
            'type' => 'image',
            'label' => $fieldLabel,
            'searchable' => !empty($imageData['alt_text']),
            'filterable' => false,
            'search_text' => $imageData['alt_text'] ?? '',
            'url' => $imageData['url'] ?? null
        ];
    }

    /**
     * Map repeater field
     */
    private function mapRepeaterField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        $textContent = $this->extractRepeaterText($fieldValue);

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$textContent],
            'type' => 'repeater',
            'label' => $fieldLabel,
            'searchable' => !empty($textContent),
            'filterable' => false,
            'search_text' => $textContent,
            'item_count' => is_array($fieldValue) ? count($fieldValue) : 0
        ];
    }

    /**
     * Map flexible content field
     */
    private function mapFlexibleContentField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        $textContent = $this->extractRepeaterText($fieldValue);

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$textContent],
            'type' => 'flexible_content',
            'label' => $fieldLabel,
            'searchable' => !empty($textContent),
            'filterable' => false,
            'search_text' => $textContent
        ];
    }

    /**
     * Map group field
     */
    private function mapGroupField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        $textContent = '';
        if (is_array($fieldValue)) {
            $textParts = [];
            foreach ($fieldValue as $subValue) {
                if (is_string($subValue) || is_numeric($subValue)) {
                    $textParts[] = $this->extractTextContent($subValue);
                }
            }
            $textContent = implode(' ', array_filter($textParts));
        }

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$textContent],
            'type' => 'group',
            'label' => $fieldLabel,
            'searchable' => !empty($textContent),
            'filterable' => false,
            'search_text' => $textContent
        ];
    }

    /**
     * Map relationship field
     */
    private function mapRelationshipField(string $fieldName, $fieldValue, string $fieldLabel, string $fieldType): array
    {
        $titles = [];

        if (is_array($fieldValue)) {
            foreach ($fieldValue as $postId) {
                if (is_numeric($postId)) {
                    $post = get_post($postId);
                    if ($post) {
                        $titles[] = $post->post_title;
                    }
                } elseif (is_object($postId) && isset($postId->post_title)) {
                    $titles[] = $postId->post_title;
                }
            }
        }

        return [
            'key' => "acf_{$fieldName}",
            'value' => $titles,
            'type' => $fieldType,
            'label' => $fieldLabel,
            'searchable' => !empty($titles),
            'filterable' => true,
            'search_text' => implode(' ', $titles)
        ];
    }

    /**
     * Map taxonomy field
     */
    private function mapTaxonomyField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        $termNames = [];

        if (is_array($fieldValue)) {
            foreach ($fieldValue as $term) {
                if (is_object($term) && isset($term->name)) {
                    $termNames[] = $term->name;
                } elseif (is_numeric($term)) {
                    $termObj = get_term($term);
                    if ($termObj && !is_wp_error($termObj)) {
                        $termNames[] = $termObj->name;
                    }
                }
            }
        }

        return [
            'key' => "acf_{$fieldName}",
            'value' => $termNames,
            'type' => 'taxonomy',
            'label' => $fieldLabel,
            'searchable' => !empty($termNames),
            'filterable' => true,
            'search_text' => implode(' ', $termNames)
        ];
    }

    /**
     * Map user field
     */
    private function mapUserField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        $userNames = [];

        if (is_array($fieldValue)) {
            foreach ($fieldValue as $userId) {
                if (is_numeric($userId)) {
                    $user = get_user_by('id', $userId);
                    if ($user) {
                        $userNames[] = $user->display_name;
                    }
                } elseif (is_object($userId) && isset($userId->display_name)) {
                    $userNames[] = $userId->display_name;
                }
            }
        }

        return [
            'key' => "acf_{$fieldName}",
            'value' => $userNames,
            'type' => 'user',
            'label' => $fieldLabel,
            'searchable' => !empty($userNames),
            'filterable' => false,
            'search_text' => implode(' ', $userNames)
        ];
    }

    /**
     * Map map field
     */
    private function mapMapField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        $address = '';
        if (is_array($fieldValue) && isset($fieldValue['address'])) {
            $address = $fieldValue['address'];
        }

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$address],
            'type' => 'map',
            'label' => $fieldLabel,
            'searchable' => !empty($address),
            'filterable' => false,
            'search_text' => $address
        ];
    }

    /**
     * Map color field
     */
    private function mapColorField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        return [
            'key' => "acf_{$fieldName}",
            'value' => [(string) $fieldValue],
            'type' => 'color',
            'label' => $fieldLabel,
            'searchable' => false,
            'filterable' => true,
            'filter_type' => 'select'
        ];
    }

    /**
     * Map generic/unknown field type
     */
    private function mapGenericField(string $fieldName, $fieldValue, string $fieldLabel, string $fieldType): array
    {
        $textValue = $this->extractTextContent($fieldValue);

        return [
            'key' => "acf_{$fieldName}",
            'value' => [$textValue],
            'type' => $fieldType,
            'label' => $fieldLabel,
            'searchable' => !empty($textValue),
            'filterable' => true,
            'search_text' => $textValue
        ];
    }

    /**
     * Check if a field value is empty
     */
    private function isEmpty($value): bool
    {
        if (is_null($value) || $value === '' || $value === false) {
            return true;
        }

        if (is_array($value) && empty($value)) {
            return true;
        }

        return false;
    }

    /**
     * Extract plain text content from HTML or rich text
     */
    private function extractTextContent($content): string
    {
        // Use StringUtils for consistent text cleaning
        return StringUtils::cleanText($content);
    }

    /**
     * Format date value for search indexing
     */
    private function formatDate($dateValue, string $fieldType): string
    {
        if (empty($dateValue)) {
            return '';
        }

        try {
            if (is_numeric($dateValue)) {
                $timestamp = (int) $dateValue;
            } else {
                $timestamp = strtotime($dateValue);
            }

            if (!$timestamp) {
                return '';
            }

            switch ($fieldType) {
                case 'date_picker':
                    return date('Y-m-d', $timestamp);
                case 'date_time_picker':
                    return date('Y-m-d H:i:s', $timestamp);
                case 'time_picker':
                    return date('H:i:s', $timestamp);
                default:
                    return date('Y-m-d', $timestamp);
            }
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * Extract image data including alt text
     */
    private function extractImageData($imageValue): array
    {
        if (empty($imageValue)) {
            return [];
        }

        if (is_array($imageValue)) {
            // Image array format
            return [
                'url' => $imageValue['url'] ?? null,
                'alt_text' => $imageValue['alt'] ?? $imageValue['title'] ?? '',
                'caption' => $imageValue['caption'] ?? ''
            ];
        } elseif (is_numeric($imageValue)) {
            // Image ID format
            $imageId = (int) $imageValue;
            return [
                'url' => wp_get_attachment_url($imageId),
                'alt_text' => get_post_meta($imageId, '_wp_attachment_image_alt', true),
                'caption' => wp_get_attachment_caption($imageId)
            ];
        }

        return [];
    }

    /**
     * Extract searchable text from repeater field
     */
    private function extractRepeaterText($repeaterValue): string
    {
        if (!is_array($repeaterValue)) {
            return '';
        }

        $textParts = [];

        foreach ($repeaterValue as $row) {
            if (is_array($row)) {
                foreach ($row as $subField => $subValue) {
                    if (is_string($subValue) || is_numeric($subValue)) {
                        $textParts[] = $this->extractTextContent($subValue);
                    }
                }
            }
        }

        return implode(' ', array_filter($textParts));
    }

    // Placeholder methods for other field types
    private function mapGalleryField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        return [
            'key' => "acf_{$fieldName}",
            'value' => ['gallery'],
            'type' => 'gallery',
            'label' => $fieldLabel,
            'searchable' => false,
            'filterable' => false,
            'item_count' => is_array($fieldValue) ? count($fieldValue) : 0
        ];
    }

    private function mapFileField(string $fieldName, $fieldValue, string $fieldLabel): array
    {
        return [
            'key' => "acf_{$fieldName}",
            'value' => [is_array($fieldValue) ? ($fieldValue['filename'] ?? '') : ''],
            'type' => 'file',
            'label' => $fieldLabel,
            'searchable' => false,
            'filterable' => false
        ];
    }

    /**
     * Build intelligently formatted search content from ACF fields
     *
     * @param array $extractedText Array of searchable text from fields
     * @return string Properly formatted search content
     */
    private function buildSearchContent(array $extractedText): string
    {
        if (empty($extractedText)) {
            return '';
        }

        // Clean and deduplicate text
        $cleanedText = array_map('trim', $extractedText);
        $cleanedText = array_filter($cleanedText);
        $cleanedText = array_unique($cleanedText);

        // Create semantic search content
        return implode('. ', $cleanedText);
    }
}
