<?php

namespace Intucart\Services;

/**
 * Analytics service for tracking events
 */
class Analytics
{
    private Logger $logger;

    /**
     * Constructor
     *
     * @param Logger $logger Logger service
     */
    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
        add_action('intucart_track_recommendation_impression', [$this, 'processRecommendationImpression'], 10, 4);
    }

    /**
     * Track a search event
     *
     * @param string $query Search query
     * @param int    $resultCount Number of results
     *
     * @return void
     */
    public function trackSearch(string $query, int $resultCount): void
    {
        try {
            global $wpdb;
            $wpdb->insert(
                $wpdb->prefix . 'intucart_search_analytics',
                [
                    'search_term' => $query,
                    'results_count' => $resultCount,
                    'created_at' => current_time('mysql')
                ],
                ['%s', '%d', '%s']
            );
        } catch (\Exception $e) {
            $this->logger->error('Failed to track search event', [
                'error' => $e->getMessage(),
                'query' => $query
            ]);
        }
    }

    /**
     * Schedule recommendation impression tracking
     *
     * @param int    $productId Product ID being recommended
     * @param int    $sourceProductId Source product ID generating the recommendation
     * @param string $type Type of recommendation (related, upsell, cross-sell)
     * @param float  $score Recommendation score
     *
     * @return void
     */
    public function trackRecommendationImpression(
        int $productId,
        int $sourceProductId,
        string $type,
        float $score
    ): void {
        wp_schedule_single_event(
            time(), // Run immediately
            'intucart_track_recommendation_impression',
            [
                $productId,
                $sourceProductId,
                $type,
                $score
            ]
        );
    }

    /**
     * Process recommendation impression tracking
     *
     * @param int    $productId Product ID being recommended
     * @param int    $sourceProductId Source product ID generating the recommendation
     * @param string $type Type of recommendation (related, upsell, cross-sell)
     * @param float  $score Recommendation score
     *
     * @return void
     */
    public function processRecommendationImpression(
        int $productId,
        int $sourceProductId,
        string $type,
        float $score
    ): void {
        try {
            global $wpdb;
            $table = $wpdb->prefix . 'intucart_recommendation_analytics';

            $result = $wpdb->query($wpdb->prepare(
                "INSERT INTO $table (product_id, source_product_id, recommendation_type, " .
                "impression_count, total_score, last_updated) " .
                "VALUES (%d, %d, %s, 1, %f, %s) " .
                "ON DUPLICATE KEY UPDATE " .
                "impression_count = impression_count + 1, " .
                "total_score = total_score + VALUES(total_score), " .
                "last_updated = VALUES(last_updated)",
                $productId,
                $sourceProductId,
                $type,
                $score,
                current_time('mysql')
            ));

            if ($result === false) {
                throw new \Exception($wpdb->last_error);
            }
        } catch (\Exception $e) {
            $this->logger->error('Failed to process recommendation tracking', [
                'error' => $e->getMessage(),
                'product_id' => $productId,
                'source_product_id' => $sourceProductId,
                'type' => $type
            ]);
        }
    }

}
