<?php

namespace Intucart\Services;

/**
 * Asset Optimization Service
 * 
 * Handles exclusions for third-party optimization plugins to prevent them from
 * interfering with chatbot assets (CSS, JS, inline styles).
 * 
 * Supported Plugins:
 * - Autoptimize
 * - WP Rocket
 * - LiteSpeed Cache
 * - Perfmatters (future)
 * - Asset CleanUp (future)
 *
 * @category Services
 * @package  Intucart
 * @author   Intucart <support@intufind.com>
 * @license  GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://intufind.com
 */
class AssetOptimizationService
{
    /**
     * Asset identifiers to protect from optimization
     */
    private const CHATBOT_CSS_FILE = 'intucart-chatbot.min.css';
    private const CHATBOT_JS_FILE = 'intucart-chatbot.min.js';
    private const CHATBOT_HANDLE = 'intucart-chatbot';

    /**
     * Register optimization exclusion filters
     *
     * @return void
     */
    public function register(): void
    {
        // Autoptimize exclusions
        add_filter('autoptimize_filter_css_exclude', array($this, 'excludeCSSFromAutoptimize'), 10, 1);
        add_filter('autoptimize_filter_js_exclude', array($this, 'excludeJSFromAutoptimize'), 10, 1);
        
        // WP Rocket exclusions
        add_filter('rocket_exclude_css', array($this, 'excludeCSSFromWPRocket'), 10, 1);
        add_filter('rocket_excluded_inline_css', array($this, 'excludeInlineCSSFromWPRocket'), 10, 1);
        add_filter('rocket_delay_css_exclusions', array($this, 'preventWPRocketCSSDelay'), 10, 1);
        add_filter('rocket_exclude_defer_js', array($this, 'excludeJSFromWPRocketDefer'), 10, 1);
        add_filter('rocket_exclude_js', array($this, 'excludeJSFromWPRocketMinify'), 10, 1);
        
        // LiteSpeed Cache exclusions
        add_filter('litespeed_optimize_css_excludes', array($this, 'excludeCSSFromLiteSpeed'), 10, 1);
        add_filter('litespeed_optimize_js_excludes', array($this, 'excludeJSFromLiteSpeed'), 10, 1);
    }

    // =========================================================================
    // Autoptimize Exclusions
    // =========================================================================

    /**
     * Prevent Autoptimize from combining/minifying chatbot CSS
     *
     * @param string $exclude Existing exclusions (comma-separated)
     * @return string Modified exclusions
     */
    public function excludeCSSFromAutoptimize(string $exclude): string
    {
        return $exclude . ', ' . self::CHATBOT_CSS_FILE;
    }

    /**
     * Prevent Autoptimize from combining/minifying chatbot JS
     *
     * @param string $exclude Existing exclusions (comma-separated)
     * @return string Modified exclusions
     */
    public function excludeJSFromAutoptimize(string $exclude): string
    {
        return $exclude . ', ' . self::CHATBOT_JS_FILE;
    }

    // =========================================================================
    // WP Rocket Exclusions
    // =========================================================================

    /**
     * Prevent WP Rocket from combining/minifying chatbot CSS
     *
     * @param array $excluded Existing exclusions
     * @return array Modified exclusions
     */
    public function excludeCSSFromWPRocket(array $excluded): array
    {
        $excluded[] = '/' . self::CHATBOT_CSS_FILE;
        $excluded[] = self::CHATBOT_CSS_FILE;
        return $excluded;
    }

    /**
     * Prevent WP Rocket from removing inline CSS (custom properties/variables)
     *
     * @param array $excluded Existing exclusions
     * @return array Modified exclusions
     */
    public function excludeInlineCSSFromWPRocket(array $excluded): array
    {
        $excluded[] = self::CHATBOT_HANDLE;
        return $excluded;
    }

    /**
     * Prevent WP Rocket from delaying chatbot CSS (Load CSS Asynchronously)
     *
     * @param array $exclusions Existing exclusions
     * @return array Modified exclusions
     */
    public function preventWPRocketCSSDelay(array $exclusions): array
    {
        $exclusions[] = '/' . self::CHATBOT_CSS_FILE;
        $exclusions[] = self::CHATBOT_CSS_FILE;
        return $exclusions;
    }

    /**
     * Prevent WP Rocket from deferring chatbot JS
     *
     * @param array $excluded Existing exclusions
     * @return array Modified exclusions
     */
    public function excludeJSFromWPRocketDefer(array $excluded): array
    {
        $excluded[] = '/' . self::CHATBOT_JS_FILE;
        $excluded[] = self::CHATBOT_JS_FILE;
        return $excluded;
    }

    /**
     * Prevent WP Rocket from minifying/combining chatbot JS
     *
     * @param array $excluded Existing exclusions
     * @return array Modified exclusions
     */
    public function excludeJSFromWPRocketMinify(array $excluded): array
    {
        $excluded[] = '/' . self::CHATBOT_JS_FILE;
        $excluded[] = self::CHATBOT_JS_FILE;
        return $excluded;
    }

    // =========================================================================
    // LiteSpeed Cache Exclusions
    // =========================================================================

    /**
     * Prevent LiteSpeed Cache from optimizing chatbot CSS
     *
     * @param array $excludes Existing exclusions
     * @return array Modified exclusions
     */
    public function excludeCSSFromLiteSpeed(array $excludes): array
    {
        $excludes[] = self::CHATBOT_CSS_FILE;
        return $excludes;
    }

    /**
     * Prevent LiteSpeed Cache from optimizing chatbot JS
     *
     * @param array $excludes Existing exclusions
     * @return array Modified exclusions
     */
    public function excludeJSFromLiteSpeed(array $excludes): array
    {
        $excludes[] = self::CHATBOT_JS_FILE;
        return $excludes;
    }

    // =========================================================================
    // Utility Methods
    // =========================================================================

    /**
     * Check if any supported optimization plugin is active
     *
     * @return array Active optimization plugins
     */
    public function getActiveOptimizationPlugins(): array
    {
        $active = array();

        // Check Autoptimize
        if (class_exists('autoptimizeConfig')) {
            $active[] = 'autoptimize';
        }

        // Check WP Rocket
        if (function_exists('get_rocket_option')) {
            $active[] = 'wp-rocket';
        }

        // Check LiteSpeed Cache
        if (defined('LSCWP_V')) {
            $active[] = 'litespeed-cache';
        }

        return $active;
    }

    /**
     * Log active optimization plugins (for debugging)
     *
     * @return void
     */
    public function logActivePlugins(): void
    {
        if (!defined('WP_DEBUG') || !WP_DEBUG) {
            return;
        }

        $active = $this->getActiveOptimizationPlugins();
        
        if (empty($active)) {
            error_log('Intucart: No optimization plugins detected');
            return;
        }

        error_log('Intucart: Active optimization plugins: ' . implode(', ', $active));
    }
}

