<?php

namespace Intucart\Services\Cache;

use Intucart\Services\Constants;

/**
 * Cache Service
 */
class CacheService
{
    private const DEFAULT_EXPIRATION = 3600; // 1 hour

    // Static properties to hold increments during request
    private static array $pending_hits = [];
    private static array $pending_misses = [];
    private static bool $shutdown_hook_added = false;

    // Cache group constants
    public const CACHE_GROUP_LAMBDA = 'intucart_lambda';
    public const CACHE_GROUP_SEARCH = 'intucart_search';
    public const CACHE_GROUP_RECOMMENDATIONS = 'intucart_recommendations';
    public const CACHE_GROUP_LICENSE = 'intucart_license';
    public const CACHE_GROUP_PRODUCT_ASSOCIATIONS = 'intucart_product_associations';
    public const CACHE_GROUP_FILTERS = 'intucart_filters';
    public const CACHE_GROUP_SHOPPING_CONTEXT = 'intucart_shopping_context';

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->registerShutdownHook();
    }

    /**
     * Register the shutdown hook to save stats, ensuring it only runs once.
     */
    private function registerShutdownHook(): void
    {
        if (!self::$shutdown_hook_added) {
            add_action('shutdown', [__CLASS__, 'saveStatsOnShutdown']);
            self::$shutdown_hook_added = true;
        }
    }

    /**
     * Get value from cache
     *
     * @param string $key   Cache key
     * @param string $group Cache group
     *
     * @return mixed|false Cached value or false if not found
     */
    public function get(string $key, string $group)
    {
        $result = false;

        if (wp_using_ext_object_cache()) {
            $result = wp_cache_get($key, $group);
        } else {
            $result = get_transient($this->getTransientKey($group, $key));
        }

        // Track cache hit/miss statistics
        $this->trackCacheAccess($group, $result !== false);

        return $result;
    }

    /**
     * Set value in cache
     *
     * @param string $key        Cache key
     * @param mixed  $value      Value to cache
     * @param string $group      Cache group
     * @param int    $expiration Cache expiration in seconds
     *
     * @return bool True on success, false on failure
     */
    public function set(
        string $key,
        $value,
        string $group,
        int $expiration = self::DEFAULT_EXPIRATION
    ): bool {
        if (wp_using_ext_object_cache()) {
            return wp_cache_set($key, $value, $group, $expiration);
        }

        $transient_key = $this->getTransientKey($group, $key);
        $result = set_transient(
            $transient_key,
            $value,
            $expiration
        );

        return $result;
    }

    /**
     * Delete value from cache
     *
     * @param string $key   Cache key
     * @param string $group Cache group
     *
     * @return bool True on success, false on failure
     */
    public function delete(string $key, string $group): bool
    {
        $success = true;

        // Delete the main key (primary success criterion)
        $main_deleted = wp_using_ext_object_cache()
            ? wp_cache_delete($key, $group)
            : delete_transient($this->getTransientKey($group, $key));
        $success = $success && $main_deleted;

        // Delete stale data
        $stale_key = $this->getStaleKey($key);
        if (wp_using_ext_object_cache()) {
            wp_cache_delete($stale_key, $group);
        } else {
            delete_transient($this->getTransientKey($group, $stale_key));
        }

        // Delete fresh timestamp
        if (wp_using_ext_object_cache()) {
            wp_cache_delete($key . '_fresh', $group);
        } else {
            delete_transient($this->getTransientKey($group, $key . '_fresh'));
        }

        // Delete associated callback transient
        $callback_key = 'revalidate_callback_' . $key;
        if (wp_using_ext_object_cache()) {
            wp_cache_delete($callback_key, $group);
        } else {
            delete_transient($callback_key);
        }

        return $main_deleted; // Return true if main key was deleted, regardless of secondary keys
    }

    /**
     * Generate transient key
     *
     * @param string $group Cache group
     * @param string $key   Cache key
     *
     * @return string
     */
    private function getTransientKey(string $group, string $key): string
    {
        return sprintf('%s_%s', $group, $key);
    }

    /**
     * Track cache access for statistics
     *
     * @param string $group Cache group
     * @param bool   $hit   Whether the cache access was a hit
     *
     * @return void
     */
    private function trackCacheAccess(string $group, bool $hit): void
    {
        // Store increments in static arrays instead of immediate DB write
        if ($hit) {
            if (!isset(self::$pending_hits[$group])) {
                self::$pending_hits[$group] = 0;
            }
            self::$pending_hits[$group]++;
        } else {
            if (!isset(self::$pending_misses[$group])) {
                self::$pending_misses[$group] = 0;
            }
            self::$pending_misses[$group]++;
        }

        // Ensure shutdown hook is registered (in case constructor wasn't called, though unlikely with DI)
        $this->registerShutdownHook();
    }

    /**
     * Save accumulated cache stats on WordPress shutdown.
     */
    public static function saveStatsOnShutdown(): void
    {
        // Process hits
        if (!empty(self::$pending_hits)) {
            $current_hits = get_option('intucart_cache_hits', []);
            foreach (self::$pending_hits as $group => $count) {
                if (!isset($current_hits[$group])) {
                    $current_hits[$group] = 0;
                }
                $current_hits[$group] += $count;
            }
            update_option('intucart_cache_hits', $current_hits, 'no'); // 'no' means don't autoload
        }

        // Process misses
        if (!empty(self::$pending_misses)) {
            $current_misses = get_option('intucart_cache_misses', []);
            foreach (self::$pending_misses as $group => $count) {
                if (!isset($current_misses[$group])) {
                    $current_misses[$group] = 0;
                }
                $current_misses[$group] += $count;
            }
            update_option('intucart_cache_misses', $current_misses, 'no'); // 'no' means don't autoload
        }

        // Reset pending stats for the next request (though static context ends anyway)
        self::$pending_hits = [];
        self::$pending_misses = [];
    }

    /**
     * Clear cache statistics
     *
     * @return void
     */
    public function clearCacheStats(): void
    {
        update_option('intucart_cache_hits', []);
        update_option('intucart_cache_misses', []);
    }

    /**
     * Clear cache for a specific group
     *
     * @param string $group Cache group to clear
     *
     * @return void
     */
    public function clearCacheGroup(string $group): void
    {
        if (wp_using_ext_object_cache()) {
            wp_cache_flush_group($group);
        } else {
            global $wpdb;
            $prefix = '_transient_' . $group . '_';
            $wpdb->query($wpdb->prepare(
                "DELETE FROM $wpdb->options WHERE option_name LIKE %s",
                $prefix . '%'
            ));
        }
    }

    /**
     * Clear all cache
     *
     * @return void
     */
    public function clearAllCache(): void
    {
        if (wp_using_ext_object_cache()) {
            wp_cache_flush();
        } else {
            global $wpdb;
            $wpdb->query("DELETE FROM $wpdb->options WHERE option_name LIKE '_transient_%'");
        }
    }

    /**
     * Clear all Intucart transients
     *
     * @return void
     */
    public function clear(): void
    {
        if (wp_using_ext_object_cache()) {
            // If using external object cache, we'd need to flush specific groups
            // This would depend on how your cache groups are structured
            wp_cache_flush();
        } else {
            global $wpdb;

            // Delete all transients with our prefix
            $wpdb->query($wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
                '_transient_intucart_%',
                '_transient_timeout_intucart_%'
            ));
        }
    }

    /**
     * Get a value from cache with background revalidation
     *
     * @param string $key        Cache key
     * @param string $group      Cache group
     * @param int    $expiration Cache expiration in seconds
     * @param array  $callback   Array containing [object, method] for revalidation
     * @param array  $params     Parameters to pass to the revalidation method
     *
     * @return mixed
     */
    public function getWithRevalidate(
        string $key,
        string $group,
        int $expiration,
        array $callback,
        array $params = []
    ): mixed {
        // Get the data with a longer expiration time
        $stale_key = $this->getStaleKey($key);
        $result = $this->get($stale_key, $group);

        if ($result === false) {
            // No data in cache, execute callback and cache result
            $result = call_user_func_array($callback, $params);

            // Don't cache fallback results - let them try again next time
            if ($result !== false && !$this->isFallbackResult($result)) {
                // Store with longer expiration for stale data
                $this->set($stale_key, $result, $group, $expiration * 2);
                // Store fresh timestamp
                $this->set($key . '_fresh', time(), $group, $expiration);
            }
            return $result;
        }

        // Check if data is stale (past fresh expiration)
        $fresh_time = $this->get($key . '_fresh', $group);
        if ($fresh_time === false || $fresh_time + $expiration < time()) {
            // Data is stale, trigger background revalidation
            $this->triggerBackgroundRevalidation($key, $group, $expiration, $callback, $params);
        }

        return $result;
    }

    /**
     * Get the key for storing stale data
     *
     * @param string $key Original cache key
     * @return string
     */
    private function getStaleKey(string $key): string
    {
        return $key . '_stale';
    }

    /**
     * Check if result indicates a fallback scenario that shouldn't be cached
     *
     * @param mixed $result The result to check
     * @return bool True if this is a fallback result that shouldn't be cached
     */
    private function isFallbackResult($result): bool
    {
        // Check if result indicates fallback is needed
        if (is_array($result) && isset($result['fallback_needed']) && $result['fallback_needed'] === true) {
            return true;
        }

        // Check for empty recommendations that might indicate an error
        if (is_array($result) && empty($result)) {
            return true;
        }

        return false;
    }

    /**
     * Trigger background revalidation of stale cache
     *
     * @param string $key        Cache key
     * @param string $group      Cache group
     * @param int    $expiration Cache expiration in seconds
     * @param array  $callback   Array containing [object, method] for revalidation
     * @param array  $params     Parameters to pass to the revalidation method
     *
     * @return void
     */
    private function triggerBackgroundRevalidation(
        string $key,
        string $group,
        int $expiration,
        array $callback,
        array $params = []
    ): void {
        // Extract class and method from the callback
        [$object, $method] = $callback;
        $className = get_class($object);
        $methodName = $method;

        // Store the callback details in a transient for the background job
        $callback_key = 'revalidate_callback_' . $key;
        $callback_data = [
            'class' => $className,
            'method' => $methodName,
            'params' => $params,
            'group' => $group,
            'key' => $key,
            'expiration' => $expiration
        ];
        set_transient($callback_key, $callback_data, $expiration);

        // Schedule background revalidation
        if (!wp_next_scheduled(Constants::CACHE_REVALIDATE_HOOK, [$key, $group, $expiration])) {
            wp_schedule_single_event(time(), Constants::CACHE_REVALIDATE_HOOK, [
                $key,
                $group,
                $expiration
            ]);
        }
    }
}
