<?php

namespace Intucart\Services;

use Intucart\Services\Cache\CacheService;

/**
 * Cache Admin functionality
 */
class CacheAdmin
{
    private CacheService $cacheService;

    /**
     * Constructor
     *
     * @param CacheService $cacheService Cache service instance
     */
    public function __construct(CacheService $cacheService)
    {
        $this->cacheService = $cacheService;
    }

    /**
     * Initialize the service
     *
     * @return void
     */
    public function initialize(): void
    {
        $this->registerHooks();
    }

    /**
     * Register WordPress hooks
     *
     * @return void
     */
    private function registerHooks(): void
    {
        add_action('wp_ajax_intucart_clear_cache_group', [$this, 'handleClearCacheGroup']);
        add_action('wp_ajax_intucart_clear_all_cache', [$this, 'handleClearAllCache']);
        add_action('wp_ajax_intucart_reset_cache_stats', [$this, 'handleResetCacheStats']);
    }

    /**
     * Handle AJAX request to clear cache for a specific group
     *
     * @return void
     */
    public function handleClearCacheGroup(): void
    {
        // Check nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!$nonce || !wp_verify_nonce($nonce, 'intucart_cache_nonce')) {
            wp_send_json_error(['message' => __('Invalid security token.', 'intufind')]);
            return;
        }

        // Check if group is provided
        if (!isset($_POST['group']) || empty($_POST['group'])) {
            wp_send_json_error(['message' => __('No cache group specified.', 'intufind')]);
            return;
        }

        $group = sanitize_text_field(wp_unslash($_POST['group']));

        try {
            $this->cacheService->clearCacheGroup($group);
            wp_send_json_success(['message' => __('Cache group cleared successfully.', 'intufind')]);
        } catch (\Exception $e) {
            wp_send_json_error(['message' => sprintf(__('Failed to clear cache group: %s', 'intufind'), $e->getMessage())]);
        }
    }

    /**
     * Handle AJAX request to clear all cache
     *
     * @return void
     */
    public function handleClearAllCache(): void
    {
        // Check nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!$nonce || !wp_verify_nonce($nonce, 'intucart_cache_nonce')) {
            wp_send_json_error(['message' => __('Invalid security token.', 'intufind')]);
            return;
        }

        try {
            $this->cacheService->clearAllCache();
            wp_send_json_success(['message' => __('All cache cleared successfully.', 'intufind')]);
        } catch (\Exception $e) {
            wp_send_json_error(['message' => sprintf(__('Failed to clear all cache: %s', 'intufind'), $e->getMessage())]);
        }
    }

    /**
     * Handle AJAX request to reset cache statistics
     *
     * @return void
     */
    public function handleResetCacheStats(): void
    {
        // Check nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!$nonce || !wp_verify_nonce($nonce, 'intucart_cache_nonce')) {
            wp_send_json_error(['message' => __('Invalid security token.', 'intufind')]);
            return;
        }

        try {
            $this->cacheService->clearCacheStats();
            wp_send_json_success(['message' => __('Cache statistics reset successfully.', 'intufind')]);
        } catch (\Exception $e) {
            wp_send_json_error(['message' => sprintf(__('Failed to reset cache statistics: %s', 'intufind'), $e->getMessage())]);
        }
    }
}
