<?php

namespace Intucart\Services\Events;

/**
 * Abstract implementation of EventInterface
 *
 * This abstract class provides a base implementation of the EventInterface.
 * It handles common functionality like storing user ID, timestamp, and parameters.
 * Concrete event classes should extend this class and implement the abstract methods.
 *
 * In the DTO-based system, concrete classes typically:
 * 1. Store event-specific DTOs as properties
 * 2. Initialize these DTOs in their constructor
 * 3. Override getParameters() to return DTO data as arrays
 */
abstract class AbstractEvent implements EventInterface
{
    protected int $userId;
    protected array $parameters;
    protected int $timestamp;

    /**
     * Constructor
     *
     * @param int   $userId     User ID
     * @param array $parameters Event parameters
     */
    public function __construct(int $userId, array $parameters = [])
    {
        $this->userId = $userId;
        $this->parameters = $parameters;
        $this->timestamp = time();
    }

    /**
     * Get event name
     *
     * @return string The unique identifier for this event type
     */
    abstract public function getName(): string;

    /**
     * Get event timestamp
     *
     * @return int Unix timestamp when the event occurred
     */
    public function getTimestamp(): int
    {
        return $this->timestamp;
    }

    /**
     * Get user ID
     *
     * @return int The ID of the user who triggered the event
     */
    public function getUserId(): int
    {
        return $this->userId;
    }

    /**
     * Get event parameters
     *
     * This base implementation returns the raw parameters array.
     * Concrete classes should override this method to return structured data
     * from their DTOs using toArray() methods.
     *
     * @return array The event data as an associative array
     */
    public function getParameters(): array
    {
        return $this->parameters;
    }

    /**
     * Get product ID
     *
     * For multi-product events like Purchase, this returns null as these events
     * contain multiple products and are handled differently.
     *
     * @return int|null The product ID, or null for multi-product events
     */
    abstract public function getProductId(): ?int;

    /**
     * Validate event data
     *
     * Ensures that all required data for this event type is present and valid.
     * Concrete classes should implement this method to validate their specific data.
     *
     * @return bool True if the event data is valid, false otherwise
     */
    abstract public function validate(): bool;
}
