<?php

namespace Intucart\Services\Events;

use Intucart\Services\Events\Data\CartItemDTO;
use Intucart\Services\Events\Data\OrderDTO;
use Intucart\Services\Events\Data\ProductDTO;

/**
 * Trait for validating cart items
 *
 * This trait provides validation methods for cart items in both array format
 * and DTO format. It's used by event classes that deal with cart items.
 */
trait CartItemValidatorTrait
{
    /**
     * Validate cart item data in array format
     *
     * @param array $parameters Cart item parameters to validate
     *
     * @return bool True if the cart item data is valid, false otherwise
     */
    private function validateCartItem(array $parameters): bool
    {
        if (!isset($parameters['product'])) {
            return false;
        }

        if (!isset($parameters['product']['id'])) {
            return false;
        }

        if (!isset($parameters['product']['name'])) {
            return false;
        }

        if (!isset($parameters['quantity'])) {
            return false;
        }

        return true;
    }

    /**
     * Validate a CartItemDTO object
     *
     * @param CartItemDTO $cartItem The cart item DTO to validate
     *
     * @return bool True if the cart item DTO is valid, false otherwise
     */
    private function validateCartItemDTO(CartItemDTO $cartItem): bool
    {
        $product = $cartItem->getProduct();

        if (!$product instanceof ProductDTO) {
            return false;
        }

        if ($product->getId() <= 0) {
            return false;
        }

        if (empty($product->getName())) {
            return false;
        }

        if ($cartItem->getQuantity() <= 0) {
            return false;
        }

        return true;
    }

    /**
     * Validate an OrderDTO object
     *
     * @param OrderDTO $order The order DTO to validate
     * @return bool
     */
    private function validateOrderDTO(OrderDTO $order): bool
    {
        // Validate order ID
        if (empty($order->getId())) {
            return false;
        }

        // Validate order total
        if ($order->getTotal() <= 0) {
            return false;
        }

        // Validate order items
        $items = $order->getItems();
        if (empty($items)) {
            return false;
        }

        // Validate each cart item in the order
        foreach ($items as $item) {
            if (!$this->validateCartItemDTO($item)) {
                return false;
            }
        }

        return true;
    }
}
