<?php

namespace Intucart\Services\Events\Data;

use Intucart\Services\Events\EventDataInterface;

/**
 * Cart Item Data Transfer Object
 *
 * Standardizes cart item data structure across the event system
 */
class CartItemDTO implements EventDataInterface
{
    private ProductDTO $product;
    private int $quantity;

    /**
     * Constructor
     *
     * @param ProductDTO $product  Product data
     * @param int        $quantity Item quantity
     */
    public function __construct(ProductDTO $product, int $quantity)
    {
        $this->product = $product;
        $this->quantity = $quantity;
    }

    /**
     * Create from array
     *
     * @param array $data Cart item data array
     *
     * @return self
     */
    public static function fromArray(array $data): self
    {
        $productData = $data['product'] ?? [];
        $product = $productData instanceof ProductDTO
            ? $productData
            : ProductDTO::fromArray(is_array($productData) ? $productData : []);

        return new self(
            $product,
            (int) ($data['quantity'] ?? 1)
        );
    }

    /**
     * Get product
     *
     * @return ProductDTO
     */
    public function getProduct(): ProductDTO
    {
        return $this->product;
    }

    /**
     * Get quantity
     *
     * @return int
     */
    public function getQuantity(): int
    {
        return $this->quantity;
    }

    /**
     * Convert to array
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            'product' => $this->product->toArray(),
            'quantity' => $this->quantity
        ];
    }

    /**
     * Specify data which should be serialized to JSON
     *
     * @return array
     */
    public function jsonSerialize(): array
    {
        return $this->toArray();
    }
}
