<?php

namespace Intucart\Services\Events\Data;

use Intucart\Services\Events\EventDataInterface;

/**
 * Order Data Transfer Object
 *
 * Standardizes order data structure across the event system
 */
class OrderDTO implements EventDataInterface, \JsonSerializable
{
    private string $id;
    private float $total;
    private string $currency;
    private array $items;

    /**
     * Constructor
     *
     * @param string $id       Order ID
     * @param float  $total    Order total
     * @param string $currency Currency code
     * @param array  $items    Order items (CartItemDTO objects)
     */
    public function __construct(
        string $id,
        float $total,
        string $currency = 'USD',
        array $items = []
    ) {
        $this->id = $id;
        $this->total = $total;
        $this->currency = $currency;
        $this->items = $items;
    }

    /**
     * Create from array
     *
     * @param array $data Order data array
     *
     * @return self
     */
    public static function fromArray(array $data): self
    {
        $items = [];
        foreach ($data['items'] ?? [] as $item) {
            if ($item instanceof CartItemDTO) {
                $items[] = $item;
            } elseif (is_array($item)) {
                $items[] = CartItemDTO::fromArray($item);
            }
        }

        return new self(
            (string) ($data['id'] ?? ''),
            (float) ($data['total'] ?? 0.0),
            (string) ($data['currency'] ?? 'USD'),
            $items
        );
    }

    /**
     * Get order ID
     *
     * @return string
     */
    public function getId(): string
    {
        return $this->id;
    }

    /**
     * Get order total
     *
     * @return float
     */
    public function getTotal(): float
    {
        return $this->total;
    }

    /**
     * Get currency
     *
     * @return string
     */
    public function getCurrency(): string
    {
        return $this->currency;
    }

    /**
     * Get order items
     *
     * @return array
     */
    public function getItems(): array
    {
        return $this->items;
    }

    /**
     * Convert to array
     *
     * @return array
     */
    public function toArray(): array
    {
        $itemsArray = [];
        foreach ($this->items as $item) {
            $itemsArray[] = $item instanceof CartItemDTO ? $item->toArray() : $item;
        }

        return [
            'id' => $this->id,
            'total' => $this->total,
            'currency' => $this->currency,
            'items' => $itemsArray
        ];
    }

    /**
     * Specify data which should be serialized to JSON
     *
     * @return array
     */
    public function jsonSerialize(): array
    {
        return $this->toArray();
    }
}
