<?php

namespace Intucart\Services\Events\Data;

use Intucart\Services\Events\EventDataInterface;

/**
 * Product Data Transfer Object
 *
 * Standardizes product data structure across the event system
 */
class ProductDTO implements EventDataInterface, \JsonSerializable
{
    private int $id;
    private string $name;
    private float $price;
    private array $categories;
    private array $subcategories;

    /**
     * Constructor
     *
     * @param int    $id            Product ID
     * @param string $name          Product name
     * @param float  $price         Product price
     * @param array  $categories    Product categories
     * @param array  $subcategories Product subcategories
     */
    public function __construct(
        int $id,
        string $name,
        float $price,
        array $categories = [],
        array $subcategories = []
    ) {
        $this->id = $id;
        $this->name = $name;
        $this->price = $price;
        $this->categories = $categories;
        $this->subcategories = $subcategories;
    }

    /**
     * Create from array
     *
     * @param array $data Product data array
     *
     * @return self
     */
    public static function fromArray(array $data): self
    {
        return new self(
            (int) ($data['id'] ?? 0),
            (string) ($data['name'] ?? ''),
            (float) ($data['price'] ?? 0.0),
            (array) ($data['categories'] ?? []),
            (array) ($data['subcategories'] ?? [])
        );
    }

    /**
     * Get product ID
     *
     * @return int
     */
    public function getId(): int
    {
        return $this->id;
    }

    /**
     * Get product name
     *
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * Get product price
     *
     * @return float
     */
    public function getPrice(): float
    {
        return $this->price;
    }

    /**
     * Get product categories
     *
     * @return array
     */
    public function getCategories(): array
    {
        return $this->categories;
    }

    /**
     * Get product subcategories
     *
     * @return array
     */
    public function getSubcategories(): array
    {
        return $this->subcategories;
    }

    /**
     * Convert to array
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            'id' => $this->id,
            'name' => $this->name,
            'price' => $this->price,
            'categories' => $this->categories,
            'subcategories' => $this->subcategories
        ];
    }

    /**
     * Specify data which should be serialized to JSON
     *
     * @return array
     */
    public function jsonSerialize(): array
    {
        return $this->toArray();
    }
}
