<?php

namespace Intucart\Services\Events;

use Intucart\Services\Events\Data\OrderDTO;

/**
 * Purchase event class
 */
class Purchase extends AbstractEvent
{
    use CartItemValidatorTrait;

    private OrderDTO $order;

    /**
     * Constructor
     *
     * @param int   $userId     User ID
     * @param array $parameters Event parameters
     */
    public function __construct(int $userId, array $parameters)
    {
        parent::__construct($userId, $parameters);

        // Convert parameters to DTO for type safety
        $this->order = OrderDTO::fromArray($parameters['order'] ?? []);
    }

    /**
     * Get event name
     *
     * @return string
     */
    public function getName(): string
    {
        return 'purchase';
    }

    /**
     * Validate event data
     *
     * @return bool
     */
    public function validate(): bool
    {
        // First check if the parameters array has the required structure
        if (!isset($this->parameters['order'])) {
            return false;
        }

        // Then validate the DTO that was created in the constructor
        return $this->validateOrderDTO($this->order);
    }

    /**
     * Get product ID
     *
     * For Purchase events, this returns null since a purchase doesn't have a single product ID.
     * The EventManager handles Purchase events differently by iterating through order items.
     *
     * @return int|null Always returns null for Purchase events
     */
    public function getProductId(): ?int
    {
        return null; // Purchase events don't have a single product ID
    }

    /**
     * Get order ID
     *
     * @return string
     */
    public function getOrderId(): string
    {
        return $this->order->getId();
    }

    /**
     * Get order total
     *
     * @return float
     */
    public function getOrderTotal(): float
    {
        return $this->order->getTotal();
    }

    /**
     * Get order items
     *
     * @return array
     */
    public function getOrderItems(): array
    {
        return $this->order->getItems();
    }

    /**
     * Get order DTO
     *
     * @return OrderDTO
     */
    public function getOrderDTO(): OrderDTO
    {
        return $this->order;
    }

    /**
     * Get the parameters of the event
     *
     * @return array
     */
    public function getParameters(): array
    {
        return [
            'order' => $this->order->toArray()
        ];
    }
}
