<?php

namespace Intucart\Services\Licensing;

use Intucart\Services\Cache\CacheService;
use Intucart\Services\Constants;
use Intucart\Services\AIClientManager;
use Intucart\Services\Logger;

/**
 * API Key service class
 * 
 * Manages API key validation, tier checking, and feature availability.
 * Users obtain API keys from the Intufind dashboard and enter them here.
 */
class License
{
    private const CACHE_GROUP = 'intucart_license';

    /**
     * Cache keys for license data
     */
    private const LICENSE_CACHE_KEY = 'license_validation';
    private const TIER_CACHE_KEY = 'tier_info';
    private const CACHE_EXPIRATION = 300; // 5 minutes (Intufind settings pages always get fresh data)
    private const TIER_CACHE_EXPIRATION = 3600; // 1 hour for tier info

    private CacheService $cache;
    private Logger $logger;
    private AIClientManager $aiClientManager;

    /**
     * Request-level cache to prevent duplicate API calls within same request
     */
    private ?array $requestLevelCache = null;

    /**
     * Constructor
     *
     * @param CacheService $cache  Cache service
     * @param Logger       $logger Logger service
     * @param AIClientManager $aiClientManager AI Client Manager for API calls
     */
    public function __construct(CacheService $cache, Logger $logger, AIClientManager $aiClientManager)
    {
        $this->cache = $cache;
        $this->logger = $logger;
        $this->aiClientManager = $aiClientManager;
        $this->initializeHooks();
    }

    /**
     * Initialize WordPress hooks
     *
     * @return void
     */
    private function initializeHooks(): void
    {
        add_action('admin_notices', [$this, 'maybeShowApiKeyErrorNotices']);
        add_action('intucart_cleanup_initial_sync_notice', [$this, 'cleanupInitialSyncNotice']);

        // Listen for usage limit exceeded events to update cache immediately
        add_action(Constants::USAGE_LIMIT_EXCEEDED_ACTION, [$this, 'markUsageLimitExceeded']);

        // Handle API key form submission
        add_action('admin_post_intucart_api_key_form', [$this, 'handleApiKeyForm']);
    }

    /**
     * Check if current page is an Intufind settings page
     * Used to determine if we should force fresh license validation
     *
     * @return bool True if on an Intufind settings page
     */
    private function isIntufindSettingsPage(): bool
    {
        if (!is_admin() || wp_doing_ajax()) {
            return false;
        }

        // Check via get_current_screen() if available (most reliable)
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if ($screen && strpos($screen->id, 'intufind') !== false) {
            return true;
        }

        // Fallback: check page parameter
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $page = isset($_GET['page']) ? sanitize_text_field($_GET['page']) : '';
        if (strpos($page, 'intucart') !== false || strpos($page, 'intufind') !== false) {
            return true;
        }

        return false;
    }

    /**
     * Get the current API key
     *
     * @return string Current API key (empty string if not set)
     */
    public function getApiKey(): string
    {
        return trim(get_option(Constants::API_KEY_OPTION, ''));
    }

    /**
     * Get the publishable key for widget/client-side use
     *
     * @return string Publishable key (empty string if not set)
     */
    public function getPublishableKey(): string
    {
        return trim(get_option(Constants::PUBLISHABLE_KEY_OPTION, ''));
    }

    /**
     * Check if a publishable key is configured
     *
     * @return bool True if publishable key is set
     */
    public function hasPublishableKey(): bool
    {
        return !empty($this->getPublishableKey());
    }

    /**
     * Store the publishable key
     *
     * @param string $publishableKey The publishable key to store
     * @return bool True on success
     */
    public function storePublishableKey(string $publishableKey): bool
    {
        return update_option(Constants::PUBLISHABLE_KEY_OPTION, $publishableKey);
    }

    /**
     * Check if an API key is configured
     *
     * @return bool True if API key is set
     */
    public function hasApiKey(): bool
    {
        return $this->aiClientManager->hasApiKey();
    }

    /**
     * Get user's tier from cloud service
     *
     * @return string|null Tier name from cloud service, null if no valid API key
     */
    public function getUserTier(): ?string
    {
        $tier_info = $this->getTierInfo();

        if ($tier_info && isset($tier_info['name'])) {
            return $tier_info['name'];
        }

        // Return null if unable to get valid tier data - no misleading fallbacks
        return null;
    }

    /**
     * Validate API key with caching
     * 
     * Cache strategy:
     * - Intufind settings pages: Always fresh (user expects current status)
     * - All other pages: 5-minute cache to reduce latency
     *
     * @param bool $forceRefresh Force a fresh API call (bypass cache)
     * @return array Validation result with success status and basic info
     */
    public function validateLicense(bool $forceRefresh = false): array
    {
        // Only bypass cache on Intufind settings pages (where user expects fresh status)
        $shouldBypassCache = $forceRefresh || $this->isIntufindSettingsPage();

        // Check request-level cache (within same PHP request)
        if (!$shouldBypassCache && $this->requestLevelCache !== null) {
            return $this->requestLevelCache;
        }

        // Check persistent cache (5-minute TTL handles expiration)
        if (!$shouldBypassCache) {
            $cached_result = $this->cache->get(self::LICENSE_CACHE_KEY, self::CACHE_GROUP);
            if ($cached_result !== false && is_array($cached_result)) {
                $this->requestLevelCache = $cached_result;
                return $cached_result;
            }
        }

        if (!$this->aiClientManager->hasApiKey()) {
            $result = [
                'valid' => false,
                'timestamp' => time(),
                'message' => 'No Secret Key configured',
            ];
            $this->updateLicenseCache($result);
            return $result;
        }

        try {
            $aiClient = $this->aiClientManager->getClient();
            $tenantStatus = $aiClient->tenant()->getStatus();
            
            $subscription = $tenantStatus['subscription'] ?? [];
            $validation = $subscription['validation'] ?? [];
            
            $result = [
                'valid' => $subscription['valid'] ?? false,
                'tier' => $subscription['tier'] ?? null,
                'expires' => $subscription['expires'] ?? null,
                'provider' => $subscription['provider'] ?? null,
                'timestamp' => time(),
                'validation' => [
                    'lastValidatedAt' => $validation['lastValidatedAt'] ?? null,
                    'nextValidationAt' => $validation['nextValidationAt'] ?? null,
                    'status' => $validation['status'] ?? null,
                    'revalidated' => $validation['revalidated'] ?? false,
                    'message' => $validation['message'] ?? null,
                ],
                'tier_limits' => $subscription['limits'] ?? null,
            ];
            
            // Auto-store publishable key from cloud response (for widget authentication)
            $publishableKey = $tenantStatus['publishableKey'] ?? null;
            if (!empty($publishableKey)) {
                $currentKey = $this->getPublishableKey();
                if ($currentKey !== $publishableKey) {
                $this->storePublishableKey($publishableKey);
                    $this->logger->info('Publishable key stored from cloud validation response', [
                        'was_empty' => empty($currentKey),
                    ]);
                }
            }
            
            if (!$result['valid']) {
                $this->logger->warning('API key validation failed from cloud', [
                    'status' => $validation['status'] ?? 'unknown',
                    'message' => $validation['message'] ?? 'API key not valid',
                ]);
            }
            
            $this->updateLicenseCache($result);
            return $result;
            
        } catch (\Intufind\AI\Exceptions\AuthenticationException $e) {
            // Authentication errors (401) - API key is genuinely invalid
            $this->logger->error('API key authentication failed', [
                'error' => $e->getMessage(),
            ]);
            
            $result = [
                'valid' => false,
                'timestamp' => time(),
                'error' => $e->getMessage(),
                'validation' => [
                    'status' => 'invalid',
                    'message' => 'Secret Key is invalid or expired. Please check your Secret Key.',
                ],
            ];
            
            // Cache invalid result for shorter duration to allow quick retry after fix
            $this->cache->set(self::LICENSE_CACHE_KEY, $result, self::CACHE_GROUP, 60);
            $this->requestLevelCache = $result;
            return $result;
            
        } catch (\Intufind\AI\Exceptions\TrialExpiredException $e) {
            // Trial expired (402) - User needs to subscribe
            $this->logger->warning('Free trial has expired', [
                'error' => $e->getMessage(),
                'trial_ended_at' => $e->getTrialEndedAt(),
                'upgrade_url' => $e->getUpgradeUrl(),
            ]);
            
            $result = [
                'valid' => false,
                'timestamp' => time(),
                'error' => $e->getMessage(),
                'validation' => [
                    'status' => 'trial_expired',
                    'message' => 'Your free trial has ended. Please subscribe to continue using Intufind.',
                    'trial_ended_at' => $e->getTrialEndedAt(),
                    'upgrade_url' => $e->getUpgradeUrl(),
                ],
            ];
            
            // Cache trial expired result for 5 minutes (they might subscribe soon)
            $this->cache->set(self::LICENSE_CACHE_KEY, $result, self::CACHE_GROUP, 300);
            $this->requestLevelCache = $result;
            return $result;
            
        } catch (\Exception $e) {
            $this->logger->error('Failed to validate API key with cloud', [
                'error' => $e->getMessage(),
            ]);
            
            // Assume valid on transient/network errors to avoid disruption
            // This prevents temporary outages from breaking the plugin
            $result = [
                'valid' => true,
                'timestamp' => time(),
                'error' => $e->getMessage(),
                'validation' => [
                    'status' => 'error',
                    'message' => 'Could not verify Secret Key status: ' . $e->getMessage(),
                ],
            ];
            
            $this->cache->set(self::LICENSE_CACHE_KEY, $result, self::CACHE_GROUP, 300);
            $this->requestLevelCache = $result;
            return $result;
        }
    }

    /**
     * Clear all API key-related caches
     *
     * @return void
     */
    public function clearLicenseCache(): void
    {
        // Clear persistent cache
        $this->cache->delete(self::LICENSE_CACHE_KEY, self::CACHE_GROUP);
        $this->cache->delete(self::TIER_CACHE_KEY, self::CACHE_GROUP);

        // Clear request-level cache
        $this->requestLevelCache = null;

        $this->logger->debug('Cleared all API key caches');
    }

    /**
     * Mark the license as trial expired and cache the result
     * 
     * Called by services (Search, Recommendations) when they catch a TrialExpiredException.
     * This updates the cache so subsequent requests know the trial is expired
     * WITHOUT making additional API calls.
     *
     * @param \Intufind\AI\Exceptions\TrialExpiredException $exception The caught exception
     * @return void
     */
    public function markTrialExpired(\Intufind\AI\Exceptions\TrialExpiredException $exception): void
    {
        $result = [
            'valid' => false,
            'timestamp' => time(),
            'error' => $exception->getMessage(),
            'validation' => [
                'status' => 'trial_expired',
                'message' => 'Your free trial has ended. Please subscribe to continue using Intufind.',
                'trial_ended_at' => $exception->getTrialEndedAt(),
                'upgrade_url' => $exception->getUpgradeUrl(),
            ],
        ];

        // Cache for 5 minutes - they might subscribe soon
        $this->cache->set(self::LICENSE_CACHE_KEY, $result, self::CACHE_GROUP, 300);
        $this->requestLevelCache = $result;

        $this->logger->info('Marked license as trial expired from API response', [
            'trial_ended_at' => $exception->getTrialEndedAt(),
        ]);
    }

    /**
     * Update cached data to reflect usage limit exceeded
     * This immediately disables features without waiting for cache expiry
     * Called via WordPress action when AIClientManager service detects a 429 response
     *
     * @param string $feature Feature that hit usage limit (e.g., 'chat', 'search')
     * @return void
     */
    public function markUsageLimitExceeded(string $feature): void
    {
        // Get current cached validation data
        $cached_result = $this->cache->get(self::LICENSE_CACHE_KEY, self::CACHE_GROUP);
        
        if ($cached_result === false || !is_array($cached_result)) {
            // No cache to update, clear everything to force fresh validation
            $this->clearLicenseCache();
            return;
        }

        // Update the cached data to show no remaining usage for the feature
        if (isset($cached_result['usage']['remaining'])) {
            switch ($feature) {
                case 'chat':
                    $cached_result['usage']['remaining']['chatMessages'] = 0;
                    break;
                case 'search':
                    $cached_result['usage']['remaining']['searchQueries'] = 0;
                    break;
                case 'recommendations':
                    $cached_result['usage']['remaining']['recommendations'] = 0;
                    break;
            }

            // Update both persistent and request-level cache
            $this->cache->set(self::LICENSE_CACHE_KEY, $cached_result, self::CACHE_GROUP, self::CACHE_EXPIRATION);
            $this->requestLevelCache = $cached_result;
        } else {
            // Cache doesn't have usage data structure, clear it to force fresh validation
            $this->logger->warning("Cache missing usage data structure, clearing cache", [
                'feature' => $feature,
                'cache_keys' => array_keys($cached_result)
            ]);
            $this->clearLicenseCache();
        }
    }

    /**
     * Update cache with fresh validation data
     * This is more efficient than clearing cache when we have fresh data
     *
     * @param array $validation Fresh validation data from cloud service
     * @return void
     */
    public function updateLicenseCache(array $validation): void
    {
        // Only cache successful validation results
        if (isset($validation['valid']) && $validation['valid'] === true) {
            $this->cache->set(self::LICENSE_CACHE_KEY, $validation, self::CACHE_GROUP, self::CACHE_EXPIRATION);

            // Also update request-level cache
            $this->requestLevelCache = $validation;
        } else {
            // For failed validations, clear the cache to prevent stale data
            $this->clearLicenseCache();

            $this->logger->debug('Cleared cache due to validation failure', [
                'error' => $validation['error_message'] ?? 'Unknown error'
            ]);
        }
    }

    /**
     * Show admin notice for API key validation errors
     *
     * @return void
     */
    public function maybeShowApiKeyErrorNotices(): void
    {
        // Only show on Intucart admin pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'intufind') === false) {
            return;
        }

        // Skip on the License/API Key tab - status is already shown inline there
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $currentTab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        if (empty($currentTab) || $currentTab === 'license') {
            return;
        }

        // Skip if no API key - the main notice handles this
        if (!$this->hasApiKey()) {
            return;
        }

        // Check for validation errors
        try {
            $validation = $this->validateLicense();
            if (is_array($validation) && !($validation['valid'] ?? false)) {
                $status = $validation['validation']['status'] ?? null;
                $message = $validation['validation']['message'] ?? null;

                if ($status && $message) {
                    $error_message = null;
                    $notice_type = 'warning';
                    $action_link = '';
                    
                    switch ($status) {
                        case 'trial_expired':
                            $error_message = 'Your free trial has ended. Search, chatbot, and recommendation features are currently disabled.';
                            $notice_type = 'error';
                            $upgrade_url = $validation['validation']['upgrade_url'] ?? 'https://intufind.com/dashboard/subscription';
                            $action_link = ' <a href="' . esc_url($upgrade_url) . '" target="_blank" class="button button-primary" style="margin-left: 10px;">Subscribe Now</a>';
                            break;
                        case 'expired':
                            $error_message = 'Your subscription has expired. Please renew to continue using all features.';
                            break;
                        case 'disabled':
                        case 'revoked':
                            $error_message = 'Your Secret Key has been revoked. Please generate a new one from the dashboard.';
                            break;
                        case 'invalid':
                            $error_message = 'Your Secret Key is invalid. Please check it and try again.';
                            break;
                    }

                    // Only show specific error messages, skip generic validation failures
                    if ($error_message) {
                        echo '<div class="notice notice-' . esc_attr($notice_type) . ' is-dismissible">';
                        echo '<p><strong>Intufind:</strong> ' . esc_html($error_message) . $action_link . '</p>';
                        echo '</div>';
                    }
                }
            }
        } catch (\Exception $e) {
            // Silently handle validation exceptions to avoid spamming users with notices
            $this->logger->debug('API key validation check failed in error notice display', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Handle API key form submission (save + validate)
     *
     * @return void
     */
    public function handleApiKeyForm(): void
    {
        // Security checks
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have permission to perform this action.', 'intufind'));
        }

        if (!check_admin_referer('intucart_api_key_form', 'intucart_api_key_nonce')) {
            wp_die(__('Security check failed.', 'intufind'));
        }

        // Get the submitted API key and action
        $new_api_key = isset($_POST['api_key']) ? trim(sanitize_text_field($_POST['api_key'])) : '';
        $action = isset($_POST['api_key_action']) ? sanitize_text_field($_POST['api_key_action']) : 'save';

        switch ($action) {
            case 'save':
                $this->handleSaveApiKey($new_api_key);
                break;

            case 'clear':
                $this->handleClearApiKey();
                break;

            case 'refresh':
                $this->handleRefreshLicenseStatus();
                break;

            default:
                $this->addAdminNotice('Invalid action specified.', 'error');
                break;
        }

        // Redirect back to API key tab
        wp_safe_redirect(admin_url('admin.php?page=intucart&tab=license'));
        exit;
    }

    /**
     * Validate API key format
     * Secret keys start with 'if_sk_', publishable keys with 'if_pk_'
     *
     * @param string $api_key The API key to validate
     * @return array{valid: bool, type: string|null, error: string|null}
     */
    private function validateApiKeyFormat(string $api_key): array
    {
        if (empty($api_key)) {
            return ['valid' => false, 'type' => null, 'error' => 'Secret Key cannot be empty'];
        }

        // Check for secret key format (required for server-side operations)
        if (strpos($api_key, 'if_sk_') === 0) {
            // Validate length: prefix (6) + key (32) = 38 characters
            if (strlen($api_key) === 38) {
                return ['valid' => true, 'type' => 'secret', 'error' => null];
            }
            return ['valid' => false, 'type' => 'secret', 'error' => 'Invalid secret key format. Secret keys should be 38 characters.'];
        }

        // Check if user accidentally entered a publishable key
        if (strpos($api_key, 'if_pk_') === 0) {
            return [
                'valid' => false,
                'type' => 'publishable',
                'error' => 'This appears to be a publishable key (if_pk_). Please enter your secret key (if_sk_) instead. You can find it in your Intufind dashboard.'
            ];
        }

        // Legacy format or unknown format - allow it but warn
        if (strlen($api_key) >= 20) {
            $this->logger->warning('API key has non-standard format', [
                'prefix' => substr($api_key, 0, 6),
                'length' => strlen($api_key),
            ]);
            return ['valid' => true, 'type' => 'unknown', 'error' => null];
        }

        return ['valid' => false, 'type' => null, 'error' => 'Invalid Secret Key format. Secret Keys should start with "if_sk_" and be 38 characters long.'];
    }

    /**
     * Handle saving and validating a new API key
     *
     * @param string $api_key The API key to save
     * @return void
     */
    private function handleSaveApiKey(string $api_key): void
    {
        if (empty($api_key)) {
            $this->addAdminNotice('Please enter a Secret Key.', 'error');
            return;
        }

        // Validate key format before saving
        $formatValidation = $this->validateApiKeyFormat($api_key);
        if (!$formatValidation['valid']) {
            $this->addAdminNotice($formatValidation['error'], 'error');
            return;
        }

        // Warn if using non-standard format
        if ($formatValidation['type'] === 'unknown') {
            $this->addAdminNotice('Warning: This Secret Key has a non-standard format. It may work, but standard keys start with "if_sk_".', 'warning');
        }

        // Store the API key and clear old credentials
        $this->aiClientManager->storeApiKey($api_key);
        $this->clearLicenseCache();
        $this->aiClientManager->resetClient();
        
        // Clear old publishable key - validateLicense() will fetch the new one from cloud
        delete_option(Constants::PUBLISHABLE_KEY_OPTION);

        // Validate the API key against the cloud
        try {
            $validation = $this->validateLicense();

            if ($validation['valid'] ?? false) {
                // Store tier info if available
                if (!empty($validation['tier'])) {
                    update_option(Constants::TIER_OPTION, $validation['tier']);
                }

                $auto_sync = isset($_POST['auto_sync']) && $_POST['auto_sync'] === '1';

                if ($auto_sync) {
                    // Trigger initial sync
                    $this->triggerInitialSync();

                    $sync_status_url = admin_url('admin.php?page=intucart-status&tab=index');
                    $this->addAdminNotice(
                        'Secret Key validated successfully! Initial content indexing has been started. <a href="' . esc_url($sync_status_url) . '">View indexing progress →</a>',
                        'success'
                    );
                } else {
                    $sync_status_url = admin_url('admin.php?page=intucart-status&tab=index');
                    $this->addAdminNotice(
                        'Secret Key validated successfully! <a href="' . esc_url($sync_status_url) . '">Go to Status page to manually sync content →</a>',
                        'success'
                    );
                }
            } else {
                // API key is invalid - clear it
                $this->aiClientManager->clearCredentials();
                $this->clearLicenseCache();

                $error_message = $validation['validation']['message'] ?? 'Secret Key validation failed.';
                $this->addAdminNotice('Invalid Secret Key: ' . $error_message . ' Please check your Secret Key and try again.', 'error');
            }
        } catch (\Exception $e) {
            // On error, keep the API key but warn the user
            $this->addAdminNotice('Secret Key saved but validation check failed: ' . $e->getMessage() . '. The key will be validated on next use.', 'warning');
        }
    }

    /**
     * Handle clearing the API key
     *
     * @return void
     */
    private function handleClearApiKey(): void
    {
        $this->logger->info('Clearing API key and associated data');

        // Clear credentials and caches
        $this->aiClientManager->clearCredentials();
        $this->clearLicenseCache();
        $this->aiClientManager->resetClient();
        $this->cleanupSyncJobs();

        // Clear publishable key and workspace ID
        delete_option(Constants::PUBLISHABLE_KEY_OPTION);
        delete_option(Constants::WORKSPACE_ID_OPTION);

        $this->addAdminNotice('Secret Key cleared successfully. All plugin features are now disabled.', 'success');
    }

    /**
     * Handle refreshing the API key status
     * Called when user clicks "Refresh Key Status" button after subscribing
     *
     * @return void
     */
    private function handleRefreshLicenseStatus(): void
    {
        $this->logger->info('Refreshing API key status on user request');

        // Clear all caches to force fresh validation
        $this->clearLicenseCache();
        $this->aiClientManager->resetClient();

        // Force fresh validation from cloud
        try {
            $validation = $this->validateLicense();

            if ($validation['valid'] ?? false) {
                $tierName = $validation['tier'] ?? 'your plan';
                $this->addAdminNotice(
                    sprintf(
                        __('License activated! You are now on the %s plan. All features are enabled.', 'intufind'),
                        ucfirst($tierName)
                    ),
                    'success'
                );
                
                // Update tier option
                if (!empty($validation['tier'])) {
                    update_option(Constants::TIER_OPTION, $validation['tier']);
                }
            } else {
                $status = $validation['validation']['status'] ?? 'unknown';
                $message = $validation['validation']['message'] ?? __('License validation failed.', 'intufind');
                
                if ($status === 'trial_expired') {
                    $upgradeUrl = $validation['validation']['upgrade_url'] ?? Constants::WEBSITE_URL . 'dashboard/subscription';
                    $this->addAdminNotice(
                        sprintf(
                            __('Your free trial has ended. %s to continue using all features.', 'intufind'),
                            '<a href="' . esc_url($upgradeUrl) . '" target="_blank">' . __('Subscribe now', 'intufind') . '</a>'
                        ),
                        'warning'
                    );
                } else {
                    $this->addAdminNotice($message, 'error');
                }
            }
        } catch (\Exception $e) {
            $this->logger->error('Failed to refresh API key status', [
                'error' => $e->getMessage()
            ]);
            $this->addAdminNotice(
                __('Could not verify Secret Key status. Please try again in a moment.', 'intufind'),
                'error'
            );
        }
    }

    /**
     * Trigger initial sync when an API key is successfully validated
     *
     * @return void
     */
    private function triggerInitialSync(): void
    {
        try {
            $this->logger->info('Triggering initial sync after API key validation');

            // Check if WordPress cron is working properly
            $this->diagnoseCronSystem();

            // Fire action hook to trigger full sync via CronManager
            $this->logger->info('About to trigger full sync action', [
                'hook' => Constants::TRIGGER_FULL_SYNC_HOOK,
                'group' => 'intucart-api-key-activation',
                'delay' => 1
            ]);
            do_action(Constants::TRIGGER_FULL_SYNC_HOOK, 'intucart-api-key-activation', 1);
            $this->logger->info('Full sync action triggered successfully');

            // Set a transient to show sync status message
            set_transient('intucart_initial_sync_triggered', true, 600); // 10 minutes

            // Schedule cleanup of the sync transient after sync should be complete
            $this->scheduleCleanupAction(600);

            // Verify that cron jobs were scheduled
            $this->verifyCronScheduling();

            // Attempt to kick off wp-cron immediately
            try {
                if (defined('DISABLE_WP_CRON') && \DISABLE_WP_CRON) {
                    $this->logger->warning('WP-Cron is disabled. Running sync handlers inline as fallback.');
                    do_action(Constants::POST_SYNC_HOOK);
                    do_action(Constants::TAXONOMY_SYNC_HOOK);
                } else {
                    if (function_exists('spawn_cron')) {
                        spawn_cron();
                        $this->logger->debug('spawn_cron() invoked to trigger scheduled events');
                    } else {
                        $cron_url = site_url('wp-cron.php?doing_wp_cron=' . microtime(true));
                        wp_remote_post($cron_url, [
                            'timeout' => 0.01,
                            'blocking' => false,
                            'sslverify' => apply_filters('https_local_ssl_verify', false),
                        ]);
                        $this->logger->debug('wp-cron trigger request sent', [
                            'url' => $cron_url
                        ]);
                    }
                }
            } catch (\Exception $e) {
                $this->logger->warning('Failed to trigger wp-cron; running sync handlers inline as fallback', [
                    'error' => $e->getMessage()
                ]);
                do_action(Constants::POST_SYNC_HOOK);
                do_action(Constants::TAXONOMY_SYNC_HOOK);
            }

            $this->logger->info('Initial sync triggered successfully via action hook');
        } catch (\Exception $e) {
            $this->logger->error('Failed to trigger initial sync after API key validation', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Schedule cleanup action using WordPress cron
     *
     * @param int $delay Delay in seconds
     * @return void
     */
    private function scheduleCleanupAction(int $delay): void
    {
        wp_schedule_single_event(time() + $delay, 'intucart_cleanup_initial_sync_notice');
        $this->logger->debug('Cleanup scheduled using WordPress cron', [
            'delay' => $delay,
            'hook' => 'intucart_cleanup_initial_sync_notice',
            'scheduled_time' => time() + $delay
        ]);
    }

    /**
     * Clean up initial sync notice transient
     *
     * @return void
     */
    public function cleanupInitialSyncNotice(): void
    {
        delete_transient('intucart_initial_sync_triggered');
        $this->logger->info('Cleaned up initial sync notice transient');
    }

    /**
     * Diagnose WordPress cron system issues
     *
     * @return void
     */
    private function diagnoseCronSystem(): void
    {
        $wp_cron_disabled = defined('DISABLE_WP_CRON') && \DISABLE_WP_CRON;
        $wp_doing_cron = defined('DOING_CRON') && DOING_CRON;

        $cron_url = site_url('wp-cron.php');

        $this->logger->info('WordPress cron system diagnosis', [
            'wp_cron_disabled' => $wp_cron_disabled,
            'doing_cron' => $wp_doing_cron,
            'cron_url' => $cron_url,
            'current_time' => time(),
            'current_time_formatted' => date('Y-m-d H:i:s', time()),
            'gmt_offset' => get_option('gmt_offset'),
            'timezone_string' => get_option('timezone_string'),
        ]);

        if ($wp_cron_disabled) {
            $this->logger->warning('WordPress cron is disabled (DISABLE_WP_CRON = true). Sync may not execute automatically.');
        }
    }

    /**
     * Verify that cron jobs were actually scheduled
     *
     * @return void
     */
    private function verifyCronScheduling(): void
    {
        $post_sync_next = wp_next_scheduled(Constants::POST_SYNC_HOOK);
        $taxonomy_sync_next = wp_next_scheduled(Constants::TAXONOMY_SYNC_HOOK);

        $this->logger->info('Cron scheduling verification', [
            'post_sync_scheduled' => $post_sync_next !== false,
            'post_sync_time' => $post_sync_next ? date('Y-m-d H:i:s', $post_sync_next) : 'not scheduled',
            'taxonomy_sync_scheduled' => $taxonomy_sync_next !== false,
            'taxonomy_sync_time' => $taxonomy_sync_next ? date('Y-m-d H:i:s', $taxonomy_sync_next) : 'not scheduled',
            'current_time' => time(),
        ]);
    }

    /**
     * Get all scheduled Intucart cron jobs for debugging
     *
     * @return array Array of scheduled cron jobs
     */
    public function getScheduledCronJobs(): array
    {
        $intucart_hooks = [
            Constants::POST_SYNC_HOOK,
            Constants::TAXONOMY_SYNC_HOOK,
            Constants::TRIGGER_FULL_SYNC_HOOK,
            'intucart_cleanup_initial_sync_notice',
        ];

        $scheduled_jobs = [];

        foreach ($intucart_hooks as $hook) {
            $next_run = wp_next_scheduled($hook);
            if ($next_run) {
                $scheduled_jobs[] = [
                    'hook' => $hook,
                    'next_run' => $next_run,
                    'next_run_formatted' => date('Y-m-d H:i:s', $next_run),
                    'time_until' => $next_run - time(),
                    'time_until_formatted' => human_time_diff(time(), $next_run),
                ];
            }
        }

        return $scheduled_jobs;
    }

    /**
     * Get cached tier information or fetch from cloud
     *
     * @return array|null Tier information or null if unavailable
     */
    public function getTierInfo(): ?array
    {
        // Check cache first
        $cached_tier = $this->cache->get(self::TIER_CACHE_KEY, self::CACHE_GROUP);
        if ($cached_tier !== false && is_array($cached_tier)) {
            return $cached_tier;
        }

        // Fetch fresh tier info
        $tier_info = $this->fetchTierInfo();

        // Cache the result if we got valid data
        if ($tier_info && is_array($tier_info)) {
            $this->cache->set(self::TIER_CACHE_KEY, $tier_info, self::CACHE_GROUP, self::TIER_CACHE_EXPIRATION);
        }

        return $tier_info;
    }

    /**
     * Add an admin notice
     *
     * @param string $message Notice message
     * @param string $type Notice type (success, error, warning, info)
     * @return void
     */
    private function addAdminNotice(string $message, string $type = 'info'): void
    {
        $notices = get_transient('intucart_admin_notices') ?: [];
        $notices[] = [
            'message' => $message,
            'type' => $type,
            'timestamp' => time()
        ];
        set_transient('intucart_admin_notices', $notices, 300); // 5 minutes
    }

        /**
     * Clean up scheduled sync jobs and sync-related transients when API key is cleared
     *
     * @return void
     */
    private function cleanupSyncJobs(): void
    {
        try {
            // Clear WordPress scheduled sync events
            $hooks = [
                Constants::POST_SYNC_HOOK,
                Constants::TAXONOMY_SYNC_HOOK,
                Constants::TRIGGER_FULL_SYNC_HOOK,
            ];

            foreach ($hooks as $hook) {
                $timestamp = wp_next_scheduled($hook);
                while ($timestamp) {
                    wp_unschedule_event($timestamp, $hook);
                    $timestamp = wp_next_scheduled($hook);
                }
            }

            // Clean up sync-related transients and notices
            delete_transient('intucart_initial_sync_triggered');
            delete_transient('intucart_sync_in_progress');

            // Clean up cleanup notice scheduled event
            $cleanup_timestamp = wp_next_scheduled('intucart_cleanup_initial_sync_notice');
            if ($cleanup_timestamp) {
                wp_unschedule_event($cleanup_timestamp, 'intucart_cleanup_initial_sync_notice');
            }

            $this->logger->info('Cleaned up scheduled sync jobs and transients after API key removal');
        } catch (\Exception $e) {
            $this->logger->error('Failed to cleanup sync jobs after API key removal', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Check if the license is valid and features can be used
     * 
     * IMPORTANT: This is a NON-BLOCKING check for frontend widget loading.
     * It only reads from cache and NEVER makes an API call.
     * If cache is empty, we optimistically assume valid (if API key exists).
     * The actual validation happens on admin pages or background tasks.
     *
     * @return bool True if license appears valid (cached or optimistic)
     */
    public function isValid(): bool
    {
        // If no API key, definitely not valid
        if (!$this->aiClientManager->hasApiKey()) {
            return false;
        }

        // Check request-level cache first (free, in-memory)
        if ($this->requestLevelCache !== null) {
            return $this->requestLevelCache['valid'] ?? false;
        }

        // Check persistent cache (transient or object cache - very fast)
        $cached_result = $this->cache->get(self::LICENSE_CACHE_KEY, self::CACHE_GROUP);
        if ($cached_result !== false && is_array($cached_result)) {
            // Store in request cache for subsequent calls in same request
            $this->requestLevelCache = $cached_result;
            return $cached_result['valid'] ?? false;
        }

        // IMPORTANT: Cache is empty, but we DON'T make an API call here
        // This would add latency to frontend page loads
        // Instead, be optimistic: if API key exists, assume valid
        // The cloud will reject invalid requests anyway, and validation
        // happens on admin pages where latency is acceptable
        return true;
    }

    /**
     * Check if trial has expired (non-blocking, cache-only)
     *
     * @return bool True if trial has expired (only if cached)
     */
    public function isTrialExpired(): bool
    {
        // Check request-level cache first
        if ($this->requestLevelCache !== null) {
            return ($this->requestLevelCache['validation']['status'] ?? '') === 'trial_expired';
        }

        // Check persistent cache
        $cached_result = $this->cache->get(self::LICENSE_CACHE_KEY, self::CACHE_GROUP);
        if ($cached_result !== false && is_array($cached_result)) {
            $this->requestLevelCache = $cached_result;
            return ($cached_result['validation']['status'] ?? '') === 'trial_expired';
        }

        // No cached data - don't know if expired, assume not
        return false;
    }

    /**
     * Get trial expiration details (non-blocking, cache-only)
     *
     * @return array|null Trial expiration info or null if not expired/not cached
     */
    public function getTrialExpirationInfo(): ?array
    {
        if (!$this->isTrialExpired()) {
            return null;
        }

        // We know we have cached data from isTrialExpired() call
        $validation = $this->requestLevelCache ?? [];
        return [
            'message' => $validation['validation']['message'] ?? 'Your free trial has ended.',
            'trial_ended_at' => $validation['validation']['trial_ended_at'] ?? null,
            'upgrade_url' => $validation['validation']['upgrade_url'] ?? 'https://intufind.com/dashboard/subscription',
        ];
    }

    /**
     * Check if semantic search is available
     *
     * @return bool True if semantic search is available
     */
    public function hasSemanticSearch(): bool
    {
        try {
            $validation = $this->validateLicense();
            if (
                is_array($validation)
                && ($validation['valid'] ?? false)
                && isset($validation['tier_limits'])
                && is_array($validation['tier_limits'])
            ) {
                $limits = $validation['tier_limits'];
                if (isset($limits['search']) && is_array($limits['search'])) {
                    $search = $limits['search'];
                    if (isset($search['semantic']) && is_array($search['semantic']) && array_key_exists('enabled', $search['semantic'])) {
                        return (bool) $search['semantic']['enabled'];
                    }
                }
            }
        } catch (\Exception $e) {
            $this->logger->debug('Cloud limits check for semantic search failed; falling back to feature list', [
                'error' => $e->getMessage(),
            ]);
        }

        // Fallback to legacy feature list if present
        $tier_info = $this->getTierInfo();
        $features = $tier_info['features'] ?? [];
        return in_array('semantic_search', $features, true);
    }

    /**
     * Check if AI recommendations are available
     *
     * @return bool True if AI recommendations are available
     */
    public function hasRecommendations(): bool
    {
        try {
            $validation = $this->validateLicense();
            if (
                is_array($validation)
                && ($validation['valid'] ?? false)
                && isset($validation['tier_limits']['recommendations'])
                && is_array($validation['tier_limits']['recommendations'])
            ) {
                $recs = $validation['tier_limits']['recommendations'];
                $enabled = (bool)($recs['enabled'] ?? false);
                $perMonth = (int)($recs['perMonth'] ?? 0);
                return $enabled && $perMonth > 0;
            }
        } catch (\Exception $e) {
            $this->logger->debug('Cloud limits check for recommendations failed; falling back to feature list', [
                'error' => $e->getMessage(),
            ]);
        }

        // Fallback to legacy feature list if present
        $tier_info = $this->getTierInfo();
        $features = $tier_info['features'] ?? [];
        return in_array('recommendations', $features, true);
    }

    /**
     * Check if chat/assistant features are available
     *
     * @return bool True if chat is available
     */
    public function hasChat(): bool
    {
        try {
            $validation = $this->validateLicense();
            if (
                is_array($validation)
                && ($validation['valid'] ?? false)
                && isset($validation['tier_limits']['chat'])
                && is_array($validation['tier_limits']['chat'])
            ) {
                $chat = $validation['tier_limits']['chat'];
                $enabled = (bool)($chat['enabled'] ?? false);
                $messagesPerMonth = (int)($chat['messagesPerMonth'] ?? 0);
                return $enabled && $messagesPerMonth > 0;
            }
        } catch (\Exception $e) {
            $this->logger->debug('Cloud limits check for chat failed; falling back to feature list', [
                'error' => $e->getMessage(),
            ]);
        }

        // Fallback to legacy feature list if present
        $tier_info = $this->getTierInfo();
        $features = $tier_info['features'] ?? [];
        return in_array('chat', $features, true);
    }

    /**
     * Check if analytics features are available
     *
     * @return bool True if analytics are available
     */
    public function hasAnalytics(): bool
    {
        try {
            $validation = $this->validateLicense();
            if (
                is_array($validation)
                && ($validation['valid'] ?? false)
                && isset($validation['tier_limits'])
                && is_array($validation['tier_limits'])
                && array_key_exists('allowAnalytics', $validation['tier_limits'])
            ) {
                return (bool) $validation['tier_limits']['allowAnalytics'];
            }
        } catch (\Exception $e) {
            $this->logger->debug('Cloud limits check for analytics failed; falling back to feature list', [
                'error' => $e->getMessage(),
            ]);
        }

        // Fallback to legacy feature list if present
        $tier_info = $this->getTierInfo();
        $features = $tier_info['features'] ?? [];
        return in_array('analytics', $features, true);
    }

    /**
     * Fetch tier information from stored options or cloud
     *
     * @return array|null Tier information or null if unavailable
     */
    private function fetchTierInfo(): ?array
    {
        // Check if we have a valid API key
        if (!$this->aiClientManager->hasApiKey()) {
            return null;
        }

        // Get tier from stored option
        $tierName = get_option(Constants::TIER_OPTION, '');
        
        if (empty($tierName)) {
            // Try to get from validation
            $validation = $this->validateLicense();
            if (($validation['valid'] ?? false) && !empty($validation['tier'])) {
                $tierName = $validation['tier'];
                update_option(Constants::TIER_OPTION, $tierName);
            }
        }
        
        if (empty($tierName)) {
            return null;
        }

        return [
            'name' => $tierName,
            'displayName' => ucfirst($tierName),
            'features' => [], // Features are enforced server-side
            'limits' => [],   // Limits are enforced server-side
            'valid' => true,
            'expires' => null,
        ];
    }

    /**
     * Invalidate cached tier information
     * Call this when API key changes or tier updates are detected
     */
    public function invalidateTierCache(): void
    {
        $this->cache->delete(self::TIER_CACHE_KEY, self::CACHE_GROUP);
        $this->logger->info('Invalidated tier information cache');
    }

    /**
     * Get tier name for display purposes
     *
     * @return string|null Tier display name, null if no valid API key
     */
    public function getTierName(): ?string
    {
        $tier_info = $this->getTierInfo();

        if (!$tier_info) {
            $userTier = $this->getUserTier();
            return $userTier ? ucfirst($userTier) : null;
        }

        return $tier_info['displayName'] ?? ($tier_info['name'] ? ucfirst($tier_info['name']) : null);
    }

}
