<?php

namespace Intucart\Services\Licensing;

use Intucart\Services\Constants;
use Intucart\Services\Logger;

/**
 * Self-hosted Plugin Updater
 * 
 * Checks the Intufind website for plugin updates and integrates with
 * the WordPress update system. This is a simplified replacement for
 * the EDD Software Licensing updater.
 */
class PluginUpdater
{
    private Logger $logger;
    private string $pluginFile;
    private string $pluginSlug;
    private string $currentVersion;
    private string $updateCheckUrl;
    private string $cacheKey;

    /**
     * Constructor
     *
     * @param Logger $logger Logger service
     */
    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
        $this->pluginFile = plugin_dir_path(dirname(dirname(dirname(__FILE__)))) . 'intucart.php';
        $this->pluginSlug = 'intufind-ai';
        $this->currentVersion = Constants::PLUGIN_VERSION;
        $this->updateCheckUrl = Constants::UPDATE_CHECK_URL;
        $this->cacheKey = 'intufind_update_check';
        
        $this->initHooks();
    }

    /**
     * Initialize WordPress hooks
     */
    private function initHooks(): void
    {
        // Hook into WordPress update check
        add_filter('pre_set_site_transient_update_plugins', [$this, 'checkForUpdate']);
        
        // Hook into plugin information modal
        add_filter('plugins_api', [$this, 'pluginInfo'], 10, 3);
        
        // Show update message in plugin row
        add_action('in_plugin_update_message-' . plugin_basename($this->pluginFile), [$this, 'updateMessage'], 10, 2);
    }

    /**
     * Check for plugin updates
     *
     * @param object $transient WordPress update transient
     * @return object Modified transient
     */
    public function checkForUpdate($transient): object
    {
        if (!is_object($transient)) {
            $transient = new \stdClass();
        }

        // Get cached update info or fetch fresh
        $updateInfo = $this->getUpdateInfo();
        
        if (!$updateInfo) {
            return $transient;
        }

        // Check if there's a newer version
        if (version_compare($this->currentVersion, $updateInfo->new_version, '<')) {
            // Add to update list
            $transient->response[plugin_basename($this->pluginFile)] = $updateInfo;
            
            $this->logger->debug('Update available', [
                'current' => $this->currentVersion,
                'new' => $updateInfo->new_version
            ]);
        } else {
            // No update available, but WordPress needs this for auto-update support
            $transient->no_update[plugin_basename($this->pluginFile)] = $updateInfo;
        }

        return $transient;
    }

    /**
     * Get plugin information for the "View details" modal
     *
     * @param mixed $result Current result
     * @param string $action API action
     * @param object $args Request arguments
     * @return mixed Plugin info or original result
     */
    public function pluginInfo($result, string $action, $args)
    {
        if ($action !== 'plugin_information') {
            return $result;
        }

        if (!isset($args->slug) || $args->slug !== $this->pluginSlug) {
            return $result;
        }

        // Fetch plugin information from our API
        $response = $this->fetchFromApi('plugin_information');
        
        if (!$response) {
            return $result;
        }

        // Convert to object with expected structure
        $pluginInfo = new \stdClass();
        $pluginInfo->name = $response['name'] ?? Constants::PLUGIN_NAME;
        $pluginInfo->slug = $response['slug'] ?? $this->pluginSlug;
        $pluginInfo->version = $response['version'] ?? $this->currentVersion;
        $pluginInfo->author = $response['author'] ?? 'Intufind';
        $pluginInfo->author_profile = $response['author_profile'] ?? Constants::WEBSITE_URL;
        $pluginInfo->requires = $response['requires'] ?? '6.0';
        $pluginInfo->tested = $response['tested'] ?? get_bloginfo('version');
        $pluginInfo->requires_php = $response['requires_php'] ?? '8.0';
        $pluginInfo->homepage = $response['homepage'] ?? Constants::WEBSITE_URL;
        $pluginInfo->download_link = $response['download_link'] ?? '';
        $pluginInfo->sections = $response['sections'] ?? [];
        $pluginInfo->banners = $response['banners'] ?? [];
        $pluginInfo->icons = $response['icons'] ?? [];
        $pluginInfo->last_updated = $response['last_updated'] ?? '';

        return $pluginInfo;
    }

    /**
     * Show update message in plugin row
     *
     * @param array $pluginData Plugin data
     * @param object $response Update response
     */
    public function updateMessage(array $pluginData, $response): void
    {
        // Show any custom message if needed
        // For now, WordPress default message is sufficient
    }

    /**
     * Get update information (with caching)
     *
     * @return object|null Update info object or null
     */
    private function getUpdateInfo(): ?object
    {
        // Check cache first
        $cached = get_transient($this->cacheKey);
        if ($cached !== false) {
            return $cached;
        }

        // Fetch from API
        $response = $this->fetchFromApi();
        
        if (!$response) {
            // Cache negative result for 1 hour to avoid hammering the server
            set_transient($this->cacheKey, null, HOUR_IN_SECONDS);
            return null;
        }

        // Build update object
        $updateInfo = new \stdClass();
        $updateInfo->slug = $response['slug'] ?? $this->pluginSlug;
        $updateInfo->plugin = plugin_basename($this->pluginFile);
        $updateInfo->new_version = $response['new_version'] ?? $this->currentVersion;
        $updateInfo->url = $response['url'] ?? Constants::WEBSITE_URL;
        $updateInfo->package = $response['package'] ?? '';
        $updateInfo->requires = $response['requires'] ?? '6.0';
        $updateInfo->tested = $response['tested'] ?? get_bloginfo('version');
        $updateInfo->requires_php = $response['requires_php'] ?? '8.0';
        $updateInfo->icons = $response['icons'] ?? [];
        $updateInfo->banners = $response['banners'] ?? [];
        $updateInfo->id = $response['id'] ?? $this->pluginSlug;

        // Cache for 12 hours
        set_transient($this->cacheKey, $updateInfo, 12 * HOUR_IN_SECONDS);

        return $updateInfo;
    }

    /**
     * Fetch data from the update check API
     *
     * @param string $action API action (empty for update check, 'plugin_information' for details)
     * @return array|null Response data or null on failure
     */
    private function fetchFromApi(string $action = ''): ?array
    {
        $url = add_query_arg([
            'action' => $action ?: 'update_check',
            'slug' => $this->pluginSlug,
            'version' => $this->currentVersion,
            'site_url' => home_url(),
            'php_version' => phpversion(),
            'wp_version' => get_bloginfo('version'),
        ], $this->updateCheckUrl);

        $response = wp_remote_get($url, [
            'timeout' => 15,
            'sslverify' => true,
            'headers' => [
                'Accept' => 'application/json',
            ],
        ]);

        if (is_wp_error($response)) {
            $this->logger->warning('Update check failed', [
                'error' => $response->get_error_message()
            ]);
            return null;
        }

        $responseCode = wp_remote_retrieve_response_code($response);
        if ($responseCode !== 200) {
            $this->logger->warning('Update check returned non-200', [
                'code' => $responseCode
            ]);
            return null;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!is_array($data)) {
            $this->logger->warning('Update check returned invalid JSON');
            return null;
        }

        return $data;
    }

    /**
     * Clear the update cache
     * Call this when you want to force a fresh update check
     */
    public function clearCache(): void
    {
        delete_transient($this->cacheKey);
        $this->logger->debug('Cleared plugin update cache');
    }
}
