<?php

namespace Intucart\Services;

/**
 * Logger service class
 */
class Logger
{
    /**
     * Log levels
     *
     * @var array
     */
    private const LOG_LEVELS = array(
        'debug',
        'info',
        'warning',
        'error',
    );

    /**
     * Option key for log settings
     *
     * @var string
     */
    private const OPTION_KEY = 'intucart_logger_settings';

    /**
     * Log a debug message
     *
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     *
     * @return void
     */
    public function debug(string $message, array $context = array()): void
    {
        $this->log('debug', $message, $context);
    }

    /**
     * Log an info message
     *
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     *
     * @return void
     */
    public function info(string $message, array $context = array()): void
    {
        $this->log('info', $message, $context);
    }

    /**
     * Log a warning message
     *
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     *
     * @return void
     */
    public function warning(string $message, array $context = array()): void
    {
        $this->log('warning', $message, $context);
    }

    /**
     * Log an error message
     *
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     *
     * @return void
     */
    public function error(string $message, array $context = array()): void
    {
        $this->log('error', $message, $context);
    }

    /**
     * Log a message with the specified level
     *
     * @param string $level   Log level.
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     *
     * @return void
     */
    private function log(string $level, string $message, array $context = array()): void
    {
        if (!$this->shouldLog($level)) {
            return;
        }

        // Skip non-error logs unless WP_DEBUG is enabled
        if ($level !== 'error' && (!defined('WP_DEBUG') || !WP_DEBUG)) {
            return;
        }

        $log_entry = array(
            'timestamp' => current_time('mysql'),
            'level'     => strtoupper($level),
            'message'   => $message,
            'context'   => $context,
        );

        $formatted_message = sprintf(
            "[%s] %s %s",
            $log_entry['level'],
            $log_entry['message'],
            ! empty($context) ? wp_json_encode($context) : ''
        );

        // Log to WordPress error log
        error_log(sprintf('[%s]%s', Constants::PLUGIN_NAME, $formatted_message));

        // Log to database if the table exists
        $this->logToDatabase($level, $message, $context);

        do_action('intucart_logged_message', $log_entry);
    }

    /**
     * Log a message to the database
     *
     * @param string $level   Log level.
     * @param string $message The message to log.
     * @param array  $context Additional context data.
     *
     * @return void
     */
    private function logToDatabase(string $level, string $message, array $context = array()): void
    {
        // Only log errors and warnings to the database
        if ($level !== 'error' && $level !== 'warning') {
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'intucart_logs';

        // Check if the logs table exists
        $table_exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table_name)) === $table_name;

        if (!$table_exists) {
            return;
        }

        $wpdb->insert(
            $table_name,
            array(
                'level'     => $level,
                'message'   => $message,
                'context'   => !empty($context) ? wp_json_encode($context) : null,
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%s', '%s')
        );
    }

    /**
     * Check if we should log messages of the specified level
     *
     * @param string $level Log level to check.
     *
     * @return bool
     */
    private function shouldLog(string $level): bool
    {
        $settings      = get_option(self::OPTION_KEY, array('minLevel' => 'debug'));
        $levelIndex    = array_search($level, self::LOG_LEVELS, true);
        $minLevelIndex = array_search($settings['minLevel'], self::LOG_LEVELS, true);

        return $levelIndex >= $minLevelIndex;
    }
}
