<?php

namespace Intucart\Services\Managers;

use Intucart\Services\Constants;
use Intucart\Services\Managers\PostTypeManager;
use Intucart\Services\Managers\TaxonomyManager;
use Intucart\Services\Logger;
use Intucart\Services\Licensing\License;

/**
 * Manages CRON jobs for vector database synchronization
 *
 * @category Services
 * @package  Intucart
 * @author   Intucart <support@intufind.com>
 * @license  GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://intufind.com
 */
class CronManager
{
    private PostTypeManager $postTypeManager;
    private TaxonomyManager $taxonomyManager;
    private Logger $logger;
    private License $license;

    /**
     * Constructor
     *
     * @param PostTypeManager  $postTypeManager  Post type manager instance
     * @param TaxonomyManager  $taxonomyManager  Taxonomy manager instance
     * @param Logger           $logger           Logger instance
     * @param License          $license          License service instance
     */
    public function __construct(
        PostTypeManager $postTypeManager,
        TaxonomyManager $taxonomyManager,
        Logger $logger,
        License $license
    ) {
        $this->postTypeManager = $postTypeManager;
        $this->taxonomyManager = $taxonomyManager;
        $this->logger = $logger;
        $this->license = $license;
    }

    /**
     * Get the timestamp for the next sync time
     *
     * @return int Timestamp for next sync
     */
    private function getNextSyncTime(): int
    {
        $hour = (int)get_option(Constants::SYNC_HOUR_OPTION, Constants::DEFAULT_SYNC_HOUR);
        $now = time();
        $date = new \DateTime();
        $date->setTimestamp($now);
        $date->setTime($hour, 0, 0);

        // If we've already passed this hour today, schedule for tomorrow
        if ($date->getTimestamp() <= $now) {
            $date->modify('+1 day');
        }

        return $date->getTimestamp();
    }

    /**
     * Initialize CRON schedules and hooks
     *
     * @return void
     */
    public function initialize(): void
    {
        // Register custom interval first
        add_filter('cron_schedules', [$this, 'addCronIntervals']);

        // Register hooks for scheduled syncs
        add_action(Constants::POST_SYNC_HOOK, [$this, 'syncPostTypes']);
        add_action(Constants::TAXONOMY_SYNC_HOOK, [$this, 'syncTaxonomies']);

        // Add handler for daily system maintenance
        add_action(Constants::DAILY_CLEANUP_HOOK, [$this, 'handleSystemMaintenance']);

        // Add handler for full sync trigger (used by license activation, etc.)
        add_action(Constants::TRIGGER_FULL_SYNC_HOOK, [$this, 'triggerFullSync'], 10, 2);

        // Schedule events if not already scheduled
        $this->scheduleEvents();
    }

    /**
     * Schedule CRON events for all content types and maintenance
     *
     * @return void
     */
    private function scheduleEvents(): void
    {
        // Ensure interval is registered before scheduling
        if (!\wp_get_schedules()[Constants::SYNC_INTERVAL]) {
            return;
        }

        // Schedule full post sync (handles all post types efficiently)
        if (!\wp_next_scheduled(Constants::POST_SYNC_HOOK)) {
            \wp_schedule_event(
                $this->getNextSyncTime(),
                Constants::SYNC_INTERVAL,
                Constants::POST_SYNC_HOOK
            );
        }

        // Schedule taxonomy sync
        if (!\wp_next_scheduled(Constants::TAXONOMY_SYNC_HOOK)) {
            \wp_schedule_event(
                $this->getNextSyncTime(),
                Constants::SYNC_INTERVAL,
                Constants::TAXONOMY_SYNC_HOOK
            );
        }

        // Schedule daily maintenance
        if (!\wp_next_scheduled(Constants::DAILY_CLEANUP_HOOK)) {
            \wp_schedule_event(
                time(),
                'daily',
                Constants::DAILY_CLEANUP_HOOK
            );
        }
    }

    /**
     * Add custom CRON intervals
     *
     * @param array $schedules Existing schedules
     *
     * @return array
     */
    public function addCronIntervals(array $schedules): array
    {
        $interval = get_option(
            Constants::SYNC_INTERVAL_OPTION,
            Constants::DEFAULT_SYNC_INTERVAL
        );
        $schedules[Constants::SYNC_INTERVAL] = [
            'interval' => $interval,
            'display' => sprintf(
                'Every %d hours (Intucart Sync)',
                $interval / 3600
            )
        ];
        return $schedules;
    }

    /**
     * Sync post types to vector database
     * Uses PostTypeManager's efficient batch processing for all post types
     *
     * @return void
     */
    public function syncPostTypes(): void
    {
        try {
            $this->logger->info('Starting full sync of all post types via PostTypeManager');

            $this->postTypeManager->handleScheduledPostSync();

            $this->logger->info('Full post types sync completed via PostTypeManager');

            // Update sync timestamp for compatibility
            update_option('intucart_last_full_post_sync', time());

            // Fire action for other components that need to react to successful sync
            do_action(Constants::POST_SYNC_COMPLETED_ACTION);
        } catch (\Exception $e) {
            $this->logger->error('Full post types sync failed', ['error' => $e->getMessage()]);
            throw $e;
        }
    }

    /**
     * Sync taxonomies to vector database
     * Delegates to TaxonomyManager's scheduled sync system
     *
     * @return void
     */
    public function syncTaxonomies(): void
    {
        try {
            $this->logger->info('Starting taxonomy sync via TaxonomyManager');

            // Delegate to TaxonomyManager's scheduled sync system
            $this->taxonomyManager->handleScheduledTaxonomySync();

            $this->logger->info('Taxonomy sync completed via TaxonomyManager');

            // Update sync timestamp for compatibility
            update_option('intucart_last_taxonomy_sync', time());

            // Fire action for other components that need to react to successful sync
            do_action(Constants::TAXONOMY_SYNC_COMPLETED_ACTION);
        } catch (\Exception $e) {
            $this->logger->error('Taxonomy sync failed', ['error' => $e->getMessage()]);
            throw $e;
        }
    }

    /**
     * Trigger immediate sync of posts and taxonomies
     * Used for license activation and manual full sync requests
     * Note: User sync is disabled for license activation to reduce server load
     *
     * @param string $group Unused parameter for compatibility (WordPress cron doesn't use groups)
     * @param int $initialDelay Initial delay in seconds before first sync starts
     * @return void
     */
    public function triggerFullSync(string $group = 'intucart-full-sync', int $initialDelay = 5): void
    {
        $this->logger->info('Triggering sync of posts and taxonomies', [
            'initial_delay' => $initialDelay
        ]);

        $currentDelay = $initialDelay;

        // Schedule full post sync (handles all post types in one call)
        $this->scheduleAction(Constants::POST_SYNC_HOOK, $currentDelay, $group);
        $this->logger->info('Scheduled full post sync (all post types)', [
            'delay' => $currentDelay
        ]);
        $currentDelay += 60; // Give more time for all post types

        // Schedule taxonomy sync
        $this->scheduleAction(Constants::TAXONOMY_SYNC_HOOK, $currentDelay, $group);
        $this->logger->info('Scheduled taxonomy sync', [
            'delay' => $currentDelay
        ]);
    }

    /**
     * Schedule a single action using WordPress cron
     *
     * @param string $hook Hook name to schedule
     * @param int $delay Delay in seconds
     * @param string $group Unused parameter for compatibility (WordPress cron doesn't use groups)
     * @return void
     */
    private function scheduleAction(string $hook, int $delay, string $group = ''): void
    {
        $scheduledTime = time() + $delay;
        wp_schedule_single_event($scheduledTime, $hook);
    }

    /**
     * Handle daily system maintenance cron job
     */
    public function handleSystemMaintenance(): void
    {
        $this->runSystemMaintenanceTasks();
    }

    /**
     * Run all system maintenance tasks
     *
     * @return void
     */
    private function runSystemMaintenanceTasks(): void
    {
        try {
            // Clean up analytics for deleted products
            $this->cleanupDeletedProductAnalytics();
            // Add more maintenance/cleanup tasks here as needed
        } catch (\Exception $e) {
            $this->logger->error('System maintenance tasks failed', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Clean up analytics rows for deleted products
     *
     * @return void
     */
    private function cleanupDeletedProductAnalytics(): void
    {
        global $wpdb;

        try {
            $analytics_table = $wpdb->prefix . 'intucart_recommendation_analytics';
            $products_table = $wpdb->posts;

            // Use prepared statement for security
            $result = $wpdb->query($wpdb->prepare("
                DELETE FROM {$analytics_table}
                WHERE product_id NOT IN (
                    SELECT ID FROM {$products_table} WHERE post_type = %s AND post_status = %s
                )
            ", 'product', 'publish'));

            if ($result !== false) {
                $this->logger->info('Cleaned up deleted product analytics', [
                    'rows_deleted' => $result
                ]);
            } else {
                $this->logger->warning('Analytics cleanup query returned false', [
                    'last_error' => $wpdb->last_error
                ]);
            }
        } catch (\Exception $e) {
            $this->logger->error('Failed to cleanup deleted product analytics', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Deactivate all scheduled actions
     *
     * @return void
     */
    public static function deactivate(): void
    {
        // Clear WordPress scheduled events
        $hooks = [
            Constants::POST_SYNC_HOOK,
            Constants::TAXONOMY_SYNC_HOOK,
            Constants::DAILY_CLEANUP_HOOK,
        ];

        foreach ($hooks as $hook) {
            // Clear all scheduled instances of this hook
            $timestamp = wp_next_scheduled($hook);
            while ($timestamp) {
                wp_unschedule_event($timestamp, $hook);
                $timestamp = wp_next_scheduled($hook);
            }
        }

        // Clean up any running sync flags (Transients)
        delete_transient('intucart_sync_in_progress');

        error_log('Intucart: Unscheduled all events using WordPress cron');
    }
}
