<?php

namespace Intucart\Services\Managers;

use Intucart\Services\Events\EventInterface;
use Intucart\Services\Logger;
use Intucart\Services\Events\Data\CartItemDTO;
use Intucart\Services\Cache\CacheService;

/**
 * Manages event collection and processing for product relationships
 *
 * Tracks purchase events to build product relationship data for recommendations.
 * Does NOT store user-level interaction data - only aggregated product relationships.
 *
 * @category Services
 * @package  Intucart
 * @author   Intucart <support@intufind.com>
 * @license  GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://intufind.com
 */
class EventManager
{
    private Logger $logger;
    private CacheService $cache;

    /**
     * Cache group for product associations
     */
    private const CACHE_GROUP_ASSOCIATIONS = 'intucart_product_associations';

    /**
     * Cache duration for product associations (15 minutes)
     */
    private const PRODUCT_ASSOCIATIONS_CACHE_TTL = 900;

    /**
     * Constructor
     *
     * @param Logger       $logger      Logger service instance
     * @param CacheService $cache       Cache service instance
     */
    public function __construct(
        Logger $logger,
        CacheService $cache
    ) {
        $this->logger = $logger;
        $this->cache = $cache;
    }

    /**
     * Track an event for product relationship building
     *
     * Only tracks purchase events to build product-to-product relationships.
     * Does NOT store user-level interaction data.
     *
     * @param EventInterface $event Event to track
     *
     * @return void
     */
    public function track(EventInterface $event): void
    {
        // Only track purchase events for product relationships
        if ($event->getName() !== 'purchase') {
            return;
        }

        $metadata = $event->getParameters();

        // Get order items from the event parameters
        $orderItems = [];

        // Convert array items to CartItemDTO objects if needed
        $rawItems = $metadata['order']['items'] ?? [];
        foreach ($rawItems as $rawItem) {
            if (is_array($rawItem)) {
                // Convert array to CartItemDTO
                $orderItems[] = CartItemDTO::fromArray($rawItem);
            } elseif ($rawItem instanceof CartItemDTO) {
                // Already a CartItemDTO
                $orderItems[] = $rawItem;
            }
        }

        // Track product relationships (which products are purchased together)
        $this->trackProductRelationships($orderItems);
    }

    /**
     * Track relationships between products purchased together and invalidate caches
     *
     * @param array $items Array of purchased items
     *
     * @return void
     */
    private function trackProductRelationships(array $items): void
    {
        global $wpdb;

        if (count($items) < 2) {
            return;
        }

        $updatedProducts = [];

        foreach ($items as $i => $item1) {
            // Extract product ID from the item
            $p1_id = null;

            if ($item1 instanceof CartItemDTO) {
                $p1_id = $item1->getProduct()->getId();
            } else {
                continue;
            }

            if (!$p1_id) {
                continue;
            }

            $updatedProducts[] = $p1_id;

            for ($j = $i + 1; $j < count($items); $j++) {
                $item2 = $items[$j];

                // Extract product ID from the second item
                $p2_id = null;

                if ($item2 instanceof CartItemDTO) {
                    $p2_id = $item2->getProduct()->getId();
                } else {
                    continue;
                }

                // Skip if same product or invalid IDs
                if ($p1_id === $p2_id || !$p1_id || !$p2_id) {
                    continue;
                }

                $updatedProducts[] = $p2_id;

                // Ensure consistent ordering of product pairs
                $first_product = min($p1_id, $p2_id);
                $second_product = max($p1_id, $p2_id);

                // Update or insert the relationship
                $wpdb->query(
                    $wpdb->prepare(
                        "INSERT INTO {$wpdb->prefix}intucart_product_relationships 
                        (product1_id, product2_id, purchase_count, last_purchased) 
                        VALUES (%d, %d, 1, NOW())
                        ON DUPLICATE KEY UPDATE 
                        purchase_count = purchase_count + 1,
                        last_purchased = NOW()",
                        $first_product,
                        $second_product
                    )
                );
            }
        }

        // Invalidate caches for all affected products
        foreach (array_unique($updatedProducts) as $productId) {
            $this->cache->delete("{$productId}_30", self::CACHE_GROUP_ASSOCIATIONS);
        }
    }

    /**
     * Get product associations based on purchase history
     *
     * @param int   $productId Product ID
     * @param float $halfLifeDays Number of days for the time decay half-life (default: 30)
     *
     * @return array Array of associated products with frequency and confidence
     */
    public function getProductAssociations(int $productId, float $halfLifeDays = 30.0): array
    {
        // Check cache first
        $cacheKey = "{$productId}_{$halfLifeDays}";
        $cached = $this->cache->get($cacheKey, self::CACHE_GROUP_ASSOCIATIONS);
        if ($cached !== false) {
            return $cached;
        }

        global $wpdb;

        // Get total purchase count for this product to calculate confidence
        $totalPurchases = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT SUM(purchase_count) 
                FROM {$wpdb->prefix}intucart_product_relationships
                WHERE product1_id = %d OR product2_id = %d",
                $productId,
                $productId
            )
        );

        // Get products from the relationships table with time decay factor
        $sql = $wpdb->prepare(
            "SELECT 
                CASE 
                    WHEN product1_id = %d THEN product2_id 
                    ELSE product1_id 
                END as product_id,
                purchase_count as frequency,
                DATEDIFF(NOW(), last_purchased) as days_since_last_purchase
            FROM {$wpdb->prefix}intucart_product_relationships
            WHERE product1_id = %d OR product2_id = %d
            ORDER BY purchase_count DESC
            LIMIT 20",
            $productId,
            $productId,
            $productId
        );

        $results = $wpdb->get_results($sql, ARRAY_A);

        // Apply time decay factor and calculate confidence for each result
        foreach ($results as &$result) {
            $daysSinceLastPurchase = (int)$result['days_since_last_purchase'];

            // Calculate decay constant based on half-life
            // Formula: decay_constant = ln(2) / half_life
            $decayConstant = 0.693 / $halfLifeDays;

            // Calculate time decay factor (exponential decay)
            $timeDecayFactor = exp(-$decayConstant * $daysSinceLastPurchase);

            // Add time decay factor to the result
            $result['time_decay'] = $timeDecayFactor;

            // Calculate confidence as frequency / total purchases for this product
            $result['confidence'] = $totalPurchases > 0 
                ? (float)$result['frequency'] / (float)$totalPurchases 
                : 0.0;
        }

        // Cache the results
        $this->cache->set(
            $cacheKey,
            $results,
            self::CACHE_GROUP_ASSOCIATIONS,
            self::PRODUCT_ASSOCIATIONS_CACHE_TTL
        );

        return $results;
    }
}
