<?php

namespace Intucart\Services\Managers;

use Intucart\Services\Logger;
use Intucart\Services\AIClientManager;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\Providers\PostTypes\PostTypeProviderInterface;
use Intucart\Services\Managers\EventManager;
use Intucart\Services\Constants;
use Intucart\Services\Licensing\License;
use Intucart\Services\PageBuilderService;

/**
 * PostTypeManager service
 *
 * Unified manager for all WordPress post types including products
 * Handles syncing, batch processing, and embedding operations
 * Self-contained batch processing with intelligent queuing and cloud sync
 */
class PostTypeManager
{
    /**
     * Product types to exclude from sync
     *
     * @var array<string>
     */
    private const EXCLUDED_PRODUCT_TYPES = [
        'variation',  // Product variations are handled via parent products
    ];

    /**
     * System post types to exclude from sync
     * These post types might support editor/title but contain sensitive info or are system-only
     *
     * @var array<string>
     */
    private const EXCLUDED_SYSTEM_POST_TYPES = [
        // WooCommerce sensitive data (these might support editor/title but contain sensitive info)
        'shop_order',          // WooCommerce orders (sensitive customer data)
        'shop_coupon',         // WooCommerce coupons (sensitive pricing data)
        'shop_order_refund',   // WooCommerce refunds (sensitive financial data)

        // Action Scheduler and similar background job systems
        'scheduled-action',    // Action Scheduler actions
        
        // Jetpack and similar plugin system types
        'jp_pay_order',        // Jetpack payment orders
        'jp_pay_product',      // Jetpack payment products

        // Other system types that might slip through content capability checks
        'acf-field',           // Advanced Custom Fields field definitions
        'acf-field-group',     // Advanced Custom Fields field group definitions
    ];

    /**
     * Post statuses allowed for sync
     *
     * Published content is publicly accessible and searchable.
     * Private content is chatbot-accessible only (non-searchable).
     *
     * @var array<string>
     */
    private const SYNCABLE_POST_STATUSES = [
        'publish',
        'private',  // Chatbot-only, automatically set as non-searchable
    ];

    /**
     * Pages to exclude from sync based on functionality
     * These are purely transactional/functional pages that don't contain useful content for discovery
     * Focus on pages that are primarily for user interaction rather than information consumption
     *
     * @var array<string>
     */
    private const EXCLUDED_PAGE_TYPES = [
        // WooCommerce functional pages (current WC version)
        'cart',
        'checkout',
        'myaccount',
        'edit-address',
        'view-order',

        // Common WordPress functional pages
        'sitemap',
    ];

    /**
     * Page slugs to exclude from sync (fallback for pages not identified by WooCommerce functions)
     * These match common transactional/functional page slugs that should be excluded
     *
     * @var array<string>
     */
    private const EXCLUDED_SLUGS = [
        'cart',
        'checkout',
        'my-account',
        'account',
        'order-received',
        'sitemap',
        'site-map',
        '404',
        'search',
        'wp-login',
        'wp-admin',
        'sample-page',        // Default WordPress sample page
        'hello-world',        // Default WordPress sample post (if created as page)
    ];

    /**
     * Page titles to exclude from sync (case-insensitive matching)
     * Used as a fallback when slug-based detection doesn't work
     * Focus on transactional/functional page titles
     *
     * @var array<string>
     */
    private const EXCLUDED_TITLES = [
        'cart',
        'checkout',
        'my account',
        'account',
        'order received',
        'sitemap',
        'site map',
        'sample page',       // Default WordPress sample page
        'hello world',       // Default WordPress sample post
    ];

    /**
     * Regex pattern to match sitemap-related posts that should be excluded from sync
     * Matches various sitemap naming conventions (case-insensitive)
     *
     * @var string
     */
    private const SITEMAP_EXCLUSION_PATTERN = '/^(.*[-_\s]?sitemap[-_\s]?.*|.*site[-_\s]?map.*|xml[-_\s]?sitemap.*)$/i';

    /**
     * Batch processing configuration constants
     */
    private const BATCH_DELAY_UPDATE = 30;    // Seconds to delay update batches
    private const BATCH_DELAY_DELETE = 10;    // Seconds to delay delete batches (faster for better UX)
    private const BATCH_QUEUE_EXPIRY = 300;   // Seconds for batch queue transient expiry (5 minutes)

    /**
     * Cache configuration constants
     */
    private const CACHE_TTL_POST_DATA = 1800; // 30 minutes cache for post data

    /**
     * Sync configuration constants
     */
    private const SYNC_SCHEDULE_RECURRENCE = 'daily';                   // How often to run full sync
    private const BATCH_PROCESS_HOOK = 'intucart_process_post_batch';    // Hook name for batch processing
    private const BATCH_QUEUE_TRANSIENT = 'intucart_post_batch_queue';   // Transient key for batch queue

    /**
     * Validation constants - generous limits to catch obvious issues
     * Real content limits are handled by the cloud service based on embedding model capabilities
     * Increased limits to work better with content chunking in the cloud
     */
    private const MAX_POST_TITLE_LENGTH = 2000;
    private const MAX_POST_CONTENT_LENGTH = 200000; // Increased from 100KB to 200KB for chunking
    private const MIN_POST_TITLE_LENGTH = 1;

    /**
     * Meta field names for sync status tracking
     */
    private const META_SYNC_STATUS = '_intucart_sync_status';           // 'success', 'error', 'pending', or empty
    private const META_SYNC_ERROR = '_intucart_sync_error';             // Error message from last failed sync
    private const META_SYNC_TIMESTAMP = '_intucart_sync_timestamp';     // Timestamp of last sync attempt
    private const META_SYNC_METHOD = '_intucart_sync_method';           // 'manual', 'auto', 'bulk', 'realtime'
    private const META_SYNC_OPERATION = '_intucart_sync_operation';     // 'create', 'update', 'delete'
    private const META_SYNC_CONTENT_HASH = '_intucart_sync_content_hash'; // Hash of content at last sync

    /**
     * Cloud API batch size limits
     * These should match the limits in productHandler.ts and other cloud handlers
     */
    private const MAX_BULK_UPSERT_SIZE = 100;  // Max items per bulk upsert request
    private const MAX_BULK_DELETE_SIZE = 100;  // Max items per bulk delete request

    private Logger $logger;
    private AIClientManager $aiClientManager;
    private CacheService $cache;
    private PostTypeProviderInterface $productProvider;
    private PostTypeProviderInterface $genericProvider;
    private License $license;
    private PageBuilderService $pageBuilderService;

    /**
     * Constructor
     *
     * @param Logger                      $logger              Logger service
     * @param AIClientManager             $aiClientManager     AI Client Manager service
     * @param CacheService                $cache               Cache service
     * @param PostTypeProviderInterface   $productProvider     Product provider
     * @param PostTypeProviderInterface   $genericProvider     Generic provider
     * @param License                     $license             License service
     * @param PageBuilderService          $pageBuilderService  Page builder service
     */
    public function __construct(
        Logger $logger,
        AIClientManager $aiClientManager,
        CacheService $cache,
        PostTypeProviderInterface $productProvider,
        PostTypeProviderInterface $genericProvider,
        License $license,
        PageBuilderService $pageBuilderService
    ) {
        $this->logger = $logger;
        $this->aiClientManager = $aiClientManager;
        $this->cache = $cache;
        $this->productProvider = $productProvider;
        $this->genericProvider = $genericProvider;
        $this->license = $license;
        $this->pageBuilderService = $pageBuilderService;
    }

    /**
     * Initialize the post type sync service
     *
     * @return void
     */
    public function initialize(): void
    {
        // Product-specific hooks for real-time syncing
        if (class_exists('WooCommerce')) {
            add_action('woocommerce_new_product', [$this, 'handleProductChange'], 10, 1);
            add_action('woocommerce_update_product', [$this, 'handleProductChange'], 10, 1);
            add_action('woocommerce_delete_product', [$this, 'handleProductDelete'], 10, 1);
        }

        // WordPress post hooks for all post types
        add_action('save_post', [$this, 'handlePostSave'], 10, 3);
        add_action('before_delete_post', [$this, 'handlePostDelete'], 10, 1);
        add_action('transition_post_status', [$this, 'handlePostStatusChange'], 10, 3);

        // Batch processing hooks
        add_action(self::BATCH_PROCESS_HOOK, [$this, 'handlePostBatch']);
    }

    /**
     * Get the provider for a given post type
     *
     * @param string $postType
     * @return PostTypeProviderInterface
     */
    public function getProviderForType(string $postType): PostTypeProviderInterface
    {
        if ($postType === 'product') {
            return $this->productProvider;
        }
        // For all other post types, use the generic provider
        return $this->genericProvider;
    }

    /**
     * Get modified posts of any post type since timestamp
     *
     * @param string $postType Post type
     * @param int $lastSync Last sync timestamp
     *
     * @return array Modified posts
     */
    public function getModifiedPosts(string $postType, int $lastSync): array
    {
        $provider = $this->getProviderForType($postType);
        return $provider->getPosts(['modified_after' => $lastSync], $postType);
    }

    /**
     * New helper to get flat payload for cloud sync
     */
    public function getCloudPayload(int $postId, string $postType): array
    {
        $provider = $this->getProviderForType($postType);
        return $provider->getCloudPayload($postId, $postType);
    }

    /**
     * Handle product creation or update (WooCommerce specific)
     *
     * @param int $product_id Product ID
     * @return void
     */
    public function handleProductChange(int $product_id): void
    {
        // Validate inputs
        if ($product_id <= 0) {
            $this->logger->warning('Invalid product change: invalid product ID', [
                'product_id' => $product_id
            ]);
            return;
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            $this->logger->warning('Product not found for change event', [
                'product_id' => $product_id
            ]);
            return;
        }

        // Skip variations - they're handled via parent products
        if ($this->isProductVariation($product)) {
            $this->logger->info('Skipping variation product update - will be handled by parent', [
                'product_id' => $product_id
            ]);
            return;
        }

        // Only sync published products
        if (!$this->shouldSyncPost($product_id, 'product')) {
            $this->logger->info('Product not eligible for sync', [
                'product_id' => $product_id,
                'status' => $product->get_status(),
                'type' => $product->get_type()
            ]);
            return;
        }

        $this->logger->info('Product changed, queueing for batch sync', [
            'product_id' => $product_id,
            'product_name' => $product->get_name(),
            'action' => 'update'
        ]);

        $this->queuePostBatchAction($product_id, 'product', 'update');
        $this->invalidatePostCache($product_id, 'product');
    }

    /**
     * Handle product deletion (WooCommerce specific)
     *
     * @param int $product_id Product ID
     * @return void
     */
    public function handleProductDelete(int $product_id): void
    {
        // Validate inputs
        if ($product_id <= 0) {
            $this->logger->warning('Invalid product delete: invalid product ID', [
                'product_id' => $product_id
            ]);
            return;
        }

        $this->logger->info('Product deleted, queueing for batch sync', [
            'product_id' => $product_id,
            'action' => 'delete'
        ]);

        $this->queuePostBatchAction($product_id, 'product', 'delete');
        $this->invalidatePostCache($product_id, 'product');
    }

    /**
     * Handle WordPress post save (universal handler)
     *
     * @param int     $post_id Post ID
     * @param \WP_Post $post    Post object
     * @param bool    $update  Whether this is an update
     * @return void
     */
    public function handlePostSave(int $post_id, \WP_Post $post, bool $update): void
    {
        if (wp_is_post_revision($post_id) || in_array($post->post_status, ['auto-draft', 'inherit'], true)) {
            return;
        }

        $postType = $post->post_type;

        // For products, prefer WooCommerce hooks if available to avoid duplicates
        if ($postType === 'product' && class_exists('WooCommerce')) {
            // WooCommerce hooks will handle this
            return;
        }

        // Only sync eligible posts
        if (!$this->shouldSyncPost($post_id, $postType)) {
            return;
        }

        $this->logger->info('Post changed, queueing for batch sync', [
            'post_id' => $post_id,
            'post_type' => $postType,
            'action' => 'update'
        ]);

        $this->queuePostBatchAction($post_id, $postType, 'update');
        $this->invalidatePostCache($post_id, $postType);
    }

    /**
     * Handle WordPress post deletion (universal handler)
     *
     * @param int $post_id Post ID
     * @return void
     */
    public function handlePostDelete(int $post_id): void
    {
        $postType = get_post_type($post_id);
        if (!$postType) {
            return;
        }

        // For products, prefer WooCommerce hooks if available to avoid duplicates
        if ($postType === 'product' && class_exists('WooCommerce')) {
            // WooCommerce hooks will handle this
            return;
        }

        $this->logger->info('Post deleted, queueing for batch sync', [
            'post_id' => $post_id,
            'post_type' => $postType,
            'action' => 'delete'
        ]);

        $this->queuePostBatchAction($post_id, $postType, 'delete');
        $this->invalidatePostCache($post_id, $postType);
    }

    /**
     * Handle WordPress post status change (universal handler)
     *
     * @param string   $new_status New post status
     * @param string   $old_status Old post status
     * @param \WP_Post $post       Post object
     * @return void
     */
    public function handlePostStatusChange(string $new_status, string $old_status, \WP_Post $post): void
    {
        $postType = $post->post_type;

        if ($new_status === $old_status) {
            return;
        }

        // For products, prefer WooCommerce hooks to avoid duplicates
        if ($postType === 'product' && class_exists('WooCommerce')) {
            return;
        }

        // Check if post was/is eligible for sync
        $wasEligible = $this->isPostStatusSyncable($old_status);
        $isEligible = $this->isPostStatusSyncable($new_status);

        if ($isEligible && !$wasEligible) {
            // Post became eligible for sync (newly published, etc.)
            $this->logger->info('Post became eligible for sync, queueing upsert for batch', [
                'post_id' => $post->ID,
                'post_type' => $postType,
                'old_status' => $old_status,
                'new_status' => $new_status
            ]);
            $this->queuePostBatchAction($post->ID, $postType, 'update');
        } elseif ($wasEligible && !$isEligible) {
            // Post no longer eligible for sync (unpublished, trashed, etc.)
            $this->logger->info('Post no longer eligible for sync, queueing delete for batch', [
                'post_id' => $post->ID,
                'post_type' => $postType,
                'old_status' => $old_status,
                'new_status' => $new_status
            ]);
            $this->queuePostBatchAction($post->ID, $postType, 'delete');
        } elseif ($wasEligible && $isEligible) {
            // Post changed but still eligible - update content
            $this->logger->info('Post status changed but still eligible, queueing update for batch', [
                'post_id' => $post->ID,
                'post_type' => $postType,
                'old_status' => $old_status,
                'new_status' => $new_status
            ]);
            $this->queuePostBatchAction($post->ID, $postType, 'update');
        }
        // If neither was nor is eligible, do nothing
    }

    /**
     * Check if a post status makes it eligible for sync
     *
     * @param string $status Post status
     * @return bool
     */
    private function isPostStatusSyncable(string $status): bool
    {
        return in_array($status, self::SYNCABLE_POST_STATUSES, true);
    }


    /**
     * Queue a post action for batch processing
     *
     * @param int    $post_id   Post ID
     * @param string $post_type Post type
     * @param string $action    'update', 'delete', or 'full_sync'
     * @return void
     */
    private function queuePostBatchAction(int $post_id, string $post_type, string $action): void
    {
        $queue = get_transient(self::BATCH_QUEUE_TRANSIENT) ?: [];

        $key = $post_id . '|' . $post_type . '|' . $action;

        $queue[$key] = [
            'post_id' => $post_id,
            'post_type' => $post_type,
            'action' => $action,
            'timestamp' => time()
        ];

        set_transient(self::BATCH_QUEUE_TRANSIENT, $queue, self::BATCH_QUEUE_EXPIRY);

        // For deletes, use a shorter delay for better UX
        $delay = ($action === 'delete') ? self::BATCH_DELAY_DELETE : self::BATCH_DELAY_UPDATE;

        // Schedule batch job if not already scheduled
        $this->schedulePostBatch($delay);
    }

    /**
     * Schedule post batch processing using WordPress cron
     *
     * @param int $delay Delay in seconds
     * @return void
     */
    private function schedulePostBatch(int $delay): void
    {
        // Check if already scheduled
        if (!wp_next_scheduled(self::BATCH_PROCESS_HOOK)) {
            wp_schedule_single_event(time() + $delay, self::BATCH_PROCESS_HOOK);
            $this->logger->debug('Post batch scheduled using WordPress cron', [
                'delay' => $delay,
                'hook' => self::BATCH_PROCESS_HOOK,
                'scheduled_time' => time() + $delay
            ]);
        }
    }

    /**
     * Process post batch queue
     *
     * @return void
     */
    public function handlePostBatch(): void
    {
        $queue = get_transient(self::BATCH_QUEUE_TRANSIENT) ?: [];
        if (empty($queue)) {
            return;
        }

        // Create a backup before deleting the queue in case processing fails
        $backupQueue = $queue;
        delete_transient(self::BATCH_QUEUE_TRANSIENT);

        $this->logger->info('Processing post batch queue', [
            'queue_size' => count($queue)
        ]);

        try {
            // Group by post type and action for efficient processing
            $groupedItems = [];

            foreach ($queue as $item) {
                $postType = $item['post_type'];
                $action = $item['action'];

                if (!isset($groupedItems[$postType])) {
                    $groupedItems[$postType] = ['sync' => [], 'delete' => []];
                }

                if ($action === 'update' || $action === 'full_sync') {
                    $groupedItems[$postType]['sync'][] = $item;
                } elseif ($action === 'delete') {
                    $groupedItems[$postType]['delete'][] = $item;
                }
            }

            // Process each post type separately
            foreach ($groupedItems as $postType => $actions) {
                // Process sync items
                if (!empty($actions['sync'])) {
                    $this->processBulkUpsertItems($postType, $actions['sync']);
                }

                // Process delete items
                if (!empty($actions['delete'])) {
                    $this->processBulkDeleteItems($postType, $actions['delete']);
                }
            }
        } catch (\Exception $e) {
            // If processing failed, restore the backup queue for retry
            $this->logger->error('Batch processing failed, restoring queue for retry', [
                'error' => $e->getMessage(),
                'queue_size' => count($backupQueue)
            ]);
            set_transient(self::BATCH_QUEUE_TRANSIENT, $backupQueue, self::BATCH_QUEUE_EXPIRY);
            throw $e;
        }
    }

    /**
     * Handle scheduled post sync (runs daily)
     *
     * @return void
     */
    public function handleScheduledPostSync(): void
    {
        $this->logger->info('Starting scheduled full post sync');

        try {
            // Get all syncable post types
            $postTypes = $this->getSyncablePostTypes();

            foreach ($postTypes as $postType) {
                $this->logger->info('Processing scheduled sync for post type', [
                    'post_type' => $postType
                ]);

                $provider = $this->getProviderForType($postType);
                $posts = $provider->getPosts([], $postType);
                $syncItems = [];
                $eligiblePostIds = [];

                foreach ($posts as $post) {
                    $postId = is_object($post) ? $post->ID : $post['ID'];

                    // Skip if not eligible for sync
                    if (!$this->shouldSyncPost($postId, $postType)) {
                        continue;
                    }

                    // Track eligible post IDs to keep in the cloud index
                    $eligiblePostIds[] = $postId;

                    $syncItems[] = [
                        'post_id' => $postId,
                        'post_type' => $postType,
                        'action' => 'full_sync',
                        'timestamp' => time()
                    ];
                }

                // Process posts in bulk for this post type
                if (!empty($syncItems)) {
                    $this->processBulkUpsertItems($postType, $syncItems);

                    $this->logger->info('Processed scheduled sync for post type', [
                        'post_type' => $postType,
                        'posts_count' => count($syncItems)
                    ]);
                } else {
                    // Even if no items to sync, update the timestamp to show the scheduled sync ran
                    $this->updatePostTypeSyncTime($postType);

                    $this->logger->info('Scheduled sync completed for post type (no items to sync)', [
                        'post_type' => $postType
                    ]);
                }

                // Clean up cloud docs not present in the eligible set for this post type
                $this->performPostCleanup($postType, $eligiblePostIds);
            }

            $this->logger->info('Completed scheduled full post sync');
        } catch (\Exception $e) {
            $this->logger->error('Scheduled post sync failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Handle manual post sync via AJAX
     * Syncs all eligible posts for the specified post type and cleans up stale posts
     *
     * @return void
     */
    public function handleManualSync(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        $postType = sanitize_text_field($_POST['post_type'] ?? '');

        if (empty($postType)) {
            wp_send_json_error(__('Post type parameter required', 'intufind'));
            return;
        }


        try {
            $provider = $this->getProviderForType($postType);
            $posts = $provider->getPosts([], $postType);

            $syncItems = [];
            $eligiblePostIds = [];
            $skippedCount = 0;
            $skippedReasons = [];

            // Note: This processes posts individually which creates N+1 database queries.
            // For admin-only manual sync operations, this is acceptable because:
            // 1. Low frequency - only triggered manually by admins
            // 2. Simplicity - easier to debug and maintain
            // 3. Error isolation - individual post failures don't break the batch
            //
            // If optimization is needed, we could batch-load posts with status filtering:
            // $filteredPosts = $this->getEligiblePostsForSync($posts, $postType);
            foreach ($posts as $post) {
                $postId = is_object($post) ? $post->ID : $post['ID'];

                // Skip if not eligible for sync
                if (!$this->shouldSyncPost($postId, $postType)) {
                    $skippedCount++;

                    // Get the actual post to determine skip reason
                    $actualPost = get_post($postId);
                    $reason = $actualPost ? $actualPost->post_status : 'post_not_found';
                    $skippedReasons[$reason] = ($skippedReasons[$reason] ?? 0) + 1;

                    continue;
                }

                // Track eligible ID and get post title for logging (manual sync only)
                $eligiblePostIds[] = $postId;
                $actualPost = get_post($postId);

                $syncItems[] = [
                    'post_id' => $postId,
                    'post_type' => $postType,
                    'action' => 'full_sync',
                    'timestamp' => time()
                ];
            }

            if (empty($syncItems)) {
                // Even if no items to sync, still clean up cloud docs not present in eligible set
                $this->performPostCleanup($postType, $eligiblePostIds);

                // Fire action for other components that need to react to successful sync
                do_action(Constants::POST_SYNC_COMPLETED_ACTION);

                $postTypeLabel = $postType === 'post' ? 'posts' : $postType . ' posts';
                wp_send_json_success([
                    'message' => sprintf('No %s found to sync', $postTypeLabel),
                    'details' => [
                        'total_found' => count($posts),
                        'skipped_count' => $skippedCount,
                        'skip_reasons' => $skippedReasons
                    ]
                ]);
                return;
            }

            $this->processBulkUpsertItems($postType, $syncItems);

            // Clean up cloud docs not present in the eligible set for this post type
            $this->performPostCleanup($postType, $eligiblePostIds);

            // Fire action for other components that need to react to successful sync
            do_action('intucart_post_sync_completed');

            wp_send_json_success([
                'message' => sprintf('%s sync completed successfully for %d posts', ucfirst($postType), count($syncItems)),
                'details' => [
                    'synced_count' => count($syncItems),
                    'total_found' => count($posts),
                    'skipped_count' => $skippedCount,
                    'skip_reasons' => $skippedReasons
                ]
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Manual post sync failed', [
                'post_type' => $postType,
                'error' => $e->getMessage()
            ]);

            wp_send_json_error(__('Post sync failed. Check logs for details.', 'intufind'));
        }
    }

    /**
     * Process bulk upsert items by sending them to cloud in batches
     *
     * @param string $postType   Post type
     * @param array  $upsertItems Array of upsert items
     * @return void
     */
    private function processBulkUpsertItems(string $postType, array $upsertItems): void
    {
        try {
            // Check if this is a manual sync operation (full_sync actions)
            $isManualSync = !empty($upsertItems) && $upsertItems[0]['action'] === 'full_sync';

            // Filter out unchanged posts for efficiency (except for manual sync)
            $changedItems = [];
            $skippedCount = 0;
            
            foreach ($upsertItems as $item) {
                // Skip unchanged posts unless it's a manual full_sync
                if (!$isManualSync && !$this->hasContentChanged($item['post_id'])) {
                    $skippedCount++;
                    
                    // Update sync timestamp to show we checked it
                    $this->setSyncStatus($item['post_id'], 'success', null, [
                        'method' => $item['action'] === 'update' ? 'auto' : 'bulk',
                        'operation' => 'skipped_unchanged',
                        'content_hash' => $this->generateContentHash($item['post_id'])
                    ]);
                    continue;
                }
                
                $changedItems[] = $item;
            }
            
            // Log skipped posts for visibility
            if ($skippedCount > 0) {
                $this->logger->info('Skipped unchanged posts in bulk sync', [
                    'post_type' => $postType,
                    'skipped_count' => $skippedCount,
                    'processing_count' => count($changedItems),
                    'total_items' => count($upsertItems)
                ]);
            }

            // If no posts need syncing, just update the sync time and return
            if (empty($changedItems)) {
                $this->updatePostTypeSyncTime($postType);
                $this->logger->info('All posts unchanged, skipping cloud sync', [
                    'post_type' => $postType,
                    'total_checked' => count($upsertItems)
                ]);
                return;
            }

            if ($isManualSync) {
                // Log each post title being synced for manual sync
                foreach ($changedItems as $item) {
                    $post = get_post($item['post_id']);
                    $postTitle = $post ? $post->post_title : 'Unknown Title';
                }
            }

            if (count($changedItems) === 1) {
                // Single update - use existing individual method
                $item = $changedItems[0];
                $this->syncSinglePostToCloud($item['post_id'], $postType);
            } else {
                // Multiple updates - use bulk processing
                $this->syncBulkPostsToCloud($postType, $changedItems);
            }

            // Update last sync time for this specific post type
            $this->updatePostTypeSyncTime($postType);
        } catch (\Exception $e) {
            $this->logger->error('Bulk post upsert failed', [
                'post_type' => $postType,
                'error' => $e->getMessage(),
                'items_count' => count($upsertItems),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Process bulk delete items by sending them to cloud in batches
     *
     * @param string $postType    Post type
     * @param array  $deleteItems Array of delete items
     * @return void
     */
    private function processBulkDeleteItems(string $postType, array $deleteItems): void
    {
        try {
            if (count($deleteItems) === 1) {
                // Single delete - use existing method
                $this->deletePostFromCloud($deleteItems[0]['post_id'], $postType);
            } else {
                // Bulk delete - send all at once
                $postIds = array_column($deleteItems, 'post_id');
                $this->deleteBulkPostsFromCloud($postType, $postIds);
            }

            $this->logger->info('Processed bulk post delete items', [
                'post_type' => $postType,
                'items_count' => count($deleteItems)
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Bulk post delete failed', [
                'post_type' => $postType,
                'error' => $e->getMessage(),
                'items_count' => count($deleteItems),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Get product associations (frequently bought together) - Product-specific functionality
     *
     * @param int   $productId    Product ID
     * @param EventManager $eventManager Event manager instance
     * @param float $halfLifeDays Number of days for the time decay half-life
     *
     * @return array Array of associated products with scores
     */
    public function getProductAssociations(
        int $productId,
        EventManager $eventManager,
        float $halfLifeDays = 30.0
    ): array {
        // Get products frequently bought together
        $associations = $eventManager->getProductAssociations($productId, $halfLifeDays);

        $results = array_map(
            function ($item) {
                // Skip invalid products in results
                if ($this->productProvider->getPost($item['product_id'], 'product') === null) {
                    return null;
                }

                // Each component should be between 0-1:
                // - confidence is already a ratio (0-1)
                // - time_decay is already exponential decay (0-1)
                // Normalize frequency using sigmoid, then scale to 0-1 range
                $normalizedFrequency = (1 / (1 + exp(-$item['frequency'])));

                // Take the geometric mean of the three components
                // This ensures the final score is between 0-1 and rewards high values in all components
                $score = pow(
                    $item['confidence'] *
                    ($item['time_decay'] ?? 1.0) *
                    $normalizedFrequency,
                    1 / 3
                );

                return [
                    'id' => $item['product_id'],
                    'score' => $score
                ];
            },
            $associations
        );

        // Filter out any null values from invalid products
        $results = array_filter($results);

        // Re-index array to ensure sequential keys
        return array_values($results);
    }

    /**
     * Check if post should be synced
     *
     * @param int    $post_id   Post ID
     * @param string $post_type Post type
     * @return bool
     */
    public function shouldSyncPost(int $post_id, string $post_type): bool
    {
        // Always load the WP_Post; required for common checks and URL validation
        $wpPost = get_post($post_id);
        if (!$wpPost) {
            return false;
        }

        // Check if this post type is enabled in user preferences
        $userPreferences = get_option('intucart_syncable_post_types', []);
        if (!empty($userPreferences) && (!isset($userPreferences[$post_type]) || $userPreferences[$post_type] !== true)) {
            return false;
        }

        // Private posts require explicit opt-in for sync (security/privacy by default)
        // Check this before other validations to ensure private content isn't accidentally synced
        if ($wpPost->post_status === 'private') {
            $excludeFromSync = get_post_meta($post_id, '_intucart_exclude_from_sync', true);
            // Private posts only sync if explicitly enabled (set to 'no' = don't exclude = do sync)
            if ($excludeFromSync !== 'no') {
                return false; // Not explicitly enabled, so exclude from sync
            }
            // If we get here, private post was explicitly enabled - continue with other checks
        }

        // Product-specific validation when WooCommerce is active
        if ($post_type === 'product' && class_exists('WooCommerce')) {
            $product = wc_get_product($post_id);
            if (!$product) {
                return false;
            }

            // Skip variations and non-syncable product statuses
            if (in_array($product->get_type(), self::EXCLUDED_PRODUCT_TYPES, true)) {
                return false;
            }
            if (!$this->isPostStatusSyncable($product->get_status())) {
                return false;
            }

            // Exclude products hidden from catalog/search
            if (is_callable([$product, 'get_catalog_visibility'])) {
                $visibility = (string) $product->get_catalog_visibility();
                if ($visibility === 'hidden') {
                    return false;
                }
            }
        } else {
            // Non-product: ensure post type matches and status is syncable
            if ($wpPost->post_type !== $post_type) {
                return false;
            }
            if (!$this->isPostStatusSyncable($wpPost->post_status)) {
                return false;
            }
        }

        // Unified exclusions and public accessibility checks
        if ($this->isExcludedPost($post_id, $wpPost, $post_type)) {
            return false;
        }
        if (!$this->postHasAccessibleUrl($wpPost)) {
            return false;
        }

        return true;
    }

    /**
     * Check that a WordPress post has an accessible URL
     *
     * Requires the post type to be viewable, not password-protected, and have a valid permalink.
     * Private posts are allowed (they have URLs but require login for public viewing).
     *
     * @param \WP_Post $post
     * @return bool
     */
    private function postHasAccessibleUrl(\WP_Post $post): bool
    {
        $typeObject = get_post_type_object($post->post_type);
        if (!$typeObject || !is_post_type_viewable($typeObject)) {
            return false;
        }
        if (!empty($post->post_password)) {
            return false;
        }
        $permalink = get_permalink($post);
        if (!is_string($permalink) || $permalink === '') {
            return false;
        }
        // Private posts have valid URLs (for chatbot context) even though they require login
        if ($post->post_status === 'private') {
            return true;
        }
        // For built-in types, require permalink to resolve back to the same post ID.
        // For custom post types (e.g., ACF-registered), skip this strict check to avoid false negatives
        // when rewrites/templates differ but the content is still publicly viewable.
        if (in_array($post->post_type, ['post', 'page'], true)) {
            $resolvedId = url_to_postid($permalink);
            if ((int) $resolvedId !== (int) $post->ID) {
                return false;
            }
        }
        return true;
    }

    /**
     * Check if product is a variation
     *
     * @param \WC_Product $product Product object
     * @return bool
     */
    private function isProductVariation(\WC_Product $product): bool
    {
        return $product instanceof \WC_Product_Variation;
    }

    /**
     * Check if a post should be excluded from sync
     *
     * @param int      $post_id   Post ID
     * @param \WP_Post $post      Post object
     * @param string   $post_type Post type
     * @return bool True if post should be excluded, false otherwise
     */
    public function isExcludedPost(int $post_id, \WP_Post $post, string $post_type): bool
    {
        // Check manual exclusion first (admin control)
        $manualExclusion = get_post_meta($post_id, '_intucart_exclude_from_sync', true);
        if ($manualExclusion === 'yes') {
            return true;
        }
        // If manually set to 'no', allow sync even if auto-excluded
        if ($manualExclusion === 'no') {
            return false;
        }

        // Check for sitemap posts using regex
        $postSlug = $post->post_name;
        if (!empty($postSlug) && preg_match(self::SITEMAP_EXCLUSION_PATTERN, $postSlug)) {
            return true;
        }

        $postTitle = trim($post->post_title);
        if (!empty($postTitle) && preg_match(self::SITEMAP_EXCLUSION_PATTERN, $postTitle)) {
            return true;
        }

        // Inline page-style exclusion heuristics (apply to all post types)
        // 1) WooCommerce special pages (only meaningful for actual pages)
        if ($post_type === 'page' && class_exists('WooCommerce') && function_exists('wc_get_page_id')) {
            foreach (self::EXCLUDED_PAGE_TYPES as $pageType) {
                $wooPageId = wc_get_page_id($pageType);
                if ($wooPageId > 0 && $wooPageId === $post_id) {
                    return true;
                }
            }
        }

        // 2) Fallback by slug (common transactional/system slugs)
        $pageSlug = $post->post_name;
        if (!empty($pageSlug) && in_array($pageSlug, self::EXCLUDED_SLUGS, true)) {
            return true;
        }

        // 3) Fallback by title (case-insensitive match)
        $pageTitle = strtolower(trim($post->post_title));
        if (!empty($pageTitle) && in_array($pageTitle, self::EXCLUDED_TITLES, true)) {
            return true;
        }

        // 4) Backward-compat filter for page-style exclusions
        if ($post_type === 'page' && apply_filters('intucart_is_page_excluded_from_sync', false, $post_id, $post)) {
            return true;
        }

        // Allow filtering through WordPress filter
        return apply_filters('intucart_is_post_excluded_from_sync', false, $post_id, $post, $post_type);
    }

    /**
     * Get all available post types (both enabled and disabled)
     *
     * @return array
     */
    public function getAvailablePostTypes(): array
    {
        // Get all registered post types (both built-in and custom)
        $args = [
            'public' => true,
        ];
        $allPostTypes = get_post_types($args, 'names');

        // Filter out post types that we don't want to sync (media, revisions, etc.)
        $excludedPostTypes = ['attachment', 'revision', 'nav_menu_item'];
        $allPostTypes = array_diff($allPostTypes, $excludedPostTypes);

        // Always include important built-in types even if not marked as public
        $alwaysInclude = ['post', 'page'];
        foreach ($alwaysInclude as $type) {
            if (post_type_exists($type)) {
                $allPostTypes[] = $type;
            }
        }

        // Always include 'product' if WooCommerce is active (in case it's not public)
        if (class_exists('WooCommerce')) {
            $allPostTypes[] = 'product';
        }

        // Filter out post types that don't support meaningful content for markdown conversion
        $contentCapableTypes = array_filter(array_unique($allPostTypes), [$this, 'isPostTypeContentCapable']);

        // Filter out page builder and system post types that shouldn't be available for selection
        $pageBuilderExclusions = $this->pageBuilderService->getExcludedPostTypes();
        $allExclusions = array_merge($pageBuilderExclusions, self::EXCLUDED_SYSTEM_POST_TYPES);
        $availableTypes = array_diff($contentCapableTypes, $allExclusions);

        return $availableTypes;
    }

    /**
     * Check if a post type supports content features necessary for meaningful markdown conversion
     *
     * @param string $postType Post type to check
     * @return bool True if post type supports content features, false otherwise
     */
    private function isPostTypeContentCapable(string $postType): bool
    {
        // Products are handled specially by WooCommerceProductProvider, always allow them
        if ($postType === 'product') {
            return true;
        }

        // Get post type object for additional checks
        $postTypeObject = get_post_type_object($postType);
        if (!$postTypeObject) {
            return false;
        }

        // Ensure the post type is meant to be viewable/accessible
        // This excludes system post types that might technically support editor but aren't content
        if (!is_post_type_viewable($postTypeObject)) {
            return false;
        }

        // Check if post type has ACF field groups (ACF custom post types often don't have editor support)
        $hasAcfFields = $this->hasAcfFieldGroups($postType);

        // Check if post type supports title (important for meaningful content)
        $hasTitle = post_type_supports($postType, 'title');

        // Check if post type supports the editor (main content field)
        $hasEditor = post_type_supports($postType, 'editor');

        // Post type is content-capable if it has:
        // 1. Title AND editor (standard WordPress content types)
        // 2. Title AND ACF fields (ACF custom post types without editor)
        if (!$hasTitle) {
            return false;
        }

        if ($hasEditor || $hasAcfFields) {
            return true;
        }

        return false;
    }

    /**
     * Check if a post type has ACF field groups assigned to it
     *
     * @param string $postType Post type to check
     * @return bool True if post type has ACF field groups
     */
    private function hasAcfFieldGroups(string $postType): bool
    {
        // Check if ACF is available (consistent with AcfAutoMapper)
        if (!function_exists('acf_get_field_groups')) {
            return false;
        }

        try {
            // Get field groups for this post type
            $fieldGroups = acf_get_field_groups(['post_type' => $postType]);

            // ACF returns an array of field groups, check if any exist
            return !empty($fieldGroups) && is_array($fieldGroups);
        } catch (\Exception $e) {
            // If ACF throws any errors, treat as no ACF fields available
            $this->logger->debug('Error checking ACF field groups', [
                'post_type' => $postType,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get syncable post types based on user preferences
     *
     * @return array
     */
    public function getSyncablePostTypes(): array
    {
        // Get all available post types (already filtered for content capability and exclusions)
        $allPostTypes = $this->getAvailablePostTypes();

        // Get user preferences for which post types to include
        $userPreferences = get_option('intucart_syncable_post_types', []);

        // If no preferences are set, enable all available post types by default
        if (empty($userPreferences)) {
            $userPreferences = array_fill_keys($allPostTypes, true);
            update_option('intucart_syncable_post_types', $userPreferences);
        }

        // Filter post types based on user preferences
        $syncablePostTypes = [];
        foreach ($allPostTypes as $postType) {
            if (isset($userPreferences[$postType]) && $userPreferences[$postType] === true) {
                $syncablePostTypes[] = $postType;
            }
        }

        return $syncablePostTypes;
    }


    /**
     * Invalidate post cache
     *
     * @param int    $post_id   Post ID
     * @param string $post_type Post type
     * @return void
     */
    private function invalidatePostCache(int $post_id, string $post_type): void
    {
        $cacheKey = "intucart_post_data_{$post_type}_{$post_id}";
        $this->cache->delete($cacheKey, 'posts');
    }

    /**
     * Get post data for sync
     *
     * @param int    $post_id   Post ID
     * @param string $post_type Post type
     * @return array Post data
     */
    private function getPostData(int $post_id, string $post_type): array
    {
        // Check cache first
        $cacheKey = "intucart_post_data_{$post_type}_{$post_id}";
        $cached = $this->cache->get($cacheKey, 'posts');
        if ($cached !== false && $cached !== null) {
            return $cached;
        }

        // Build cloud payload and stamp last_updated
        $payload = $this->getCloudPayload($post_id, $post_type);
        $payload['last_updated'] = current_time('mysql');

        // Cache and return
        $this->cache->set($cacheKey, $payload, 'posts', self::CACHE_TTL_POST_DATA);
        return $payload;
    }

    /**
     * Validate post data structure
     *
     * @param array $postData Post data to validate
     * @return bool True if valid, false otherwise
     */
    private function validatePostData(array $postData): bool
    {
        // Required fields
        $requiredFields = ['id'];

        foreach ($requiredFields as $field) {
            if (!isset($postData[$field])) {
                $this->logger->warning('Invalid post data: missing required field', [
                    'field' => $field
                ]);
                return false;
            }
        }

        // Validate data types
        if (!is_numeric($postData['id']) || $postData['id'] <= 0) {
            $this->logger->warning('Invalid post data: invalid ID', [
                'id' => $postData['id']
            ]);
            return false;
        }

        // Validate content field if present
        if (isset($postData['content']) && strlen($postData['content']) > self::MAX_POST_CONTENT_LENGTH) {
            $this->logger->warning('Invalid post data: content too long', [
                'content_length' => strlen($postData['content'])
            ]);
            return false;
        }

        // Ensure we have meaningful content (title or content)
        $hasContent = false;
        if (!empty($postData['title'])) {
            $hasContent = true;
        } elseif (!empty($postData['content'])) {
            $hasContent = true;
        }

        if (!$hasContent) {
            $this->logger->warning('Invalid post data: no meaningful content found', [
                'post_id' => $postData['id']
            ]);
            return false;
        }

        // Validate title length if present
        if (isset($postData['title'])) {
            $titleLength = strlen($postData['title']);
            if ($titleLength > self::MAX_POST_TITLE_LENGTH || $titleLength < self::MIN_POST_TITLE_LENGTH) {
                $this->logger->warning('Invalid post data: title length out of bounds', [
                    'title_length' => $titleLength
                ]);
                return false;
            }
        }

        return true;
    }

    /**
     * Sync single post to cloud
     *
     * @param int    $post_id   Post ID
     * @param string $post_type Post type
     * @return void
     */
    private function syncSinglePostToCloud(int $post_id, string $post_type): void
    {
        $postData = $this->getPostData($post_id, $post_type);

        if (empty($postData)) {
            $this->logger->error('Post data is empty', [
                'post_id' => $post_id,
                'post_type' => $post_type
            ]);
            throw new \InvalidArgumentException('Post not found: ' . $post_id);
        }

        if (!$this->validatePostData($postData)) {
            $this->logger->error('Post data validation failed', [
                'post_id' => $post_id,
                'post_type' => $post_type
            ]);
            throw new \InvalidArgumentException('Invalid post data for: ' . $post_id);
        }

        try {
            // Generate content hash and determine operation type
            $contentHash = $this->generateContentHash($post_id);
            $existingSyncStatus = get_post_meta($post_id, self::META_SYNC_STATUS, true);
            $operationType = ($existingSyncStatus === 'success') ? 'update' : 'create';

            // Set pending status before sync attempt
            $this->setSyncStatus($post_id, 'pending', null, [
                'method' => 'manual',
                'operation' => $operationType,
                'content_hash' => $contentHash
            ]);

            // Ensure permalink available
            $postData['url'] = get_permalink($postData['id']);

            // Use appropriate AI SDK service based on post type
            // SDK throws exceptions on failure, so if we get here it succeeded
            $aiClient = $this->aiClientManager->getClient();
            if ($post_type === 'product') {
                $aiClient->products()->upsert($postData);
            } else {
                $aiClient->posts()->upsert($postData);
            }

            // Set success status
            $this->setSyncStatus($post_id, 'success', null, [
                'method' => 'manual',
                'operation' => $operationType,
                'content_hash' => $contentHash
            ]);

            $this->logger->info('Post synced to cloud successfully', [
                'post_id' => $post_id,
                'post_type' => $post_type
            ]);
        } catch (\Exception $e) {
            $contentHash = $this->generateContentHash($post_id);
            $existingSyncStatus = get_post_meta($post_id, self::META_SYNC_STATUS, true);
            $operationType = ($existingSyncStatus === 'success') ? 'update' : 'create';

            $this->setSyncStatus($post_id, 'error', $e->getMessage(), [
                'method' => 'manual',
                'operation' => $operationType,
                'content_hash' => $contentHash
            ]);
            $this->logger->error('Exception during cloud upsert', [
                'post_id' => $post_id,
                'post_type' => $post_type,
                'error_message' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Sync multiple posts to cloud in bulk
     *
     * @param string $post_type  Post type
     * @param array  $syncItems  Array of sync items
     * @return void
     */
    private function syncBulkPostsToCloud(string $post_type, array $syncItems): void
    {
        // Prepare entities for bulk upsert
        $entities = [];
        $validItems = [];

        foreach ($syncItems as $item) {
            try {
                $postData = $this->getPostData($item['post_id'], $post_type);

                if (empty($postData)) {
                    continue;
                }

                if (!$this->validatePostData($postData)) {
                    continue;
                }

                $uploadPayload = $postData;
                $uploadPayload['url'] = get_permalink($postData['id']);
                $entities[] = $uploadPayload;

                $validItems[] = $item;
            } catch (\Exception $e) {
                $this->logger->error('Failed to prepare post for bulk sync', [
                    'post_id' => $item['post_id'],
                    'post_type' => $post_type,
                    'error' => $e->getMessage()
                ]);
            }
        }

        if (empty($entities)) {
            return;
        }

        // Set pending status for all posts before starting bulk sync
        foreach ($validItems as $item) {
            $contentHash = $this->generateContentHash($item['post_id']);
            $existingSyncStatus = get_post_meta($item['post_id'], self::META_SYNC_STATUS, true);
            $operationType = ($existingSyncStatus === 'success') ? 'update' : 'create';

            $this->setSyncStatus($item['post_id'], 'pending', null, [
                'method' => 'bulk',
                'operation' => $operationType,
                'content_hash' => $contentHash
            ]);
        }

        // Chunk entities to respect cloud API limits
        $entityChunks = array_chunk($entities, self::MAX_BULK_UPSERT_SIZE);
        $validItemChunks = array_chunk($validItems, self::MAX_BULK_UPSERT_SIZE);
        $successfulChunks = 0;
        $totalSynced = 0;

        $this->logger->info('Starting bulk post sync', [
            'post_type' => $post_type,
            'total_entities' => count($entities),
            'total_chunks' => count($entityChunks)
        ]);

        foreach ($entityChunks as $chunkIndex => $chunk) {
            $chunkItems = $validItemChunks[$chunkIndex];
            try {
                // Use appropriate AI SDK service based on post type
                $aiClient = $this->aiClientManager->getClient();
                if ($post_type === 'product') {
                    $result = $aiClient->products()->bulkUpsert($chunk);
                } else {
                    $result = $aiClient->posts()->bulkUpsert($chunk);
                }

                // SDK throws on HTTP errors; if we get here, request succeeded
                // Process results - some items may have failed individually
                $successfulChunks++;
                $chunkSuccessCount = count($result['successful_ids'] ?? []);
                $totalSynced += $chunkSuccessCount;

                // Create lookup map for successful IDs
                $successfulIdSet = array_flip($result['successful_ids'] ?? []);

                // Set status for each post in this chunk based on individual results
                foreach ($chunkItems as $item) {
                    $postId = (string) $item['post_id'];
                    $contentHash = $this->generateContentHash($item['post_id']);
                    $existingSyncStatus = get_post_meta($item['post_id'], self::META_SYNC_STATUS, true);
                    $operationType = ($existingSyncStatus === 'success') ? 'update' : 'create';

                    if (isset($successfulIdSet[$postId])) {
                        $this->setSyncStatus($item['post_id'], 'success', null, [
                            'method' => 'bulk',
                            'operation' => $operationType,
                            'content_hash' => $contentHash
                        ]);
                    } else {
                        // Find specific error for this post from failed_items
                        $specificError = 'Unknown error';
                        foreach ($result['failed_items'] ?? [] as $failedItem) {
                            if ((string) $failedItem['id'] === $postId) {
                                $specificError = $failedItem['error'];
                                break;
                            }
                        }
                        $this->setSyncStatus($item['post_id'], 'error', $specificError, [
                            'method' => 'bulk',
                            'operation' => $operationType,
                            'content_hash' => $contentHash
                        ]);
                    }
                }

                // Log detailed results for this chunk
                if (!empty($result['failed_items'])) {
                    $this->logger->warning('Some posts failed in bulk sync chunk', [
                        'post_type' => $post_type,
                        'chunk' => $chunkIndex + 1,
                        'successful_count' => $chunkSuccessCount,
                        'failed_count' => count($result['failed_items']),
                        'failed_items' => array_slice($result['failed_items'], 0, 5),
                    ]);
                }
            } catch (\Exception $e) {
                // Set error status for all posts in this failed chunk
                foreach ($chunkItems as $item) {
                    $contentHash = $this->generateContentHash($item['post_id']);
                    $existingSyncStatus = get_post_meta($item['post_id'], self::META_SYNC_STATUS, true);
                    $operationType = ($existingSyncStatus === 'success') ? 'update' : 'create';

                    $this->setSyncStatus($item['post_id'], 'error', $e->getMessage(), [
                        'method' => 'bulk',
                        'operation' => $operationType,
                        'content_hash' => $contentHash
                    ]);
                }

                $this->logger->error('Bulk post sync chunk failed', [
                    'post_type' => $post_type,
                    'chunk' => $chunkIndex + 1,
                    'error' => $e->getMessage()
                ]);
                continue;
            }
        }

        if ($successfulChunks === 0) {
            throw new \Exception('All bulk post sync chunks failed for type: ' . $post_type);
        }

        $this->logger->info('Bulk post sync completed', [
            'post_type' => $post_type,
            'entities_synced' => $totalSynced,
            'successful_chunks' => $successfulChunks,
            'total_chunks' => count($entityChunks)
        ]);
    }

    /**
     * Delete post from cloud
     *
     * @param int    $post_id   Post ID
     * @param string $post_type Post type
     * @return void
     */
    private function deletePostFromCloud(int $post_id, string $post_type): void
    {
        try {
            // Set pending status before delete attempt
            $this->setSyncStatus($post_id, 'pending', null, [
                'method' => 'manual',
                'operation' => 'delete'
            ]);

            // Use appropriate AI SDK service based on post type
            // SDK throws exceptions on failure, so if we get here it succeeded
            $aiClient = $this->aiClientManager->getClient();
            if ($post_type === 'product') {
                $aiClient->products()->delete((string)$post_id);
            } else {
                $aiClient->posts()->delete((string)$post_id);
            }

            // Clear sync status after successful deletion (post is no longer in cloud)
            $this->clearSyncStatus($post_id);
        } catch (\Exception $e) {
            $this->setSyncStatus($post_id, 'error', $e->getMessage(), [
                'method' => 'manual',
                'operation' => 'delete'
            ]);
            throw $e;
        }
    }

    /**
     * Delete multiple posts from cloud in bulk
     *
     * @param string $post_type Post type
     * @param array  $post_ids  Array of post IDs
     * @return void
     */
    private function deleteBulkPostsFromCloud(string $post_type, array $post_ids): void
    {
        if (empty($post_ids)) {
            return;
        }

        // Convert to string IDs for cloud API (SDK wraps in 'ids' key)
        $stringIds = array_map('strval', $post_ids);

        // Chunk IDs to respect cloud API limits
        $idChunks = array_chunk($stringIds, self::MAX_BULK_DELETE_SIZE);
        $successfulChunks = 0;
        $totalDeleted = 0;

        foreach ($idChunks as $chunkIndex => $chunk) {
            try {
                // Use appropriate AI SDK service based on post type
                // SDK throws exceptions on failure, so if we get here it succeeded
                $aiClient = $this->aiClientManager->getClient();
                if ($post_type === 'product') {
                    $aiClient->products()->bulkDelete($chunk);
                } else {
                    $aiClient->posts()->bulkDelete($chunk);
                }

                $successfulChunks++;
                $totalDeleted += count($chunk);
            } catch (\Exception $e) {
                $this->logger->error('Bulk post delete chunk failed', [
                    'post_type' => $post_type,
                    'chunk' => $chunkIndex + 1,
                    'error' => $e->getMessage()
                ]);
                continue;
            }
        }

        if ($successfulChunks === 0) {
            throw new \Exception('All bulk post delete chunks failed for type: ' . $post_type);
        }

        $this->logger->info('Bulk post delete completed', [
            'post_type' => $post_type,
            'ids_deleted' => $totalDeleted
        ]);
    }

    /**
     * Get the appropriate limit for a post type from tier limits
     *
     * @param string $postType Post type
     * @param array $tierLimits Tier limits array
     * @return int The limit for this post type
     */
    private function getPostTypeLimit(string $postType, array $tierLimits): int
    {
        switch ($postType) {
            case 'product':
                return $tierLimits['productsIndexed'] ?? 0;
            case 'user':
                return $tierLimits['usersIndexed'] ?? 0;
            case 'taxonomy':
                return $tierLimits['taxonomiesIndexed'] ?? 0;
            default:
                // All other post types use postsIndexed limit
                return $tierLimits['postsIndexed'] ?? 0;
        }
    }

    /**
     * Update post type sync time
     *
     * @param string $postType Post type
     * @return void
     */
    private function updatePostTypeSyncTime(string $postType): void
    {
        $syncTime = time();
        $optionName = "intucart_last_{$postType}_sync";
        update_option($optionName, $syncTime);
    }

    /**
     * Perform cleanup using cloud-based ID retrieval and batch processing
     *
     * @param string $postType Post type to clean up
     * @param array $validPostIds Array of valid WordPress post IDs
     * @return void
     */
    private function performPostCleanup(string $postType, array $validPostIds): void
    {
        // Get all post IDs from the cloud for this post type
        $cloudPostIds = $this->getCloudPostIds($postType);

        if (empty($cloudPostIds)) {
            return;
        }

        // Find IDs that exist in cloud but not in WordPress
        $validIdSet = array_flip(array_map('strval', $validPostIds));
        $staleIds = [];

        foreach ($cloudPostIds as $cloudId) {
            if (!isset($validIdSet[$cloudId])) {
                $staleIds[] = $cloudId;
            }
        }

        if (empty($staleIds)) {
            return;
        }

        // Process stale IDs in batches to respect OpenSearch terms query limits
        $this->deleteStalePosts($postType, $staleIds);
    }

    /**
     * Get all post IDs from the cloud for a specific post type
     *
     * @param string $postType Post type to retrieve IDs for
     * @return array Array of post IDs from the cloud
     */
    private function getCloudPostIds(string $postType): array
    {
        $allIds = [];
        $offset = 0;
        $limit = 10000; // Use reasonable batch size
        $hasMore = true;

        while ($hasMore) {
            try {
                // Use appropriate AI SDK service based on post type
                $aiClient = $this->aiClientManager->getClient();
                if ($postType === 'product') {
                    $response = $aiClient->products()->getIds(['limit' => $limit, 'offset' => $offset]);
                } else {
                    $response = $aiClient->posts()->getIds(['post_type' => $postType, 'limit' => $limit, 'offset' => $offset]);
                }

                // Check if response is valid (getIds returns data directly, not success wrapper)
                if (!is_array($response)) {
                    $this->logger->warning('Invalid response from getIds API', [
                        'post_type' => $postType,
                        'response_type' => gettype($response)
                    ]);
                    break;
                }

                $batchIds = $response['ids'] ?? [];
                $allIds = array_merge($allIds, $batchIds);

                $hasMore = $response['has_more'] ?? false;
                $offset += $limit;
            } catch (\Exception $e) {
                $this->logger->error('Exception while retrieving cloud post IDs', [
                    'post_type' => $postType,
                    'offset' => $offset,
                    'error' => $e->getMessage()
                ]);
                break;
            }
        }

        return $allIds;
    }

    /**
     * Delete stale posts in batches to respect OpenSearch terms query limits
     *
     * @param string $postType Post type
     * @param array $staleIds Array of stale post IDs to delete
     * @return void
     */
    private function deleteStalePosts(string $postType, array $staleIds): void
    {
        $batchSize = 5000; // Safe batch size well below OpenSearch 65k terms limit
        $batches = array_chunk($staleIds, $batchSize);
        $totalDeleted = 0;
        $totalFailed = 0;

        foreach ($batches as $batchIndex => $batch) {
            try {
                $filters = [
                    ['term' => ['post_type' => $postType]],
                    ['terms' => ['external_id' => $batch]]
                ];

                // Use appropriate AI SDK service based on post type
                $aiClient = $this->aiClientManager->getClient();
                if ($postType === 'product') {
                    $cleanupResponse = $aiClient->products()->deleteByQuery($filters);
                } else {
                    $cleanupResponse = $aiClient->posts()->deleteByQuery($filters);
                }

                // SDK throws on failure, so if we get here it succeeded
                $deletedCount = $cleanupResponse['deleted_count'] ?? 0;
                $totalDeleted += $deletedCount;

                $this->logger->info('Batch cleanup completed', [
                    'post_type' => $postType,
                    'batch' => $batchIndex + 1,
                    'batch_size' => count($batch),
                    'deleted_count' => $deletedCount
                ]);
            } catch (\Exception $e) {
                $totalFailed += count($batch);
                $this->logger->error('Exception during batch cleanup', [
                    'post_type' => $postType,
                    'batch' => $batchIndex + 1,
                    'batch_size' => count($batch),
                    'error' => $e->getMessage()
                ]);
            }
        }
    }

    /**
     * Set sync status for a post with enhanced metadata
     *
     * @param int $post_id Post ID
     * @param string $status Status: 'success', 'error', 'pending'
     * @param string|null $error_message Error message if status is 'error'
     * @param array $metadata Additional sync metadata
     *   - method: 'manual', 'auto', 'bulk', 'realtime'
     *   - operation: 'create', 'update', 'delete'
     *   - content_hash: Hash of content at sync time
     * @return void
     */
    public function setSyncStatus(int $post_id, string $status, ?string $error_message = null, array $metadata = []): void
    {
        update_post_meta($post_id, self::META_SYNC_STATUS, $status);
        update_post_meta($post_id, self::META_SYNC_TIMESTAMP, time());

        // Handle error message
        if ($status === 'error' && $error_message) {
            update_post_meta($post_id, self::META_SYNC_ERROR, $error_message);
        } else {
            // Clear error message for successful syncs
            delete_post_meta($post_id, self::META_SYNC_ERROR);
        }

        // Handle additional metadata
        if (!empty($metadata['method'])) {
            update_post_meta($post_id, self::META_SYNC_METHOD, $metadata['method']);
        }

        if (!empty($metadata['operation'])) {
            update_post_meta($post_id, self::META_SYNC_OPERATION, $metadata['operation']);
        }

        if (!empty($metadata['content_hash'])) {
            update_post_meta($post_id, self::META_SYNC_CONTENT_HASH, $metadata['content_hash']);
        }
    }

    /**
     * Get sync status for a post with all metadata
     *
     * @param int $post_id Post ID
     * @return array Array with all sync metadata
     */
    public function getSyncStatus(int $post_id): array
    {
        return [
            'status' => get_post_meta($post_id, self::META_SYNC_STATUS, true) ?: '',
            'error' => get_post_meta($post_id, self::META_SYNC_ERROR, true) ?: '',
            'timestamp' => (int) get_post_meta($post_id, self::META_SYNC_TIMESTAMP, true) ?: 0,
            'method' => get_post_meta($post_id, self::META_SYNC_METHOD, true) ?: '',
            'operation' => get_post_meta($post_id, self::META_SYNC_OPERATION, true) ?: '',
            'content_hash' => get_post_meta($post_id, self::META_SYNC_CONTENT_HASH, true) ?: '',
        ];
    }

    /**
     * Clear all sync status and metadata for a post
     *
     * @param int $post_id Post ID
     * @return void
     */
    public function clearSyncStatus(int $post_id): void
    {
        delete_post_meta($post_id, self::META_SYNC_STATUS);
        delete_post_meta($post_id, self::META_SYNC_ERROR);
        delete_post_meta($post_id, self::META_SYNC_TIMESTAMP);
        delete_post_meta($post_id, self::META_SYNC_METHOD);
        delete_post_meta($post_id, self::META_SYNC_OPERATION);
        delete_post_meta($post_id, self::META_SYNC_CONTENT_HASH);
    }

    /**
     * Generate content hash for a post to detect changes
     *
     * @param int $post_id Post ID
     * @return string Content hash
     */
    public function generateContentHash(int $post_id): string
    {
        $post = get_post($post_id);
        if (!$post) {
            return '';
        }

        // Include key content fields that would trigger re-sync
        $content_data = [
            'title' => $post->post_title,
            'content' => $post->post_content,
            'excerpt' => $post->post_excerpt,
            'modified' => $post->post_modified_gmt,
        ];

        // For products, include relevant meta that affects search
        if ($post->post_type === 'product' && class_exists('WooCommerce')) {
            $product = wc_get_product($post_id);
            if ($product) {
                // Core product data that affects search/filtering
                $content_data['price'] = $product->get_price();
                $content_data['regular_price'] = $product->get_regular_price();
                $content_data['sale_price'] = $product->get_sale_price();
                $content_data['stock_status'] = $product->get_stock_status();
                $content_data['stock_quantity'] = $product->get_stock_quantity();
                $content_data['featured'] = $product->is_featured();
                $content_data['on_sale'] = $product->is_on_sale();
                $content_data['sku'] = $product->get_sku();
                $content_data['weight'] = $product->get_weight();
                $content_data['dimensions'] = [
                    'length' => $product->get_length(),
                    'width' => $product->get_width(),
                    'height' => $product->get_height(),
                ];
                
                // Categories and tags (affect filtering/search)
                $content_data['categories'] = $product->get_category_ids();
                $content_data['tags'] = wp_get_post_terms($post_id, 'product_tag', ['fields' => 'ids']);
                $content_data['brands'] = wp_get_post_terms($post_id, 'product_brand', ['fields' => 'ids']);
                
                // Product attributes (affect filtering)
                $attributes_hash = [];
                foreach ($product->get_attributes() as $attribute) {
                    if (is_object($attribute) && method_exists($attribute, 'get_visible') && $attribute->get_visible()) {
                        $slug = $attribute->get_name();
                        if ($attribute->is_taxonomy() && method_exists($attribute, 'get_terms')) {
                            $terms = $attribute->get_terms();
                            if (!empty($terms) && !is_wp_error($terms)) {
                                $attributes_hash[$slug] = array_map(function($term) {
                                    return $term->term_id;
                                }, $terms);
                            }
                        } elseif (method_exists($attribute, 'get_options')) {
                            $attributes_hash[$slug] = $attribute->get_options();
                        }
                    }
                }
                $content_data['attributes'] = $attributes_hash;
                
                // Product flags that affect search/filtering
                $content_data['product_flags'] = [
                    'downloadable' => $product->is_downloadable(),
                    'virtual' => $product->is_virtual(),
                    'purchasable' => $product->is_purchasable(),
                    'sold_individually' => $product->is_sold_individually(),
                ];
                
                // Image changes
                $content_data['image_id'] = $product->get_image_id();
            }
        }

        return md5(serialize($content_data));
    }

    /**
     * Check if post content has changed since last sync
     *
     * @param int $post_id Post ID
     * @return bool True if content has changed
     */
    public function hasContentChanged(int $post_id): bool
    {
        $currentHash = $this->generateContentHash($post_id);
        $lastHash = get_post_meta($post_id, self::META_SYNC_CONTENT_HASH, true);

        return $currentHash !== $lastHash;
    }
}
