<?php

namespace Intucart\Services\Managers;

use Intucart\Services\Logger;
use Intucart\Services\AIClientManager;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\Constants;
use Intucart\Utils\StringUtils;

/**
 * TaxonomyManager service
 *
 * Handles syncing WordPress taxonomies and terms to IntUCart Cloud
 * Self-contained batch processing with intelligent queuing and cloud sync
 */
class TaxonomyManager
{
    /**
     * Taxonomies to exclude from sync (system/internal taxonomies)
     *
     * @var array<string>
     */
    private const EXCLUDED_TAXONOMIES = [
        'nav_menu',              // WordPress navigation menus
        'link_category',         // WordPress link categories (deprecated)
        'post_format',           // WordPress post formats
        'wp_theme',              // WordPress theme taxonomy
        'wp_template_part_area', // WordPress template parts
        'wp_pattern_category',   // WordPress pattern categories
        'action-group',          // Action Scheduler groups
        'elementor_library_type', // Elementor library types
        'elementor_library_category', // Elementor library categories
        'wp_block',              // WordPress reusable blocks
        'wp_navigation',         // WordPress navigation blocks
        'wp_global_styles',      // WordPress global styles
        'oembed_cache',          // WordPress oEmbed cache
        'customize_changeset',   // WordPress customizer changesets
        'user_request',          // WordPress privacy user requests
        'scheduled-action',      // Action Scheduler actions
    ];

    /**
     * Batch processing configuration constants
     */
    private const BATCH_DELAY_UPDATE = 30;    // Seconds to delay update batches
    private const BATCH_DELAY_DELETE = 10;    // Seconds to delay delete batches (faster for better UX)
    private const BATCH_QUEUE_EXPIRY = 300;   // Seconds for batch queue transient expiry (5 minutes)

    /**
     * Cache configuration constants
     */
    private const CACHE_TTL_TAXONOMY_DATA = 3600; // 1 hour cache for taxonomy data

    /**
     * Sync configuration constants
     */
    private const SYNC_SCHEDULE_RECURRENCE = 'daily';                    // How often to run full sync
    private const BATCH_PROCESS_HOOK = 'intucart_process_taxonomy_batch'; // Hook name for batch processing
    private const BATCH_QUEUE_TRANSIENT = 'intucart_taxonomy_batch_queue'; // Transient key for batch queue

    /**
     * Validation constants - generous limits for term names and descriptions
     */
    private const MAX_TERM_NAME_LENGTH = 200;        // Max characters for term name
    private const MAX_TERM_DESCRIPTION_LENGTH = 1000; // Max characters for term description
    private const MIN_TERM_NAME_LENGTH = 1;          // Minimum meaningful term name

    /**
     * Cloud API batch size limits
     * These should match the limits in taxonomyHandler.ts and other cloud handlers
     */
    private const MAX_BULK_UPSERT_SIZE = 100;  // Max items per bulk upsert request
    private const MAX_BULK_DELETE_SIZE = 100;  // Max items per bulk delete request

    private Logger $logger;
    private AIClientManager $aiClientManager;
    private CacheService $cache;

    /**
     * Constructor
     *
     * @param Logger           $logger           Logger service
     * @param AIClientManager  $aiClientManager  AI client manager service
     * @param CacheService     $cache            Cache service
     */
    public function __construct(
        Logger $logger,
        AIClientManager $aiClientManager,
        CacheService $cache
    ) {
        $this->logger = $logger;
        $this->aiClientManager = $aiClientManager;
        $this->cache = $cache;
    }

    /**
     * Sync taxonomy data to cloud (handles all sync actions)
     *
     * @param array $payload Sync payload with action type and data
     * @return array|null Response data or null on failure
     */
    private function syncTaxonomy(array $payload): ?array
    {
        // Validate payload structure
        if (!isset($payload['action']) || !isset($payload['data'])) {
            return [
                'error' => 'Invalid payload: missing action or data',
                'success' => false,
            ];
        }

        $action = $payload['action'];
        $data = $payload['data'];

        // Route to appropriate method based on action
        switch ($action) {
            case 'taxonomy_full_sync':
            case 'taxonomy_update':
                // Convert taxonomy data to individual term upsert requests
                return $this->upsertTaxonomy($data);

            case 'taxonomy_delete':
                // Convert to term delete request
                return $this->deleteTaxonomyTerm($data);

            case 'taxonomy_bulk_delete':
                // Convert to bulk delete request
                return $this->bulkDeleteTaxonomyTerms($data);

            case 'taxonomy_bulk_upsert':
                // Convert to bulk upsert request
                return $this->bulkUpsertTaxonomyTerms($data);

            default:
                return [
                    'error' => 'Unknown action: ' . $action,
                    'success' => false,
                ];
        }
    }

    /**
     * Handle taxonomy upsert (full sync or update)
     */
    private function upsertTaxonomy(array $data): ?array
    {
        if (!isset($data['taxonomy']) || !isset($data['terms'])) {
            return [
                'error' => 'Invalid taxonomy data: missing taxonomy or terms',
                'success' => false,
            ];
        }

        $taxonomy = $data['taxonomy'];
        $terms = $data['terms'];

        if (empty($terms)) {
            return [
                'message' => 'No terms to sync for taxonomy: ' . $taxonomy,
                'success' => true,
            ];
        }

        // Convert terms to the format expected by the cloud taxonomy handler
        $upsertRequests = [];
        foreach ($terms as $term) {
            $upsertRequests[] = [
                'id' => (string)$term['id'],
                'taxonomy_name' => $taxonomy,
                'term_name' => $term['name'],
                'slug' => $term['slug'],
                'description' => $term['description'] ?? '',
                'count' => $term['count'] ?? 0,
                'parent' => $term['parent'] ?? 0,
                'meta' => $term['meta'] ?? null,
            ];
        }

        // Use AI SDK for bulk upsert
        try {
            $aiClient = $this->aiClientManager->getClient();
            $taxonomyService = $aiClient->taxonomy();
            
            return $taxonomyService->bulkUpsert($upsertRequests);
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Handle taxonomy term delete
     */
    private function deleteTaxonomyTerm(array $data): ?array
    {
        if (!isset($data['taxonomy']) || !isset($data['term_id'])) {
            return [
                'error' => 'Invalid delete data: missing taxonomy or term_id',
                'success' => false,
            ];
        }

        try {
            $aiClient = $this->aiClientManager->getClient();
            $taxonomyService = $aiClient->taxonomy();
            
            return $taxonomyService->delete((string)$data['term_id']);
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Handle taxonomy bulk delete
     */
    private function bulkDeleteTaxonomyTerms(array $data): ?array
    {
        if (!isset($data['taxonomy']) || !isset($data['term_ids'])) {
            return [
                'error' => 'Invalid bulk delete data: missing taxonomy or term_ids',
                'success' => false,
            ];
        }

        // Convert to format expected by cloud handler: { items: [{ taxonomy_name, id }] }
        $items = [];
        foreach ($data['term_ids'] as $termId) {
            $items[] = [
                'taxonomy_name' => $data['taxonomy'],
                'id' => $termId,
            ];
        }

        try {
            $aiClient = $this->aiClientManager->getClient();
            $taxonomyService = $aiClient->taxonomy();
            
            return $taxonomyService->bulkDelete($items);
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Handle taxonomy bulk upsert
     */
    private function bulkUpsertTaxonomyTerms(array $data): ?array
    {
        if (!isset($data['items']) || !is_array($data['items'])) {
            return [
                'error' => 'Invalid bulk upsert data: missing items array',
                'success' => false,
            ];
        }

        try {
            $aiClient = $this->aiClientManager->getClient();
            $taxonomyService = $aiClient->taxonomy();
            
            return $taxonomyService->bulkUpsert($data['items']);
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Get taxonomy term IDs from cloud for cleanup operations
     *
     * @param array $filters Filters for taxonomy terms
     * @param int $limit Maximum number of IDs
     * @param int $offset Offset for pagination
     * @return array Response with taxonomy term IDs
     */
    private function getTaxonomyIds(array $filters = [], int $limit = 10000, int $offset = 0): array
    {
        try {
            $aiClient = $this->aiClientManager->getClient();
            $taxonomyService = $aiClient->taxonomy();
            
            return $taxonomyService->getIds($filters, $limit, $offset);
        } catch (\Exception $e) {
            $this->logger->error('Failed to get taxonomy IDs', [
                'error' => $e->getMessage(),
                'filters' => $filters
            ]);
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Delete taxonomy terms by query filters
     *
     * @param array $filters OpenSearch filters for deletion
     * @return array Response from cloud service
     */
    private function deleteByQuery(array $filters): array
    {
        try {
            $aiClient = $this->aiClientManager->getClient();
            $taxonomyService = $aiClient->taxonomy();
            
            return $taxonomyService->deleteByQuery($filters);
        } catch (\Exception $e) {
            $this->logger->error('Delete by query failed', [
                'error' => $e->getMessage(),
                'filters' => $filters
            ]);
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Initialize the taxonomy sync service
     *
     * @return void
     */
    public function initialize(): void
    {
        // Term hooks for real-time syncing
        add_action('created_term', [$this, 'handleTermChange'], 10, 3);
        add_action('edited_term', [$this, 'handleTermChange'], 10, 3);
        add_action('delete_term', [$this, 'handleTermDelete'], 10, 4);

        // Taxonomy hooks
        add_action('registered_taxonomy', [$this, 'handleTaxonomyRegistered'], 10, 3);

        // Batch processing hooks
        add_action(self::BATCH_PROCESS_HOOK, [$this, 'handleTaxonomyBatch']);
        add_action(Constants::TAXONOMY_SYNC_HOOK, [$this, 'handleScheduledTaxonomySync']);

        // Admin hooks for manual sync
        add_action('wp_ajax_intucart_sync_taxonomies', [$this, 'handleManualTaxonomySync']);

        // Schedule daily full sync
        if (!wp_next_scheduled(Constants::TAXONOMY_SYNC_HOOK)) {
            wp_schedule_event(time(), self::SYNC_SCHEDULE_RECURRENCE, Constants::TAXONOMY_SYNC_HOOK);
        }
    }

    /**
     * Handle term creation or update
     *
     * @param int    $term_id  Term ID
     * @param int    $tt_id    Term taxonomy ID
     * @param string $taxonomy Taxonomy slug
     * @return void
     */
    public function handleTermChange(int $term_id, int $tt_id, string $taxonomy): void
    {
        // Validate inputs
        if (!$this->validateTaxonomySlug($taxonomy)) {
            return;
        }

        if ($term_id <= 0) {
            $this->logger->warning('Invalid term change: invalid term ID', [
            'term_id' => $term_id,
            'taxonomy' => $taxonomy
            ]);
            return;
        }

        if (!$this->shouldSyncTaxonomy($taxonomy)) {
            return;
        }

        $this->logger->info('Term changed, queueing for batch sync', [
        'term_id' => $term_id,
        'taxonomy' => $taxonomy,
        'action' => 'update'
        ]);

        $this->queueTaxonomyBatchAction($taxonomy, 'update', $term_id);
        $this->invalidateTaxonomyCache($taxonomy);
    }

    /**
     * Handle term deletion
     *
     * @param int    $term_id  Term ID
     * @param int    $tt_id    Term taxonomy ID
     * @param string $taxonomy Taxonomy slug
     * @param object $deleted_term Deleted term object
     * @return void
     */
    public function handleTermDelete(int $term_id, int $tt_id, string $taxonomy, $deleted_term): void
    {
        // Validate inputs
        if (!$this->validateTaxonomySlug($taxonomy)) {
            return;
        }

        if ($term_id <= 0) {
            $this->logger->warning('Invalid term delete: invalid term ID', [
            'term_id' => $term_id,
            'taxonomy' => $taxonomy
            ]);
            return;
        }

        if (!$this->shouldSyncTaxonomy($taxonomy)) {
            return;
        }

        $this->logger->info('Term deleted, queueing for batch sync', [
        'term_id' => $term_id,
        'taxonomy' => $taxonomy,
        'action' => 'delete'
        ]);

        $this->queueTaxonomyBatchAction($taxonomy, 'delete', $term_id);
        $this->invalidateTaxonomyCache($taxonomy);
    }

    /**
     * Handle taxonomy registration (for dynamic product attributes)
     *
     * @param string $taxonomy Taxonomy name
     * @param array  $object_type Object types
     * @param array  $args Taxonomy arguments
     * @return void
     */
    public function handleTaxonomyRegistered(string $taxonomy, $object_type, $args): void
    {
        // Auto-sync WooCommerce product attributes
        if (strpos($taxonomy, 'pa_') === 0) {
            // Check if this taxonomy has been recently synced or is already queued
            $lastSync = get_option("intucart_last_{$taxonomy}_sync", 0);
            $recentSyncThreshold = time() - (6 * HOUR_IN_SECONDS); // Don't re-sync within 6 hours

            if ($lastSync > $recentSyncThreshold) {
                // Already synced recently, skip
                return;
            }

            // Check if already queued
            $queue = get_transient(self::BATCH_QUEUE_TRANSIENT) ?: [];
            $queueKey = $taxonomy . '|full_sync';
            if (isset($queue[$queueKey])) {
                // Already queued, skip
                return;
            }

            $this->logger->info('Product attribute taxonomy registered', [
            'taxonomy' => $taxonomy,
            'object_type' => $object_type
            ]);

            // Schedule full sync for this new attribute
            $this->queueTaxonomyBatchAction($taxonomy, 'full_sync');
        }
    }

    /**
     * Queue a taxonomy action for batch processing
     *
     * @param string   $taxonomy Taxonomy slug
     * @param string   $action   'update', 'delete', or 'full_sync'
     * @param int|null $term_id  Term ID (null for full_sync)
     * @return void
     */
    private function queueTaxonomyBatchAction(string $taxonomy, string $action, ?int $term_id = null): void
    {
        // All actions (including deletes) go through the same batching system for consistency
        $queue = get_transient(self::BATCH_QUEUE_TRANSIENT) ?: [];

        $key = $taxonomy . '|' . $action;
        if ($term_id) {
            $key .= '|' . $term_id;
        }

        $queue[$key] = [
        'taxonomy' => $taxonomy,
        'action' => $action,
        'term_id' => $term_id,
        'timestamp' => time()
        ];

        set_transient(self::BATCH_QUEUE_TRANSIENT, $queue, self::BATCH_QUEUE_EXPIRY);

        // For deletes, use a shorter delay for better UX
        $delay = ($action === 'delete') ? self::BATCH_DELAY_DELETE : self::BATCH_DELAY_UPDATE;

        // Schedule batch job if not already scheduled
        $this->scheduleTaxonomyBatch($delay);

        $this->logger->info('Queued taxonomy action for batch processing', [
        'taxonomy' => $taxonomy,
        'action' => $action,
        'term_id' => $term_id,
        'delay' => $delay
        ]);
    }

    /**
     * Schedule taxonomy batch processing using WordPress cron
     *
     * @param int $delay Delay in seconds
     * @return void
     */
    private function scheduleTaxonomyBatch(int $delay): void
    {
        // Check if already scheduled
        if (!wp_next_scheduled(self::BATCH_PROCESS_HOOK)) {
            wp_schedule_single_event(time() + $delay, self::BATCH_PROCESS_HOOK);
        }
    }

    /**
     * Process taxonomy batch queue
     *
     * @return void
     */
    public function handleTaxonomyBatch(): void
    {
        $queue = get_transient(self::BATCH_QUEUE_TRANSIENT) ?: [];
        if (empty($queue)) {
            return;
        }

        delete_transient(self::BATCH_QUEUE_TRANSIENT);

        $this->logger->info('Processing taxonomy batch queue', [
        'queue_size' => count($queue)
        ]);

        // Group actions by taxonomy
        $taxonomyActions = [];

        foreach ($queue as $item) {
            $taxonomy = $item['taxonomy'];
            if (!isset($taxonomyActions[$taxonomy])) {
                $taxonomyActions[$taxonomy] = [];
            }
            $taxonomyActions[$taxonomy][] = $item;
        }

        // Separate sync and delete items for different batch types
        $syncItems = [];
        $deleteItems = [];

        // Process each taxonomy and prepare batch items
        foreach ($taxonomyActions as $taxonomy => $actions) {
            $needsFullSync = false;
            foreach ($actions as $action) {
                if ($action['action'] === 'full_sync') {
                    $needsFullSync = true;
                    break;
                }
            }

            if ($needsFullSync) {
                // Full sync for this taxonomy
                $taxonomyData = $this->getTaxonomyData($taxonomy);
                if (!empty($taxonomyData['terms'])) {
                    $syncItems[] = [
                    'action' => 'taxonomy_full_sync',
                    'data' => [
                        'taxonomy' => $taxonomy,
                        'hierarchical' => $taxonomyData['hierarchical'],
                        'terms' => $taxonomyData['terms'],
                        'name' => $taxonomyData['name'],
                        'label' => $taxonomyData['label'],
                        'public' => $taxonomyData['public'] ?? false
                    ]
                    ];
                }
            } else {
                // Individual term updates and deletes
                foreach ($actions as $action) {
                    if ($action['action'] === 'update' && $action['term_id']) {
                        $term = get_term($action['term_id'], $taxonomy);
                        if (!is_wp_error($term) && $term) {
                            $termData = $this->formatTermData($term);

                            // Validate term data before adding to sync queue
                            if (!$this->validateTermData($termData)) {
                                $this->logger->warning('Skipping invalid term in batch processing', [
                                'term_id' => $action['term_id'],
                                'taxonomy' => $taxonomy
                                ]);
                                continue;
                            }

                            $fullTaxonomyData = $this->getTaxonomyData($taxonomy);

                            $syncItems[] = [
                            'action' => 'taxonomy_update',
                            'data' => [
                                'taxonomy' => $taxonomy,
                                'term' => $termData,
                                'hierarchical' => $fullTaxonomyData['hierarchical'],
                                'terms' => $fullTaxonomyData['terms'],
                                'name' => $fullTaxonomyData['name'],
                                'label' => $fullTaxonomyData['label'],
                                'public' => $fullTaxonomyData['public'] ?? false
                            ]
                            ];
                        }
                    } elseif ($action['action'] === 'delete' && $action['term_id']) {
                        $deleteItems[] = [
                        'taxonomy' => $taxonomy,
                        'term_id' => $action['term_id']
                        ];
                    }
                }
            }
        }

        // Process sync items in bulk
        if (!empty($syncItems)) {
            $this->processBulkUpsertItems($syncItems);
        }

        // Process delete items in bulk
        if (!empty($deleteItems)) {
            $this->processBulkDeleteItems($deleteItems);
        }
    }

    /**
     * Get complete taxonomy data including all terms and hierarchy
     *
     * @param string $taxonomy Taxonomy slug
     * @return array Taxonomy data
     */
    private function getTaxonomyData(string $taxonomy): array
    {
        // Check cache first
        $cacheKey = "taxonomy_data_{$taxonomy}";
        $cached = $this->cache->get($cacheKey, 'taxonomies');
        if ($cached !== false) {
            return $cached;
        }

        $taxonomyObject = get_taxonomy($taxonomy);
        if (!$taxonomyObject) {
            return [];
        }

        $terms = get_terms([
        'taxonomy' => $taxonomy,
        'hide_empty' => false,
        'orderby' => 'term_order',
        'order' => 'ASC',
        'number' => 0  // Get ALL terms, no limit
        ]);

        if (is_wp_error($terms)) {
            return [];
        }

        $formattedTerms = [];
        foreach ($terms as $term) {
            $formattedTerms[] = $this->formatTermData($term);
        }

        $data = [
        'name' => $taxonomyObject->name,
        'label' => $taxonomyObject->label,
        'hierarchical' => $taxonomyObject->hierarchical,
        'public' => $taxonomyObject->public,
        'terms' => $formattedTerms,
        'last_updated' => current_time('mysql')
        ];

        // Cache for configured TTL
        $this->cache->set($cacheKey, $data, 'taxonomies', (int) self::CACHE_TTL_TAXONOMY_DATA);

        return $data;
    }

    /**
     * Format term data for cloud sync
     *
     * @param \WP_Term $term Term object
     * @return array Formatted term data
     */
    private function formatTermData(\WP_Term $term): array
    {
        // Clean term name and description using StringUtils
        $cleanName = StringUtils::cleanText($term->name);
        $cleanDescription = StringUtils::cleanText($term->description);

        $data = [
        'id' => $term->term_id,
        'name' => $cleanName,
        'slug' => $term->slug,
        'description' => $cleanDescription,
        'count' => $term->count,
        'parent' => $term->parent,
        'source' => 'wordpress'
        ];

        // Add term meta
        $meta = get_term_meta($term->term_id);
        if (!empty($meta)) {
            $data['meta'] = [];
            foreach ($meta as $key => $values) {
                $data['meta'][$key] = count($values) === 1 ? $values[0] : $values;
            }
        }

        return $data;
    }

    /**
     * Handle scheduled taxonomy sync (runs daily)
     *
     * @return void
     */
    public function handleScheduledTaxonomySync(): void
    {
        $this->logger->info('Starting scheduled full taxonomy sync');

        try {
            // Check if we have a valid API key
            if (!$this->aiClientManager->hasApiKey()) {
                $this->logger->info('Scheduled taxonomy sync skipped: no API key configured');
                return;
            }

            $batchItems = [];

            // Get all registered taxonomies
            $allTaxonomies = get_taxonomies([], 'names');

            // Prepare batch items for all eligible taxonomies
            $eligibleTaxonomies = [];
            $skippedTaxonomies = [];

            foreach ($allTaxonomies as $taxonomy) {
                if ($this->shouldSyncTaxonomy($taxonomy)) {
                    $eligibleTaxonomies[] = $taxonomy;
                    $taxonomyData = $this->getTaxonomyData($taxonomy);

                    if (!empty($taxonomyData['terms'])) {
                        $batchItems[] = [
                        'action' => 'taxonomy_full_sync',
                        'data' => [
                            'taxonomy' => $taxonomy,
                            'hierarchical' => $taxonomyData['hierarchical'],
                            'terms' => $taxonomyData['terms'],
                            'name' => $taxonomyData['name'],
                            'label' => $taxonomyData['label'],
                            'public' => $taxonomyData['public'] ?? false
                        ]
                        ];

                        // Cleanup stale terms for this taxonomy after processing
                        $validTermIds = array_column($taxonomyData['terms'], 'id');
                        $this->performTaxonomyCleanup($taxonomy, $validTermIds);
                    } else {
                        $this->logger->info('Taxonomy has no terms to sync', [
                            'taxonomy' => $taxonomy
                        ]);
                    }
                } else {
                    $skippedTaxonomies[] = $taxonomy;
                }
            }

            // Process taxonomies in bulk
            if (!empty($batchItems)) {
                $this->processBulkUpsertItems($batchItems);

                $this->logger->info('Processed scheduled full taxonomy sync', [
                    'taxonomies_count' => count($batchItems),
                ]);
            } else {
                $this->logger->info('No taxonomies to sync in scheduled full sync');
            }
        } catch (\Exception $e) {
            $this->logger->error('Scheduled taxonomy sync failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Handle manual taxonomy sync via AJAX
     *
     * @return void
     */
    public function handleManualTaxonomySync(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        check_ajax_referer('intucart_status_nonce', 'nonce');

        $taxonomy = sanitize_text_field($_POST['taxonomy'] ?? '');

        // If taxonomy parameter is empty, perform a full sync for all eligible taxonomies
        if (empty($taxonomy)) {
            try {
                $eligibleTaxonomies = self::getSyncedTaxonomies();

                if (empty($eligibleTaxonomies)) {
                    wp_send_json_success([
                        'message' => 'No eligible public/queryable taxonomies found to sync',
                        'synced' => [],
                        'skipped' => [],
                    ]);
                    return;
                }

                $synced = [];
                $skipped = [];

                foreach ($eligibleTaxonomies as $tx) {
                    // Extra safety: validate and re-check eligibility
                    if (!$this->validateTaxonomySlug($tx) || !$this->shouldSyncTaxonomy($tx)) {
                        $skipped[] = [ 'taxonomy' => $tx, 'reason' => 'invalid_or_not_syncable' ];
                        continue;
                    }

                    $data = $this->getTaxonomyData($tx);

                    if (empty($data['terms'])) {
                        // Still perform cleanup when no terms
                        $this->performTaxonomyCleanup($tx, []);
                        // Update timestamp even for empty taxonomies
                        update_option("intucart_last_{$tx}_sync", time());
                        $synced[] = [ 'taxonomy' => $tx, 'terms_synced' => 0, 'cleaned' => true ];
                        continue;
                    }

                    // Sync and cleanup
                    $this->syncTaxonomyToCloud($tx, $data);
                    $validTermIds = array_column($data['terms'], 'id');
                    $this->performTaxonomyCleanup($tx, $validTermIds);
                    $synced[] = [ 'taxonomy' => $tx, 'terms_synced' => count($validTermIds), 'cleaned' => true ];
                }

                // Update global taxonomy sync time for status tracking
                update_option('intucart_last_taxonomy_sync', time());

                // Fire action for other components that need to react to successful sync
                do_action(Constants::TAXONOMY_SYNC_COMPLETED_ACTION);

                wp_send_json_success([
                    'message' => 'All eligible taxonomies synced',
                    'synced' => $synced,
                    'skipped' => $skipped,
                ]);
            } catch (\Exception $e) {
                $this->logger->error('Manual full taxonomy sync failed', [
                    'error' => $e->getMessage(),
                ]);
                wp_send_json_error(__('Full taxonomy sync failed. Check logs for details.', 'intufind'));
            }
            return;
        }

        // Single taxonomy path (existing behavior)
        if (!$this->validateTaxonomySlug($taxonomy)) {
            wp_send_json_error(__('Invalid taxonomy slug format', 'intufind'));
            return;
        }

        if (!$this->shouldSyncTaxonomy($taxonomy)) {
            wp_send_json_error(__('Taxonomy not configured for sync', 'intufind'));
            return;
        }

        try {
            $taxonomyData = $this->getTaxonomyData($taxonomy);
            if (empty($taxonomyData['terms'])) {
                // Even if no terms to sync, run cleanup for this taxonomy
                $this->performTaxonomyCleanup($taxonomy, []);
                
                // Update the timestamp even for empty taxonomies so status shows correct sync time
                update_option("intucart_last_{$taxonomy}_sync", time());

                // Fire action for other components that need to react to successful sync
                do_action(Constants::TAXONOMY_SYNC_COMPLETED_ACTION);

                wp_send_json_success([
                    'message' => sprintf('Taxonomy "%s" has no terms to sync', $taxonomy)
                ]);
                return;
            }

            $this->syncTaxonomyToCloud($taxonomy, $taxonomyData);

            // Cleanup stale terms for this taxonomy after processing
            $validTermIds = array_column($taxonomyData['terms'], 'id');
            $this->performTaxonomyCleanup($taxonomy, $validTermIds);

            // Update global taxonomy sync time for status tracking
            update_option('intucart_last_taxonomy_sync', time());

            // Fire action for other components that need to react to successful sync
            do_action('intucart_taxonomy_sync_completed');

            wp_send_json_success([
                'message' => sprintf('Taxonomy "%s" sync completed successfully', $taxonomy)
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Manual taxonomy sync failed', [
                'taxonomy' => $taxonomy,
                'error' => $e->getMessage()
            ]);

            wp_send_json_error(__('Taxonomy sync failed. Check logs for details.', 'intufind'));
        }
    }

    /**
     * Check if taxonomy should be synced
     *
     * @param string $taxonomy Taxonomy slug
     * @return bool
     */
    private function shouldSyncTaxonomy(string $taxonomy): bool
    {
        // Exclude system/internal taxonomies
        if (in_array($taxonomy, self::EXCLUDED_TAXONOMIES, true)) {
            return false;
        }

        // Get taxonomy object to check if it's public or queryable
        $taxonomyObj = get_taxonomy($taxonomy);
        if (!$taxonomyObj) {
            return false;
        }

        // Check if taxonomy is enabled in user preferences
        $userPreferences = get_option('intucart_syncable_taxonomies', []);
        if (!empty($userPreferences) && (!isset($userPreferences[$taxonomy]) || $userPreferences[$taxonomy] !== true)) {
            return false;
        }

        // Only sync public taxonomies or those that are queryable
        // This includes all WooCommerce taxonomies, WordPress categories/tags, custom taxonomies, etc.
        return $taxonomyObj->public || $taxonomyObj->publicly_queryable || $taxonomyObj->show_ui;
    }

    /**
     * Invalidate taxonomy cache
     *
     * @param string $taxonomy Taxonomy slug
     * @return void
     */
    private function invalidateTaxonomyCache(string $taxonomy): void
    {
        $cacheKey = "taxonomy_data_{$taxonomy}";
        $this->cache->delete($cacheKey, 'taxonomies');
    }

    /**
     * Process bulk upsert items by sending them to cloud in batches
     *
     * @param array $upsertItems Array of upsert items
     * @return void
     */
    private function processBulkUpsertItems(array $upsertItems): void
    {
        try {
            // Group items by action type
            $fullSyncItems = [];
            $updateItems = [];

            foreach ($upsertItems as $item) {
                if ($item['action'] === 'taxonomy_full_sync') {
                    $fullSyncItems[] = $item;
                } else {
                    $updateItems[] = $item;
                }
            }

            // Process full sync items individually (they're already complete taxonomies)
            foreach ($fullSyncItems as $item) {
                $taxonomy = $item['data']['taxonomy'];
                $this->syncTaxonomyToCloud($taxonomy, $item['data']);
            }

            // Process update items in bulk if we have multiple
            if (!empty($updateItems)) {
                if (count($updateItems) === 1) {
                    // Single update - use existing individual method
                    $item = $updateItems[0];
                    $fullTaxonomyData = $this->getTaxonomyData($item['data']['taxonomy']);
                    $this->syncSingleTaxonomyTermToCloud(
                        $item['data']['taxonomy'],
                        $item['data']['term'],
                        $fullTaxonomyData
                    );
                } else {
                    // Multiple updates - use new bulk upsert
                    $this->syncBulkTaxonomyTermsToCloud($updateItems);
                }
            }

            $this->logger->info('Processed bulk taxonomy upsert items', [
                'items_count' => count($upsertItems),
                'full_sync_count' => count($fullSyncItems),
                'update_count' => count($updateItems)
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Bulk taxonomy upsert failed', [
                'error' => $e->getMessage(),
                'items_count' => count($upsertItems),
            ]);
            throw $e;
        }
    }

    /**
     * Process bulk delete items by sending them to cloud in batches
     *
     * @param array $deleteItems Array of delete items
     * @return void
     */
    private function processBulkDeleteItems(array $deleteItems): void
    {
        try {
            // Group deletes by taxonomy for efficient processing
            $taxonomyGroups = [];
            foreach ($deleteItems as $item) {
                $taxonomy = $item['taxonomy'];
                if (!isset($taxonomyGroups[$taxonomy])) {
                    $taxonomyGroups[$taxonomy] = [];
                }
                $taxonomyGroups[$taxonomy][] = $item['term_id'];
            }

            // Send bulk deletes for each taxonomy
            foreach ($taxonomyGroups as $taxonomy => $termIds) {
                if (count($termIds) === 1) {
                    // Single delete - use existing method
                    $this->deleteTaxonomyTermFromCloud($taxonomy, $termIds[0]);
                } else {
                    // Bulk delete - send all at once
                    $this->deleteBulkTaxonomyTermsFromCloud($taxonomy, $termIds);
                }
            }

            $this->logger->info('Processed bulk taxonomy delete items', [
                'items_count' => count($deleteItems),
                'taxonomies_count' => count($taxonomyGroups)
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Bulk taxonomy delete failed', [
                'error' => $e->getMessage(),
                'items_count' => count($deleteItems),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Validate term data structure
     *
     * @param array $termData Term data to validate
     * @return bool True if valid, false otherwise
     */
    private function validateTermData(array $termData): bool
    {
        // Required fields
        $requiredFields = ['id', 'name', 'slug'];

        foreach ($requiredFields as $field) {
            if (!isset($termData[$field]) || empty($termData[$field])) {
                $this->logger->warning('Invalid term data: missing required field', [
                    'field' => $field,
                    'term_data' => $termData
                ]);
                return false;
            }
        }

        // Validate data types
        if (!is_numeric($termData['id']) || $termData['id'] <= 0) {
            $this->logger->warning('Invalid term data: invalid ID', [
                'id' => $termData['id']
            ]);
            return false;
        }

        if (!is_string($termData['name']) || !is_string($termData['slug'])) {
            $this->logger->warning('Invalid term data: name and slug must be strings', [
                'name_type' => gettype($termData['name']),
                'slug_type' => gettype($termData['slug'])
            ]);
            return false;
        }

        // Validate parent if present
        if (isset($termData['parent']) && $termData['parent'] !== 0 && !is_numeric($termData['parent'])) {
            $this->logger->warning('Invalid term data: invalid parent ID', [
                'parent' => $termData['parent']
            ]);
            return false;
        }

        return true;
    }

    /**
     * Validate taxonomy data structure
     *
     * @param array $taxonomyData Taxonomy data to validate
     * @return bool True if valid, false otherwise
     */
    private function validateTaxonomyData(array $taxonomyData): bool
    {
        // Required fields
        $requiredFields = ['name', 'label', 'terms'];

        foreach ($requiredFields as $field) {
            if (!isset($taxonomyData[$field])) {
                $this->logger->warning('Invalid taxonomy data: missing required field', [
                    'field' => $field,
                    'taxonomy_data_keys' => array_keys($taxonomyData)
                ]);
                return false;
            }
        }

        // Validate terms array
        if (!is_array($taxonomyData['terms'])) {
            $this->logger->warning('Invalid taxonomy data: terms must be an array', [
                'terms_type' => gettype($taxonomyData['terms'])
            ]);
            return false;
        }

        // Validate each term
        foreach ($taxonomyData['terms'] as $index => $term) {
            if (!is_array($term) || !$this->validateTermData($term)) {
                $this->logger->warning('Invalid taxonomy data: invalid term at index', [
                    'index' => $index,
                    'term' => $term
                ]);
                return false;
            }
        }

        return true;
    }

    /**
     * Validate taxonomy slug
     *
     * @param string $taxonomy Taxonomy slug to validate
     * @return bool True if valid, false otherwise
     */
    private function validateTaxonomySlug(string $taxonomy): bool
    {
        // Check if empty
        if (empty($taxonomy)) {
            $this->logger->warning('Invalid taxonomy: empty taxonomy slug');
            return false;
        }

        // Check for valid characters (WordPress taxonomy naming rules)
        if (!preg_match('/^[a-z0-9_-]+$/', $taxonomy)) {
            $this->logger->warning('Invalid taxonomy: contains invalid characters', [
                'taxonomy' => $taxonomy
            ]);
            return false;
        }

        // Check length (WordPress limit is 32 characters)
        if (strlen($taxonomy) > 32) {
            $this->logger->warning('Invalid taxonomy: slug too long', [
                'taxonomy' => $taxonomy,
                'length' => strlen($taxonomy)
            ]);
            return false;
        }

        return true;
    }

    /**
     * Validate sync payload before sending to cloud
     *
     * @param array $payload Payload to validate
     * @return bool True if valid, false otherwise
     */
    private function validateSyncPayload(array $payload): bool
    {
        // Required fields
        $requiredFields = ['action', 'timestamp', 'data'];

        foreach ($requiredFields as $field) {
            if (!isset($payload[$field])) {
                $this->logger->error('Invalid sync payload: missing required field', [
                    'field' => $field,
                    'payload_keys' => array_keys($payload)
                ]);
                return false;
            }
        }

        // Validate action
        $validActions = [
            'taxonomy_full_sync',
            'taxonomy_update',
            'taxonomy_delete',
            'taxonomy_bulk_delete',
            'taxonomy_bulk_upsert'
        ];

        if (!in_array($payload['action'], $validActions, true)) {
            $this->logger->error('Invalid sync payload: unknown action', [
                'action' => $payload['action'],
                'valid_actions' => $validActions
            ]);
            return false;
        }

        // Validate timestamp
        if (!is_numeric($payload['timestamp']) || $payload['timestamp'] <= 0) {
            $this->logger->error('Invalid sync payload: invalid timestamp', [
                'timestamp' => $payload['timestamp']
            ]);
            return false;
        }

        return true;
    }

    /**
     * Build a standardized payload for cloud sync operations
     *
     * @param string $action Action type
     * @param array  $data   Data payload
     * @return array Formatted payload
     */
    private function buildSyncPayload(string $action, array $data): array
    {
        return [
            'action' => $action,
            'timestamp' => time(),
            'data' => $data
        ];
    }

    /**
     * Sync taxonomy data to cloud
     *
     * @param string $taxonomy Taxonomy slug
     * @param array  $taxonomyData Taxonomy data
     * @return void
     */
    private function syncTaxonomyToCloud(string $taxonomy, array $taxonomyData): void
    {
        // Validate inputs
        if (!$this->validateTaxonomySlug($taxonomy)) {
            throw new \InvalidArgumentException('Invalid taxonomy slug: ' . $taxonomy);
        }

        if (!$this->validateTaxonomyData($taxonomyData)) {
            throw new \InvalidArgumentException('Invalid taxonomy data for: ' . $taxonomy);
        }

        $payload = $this->buildSyncPayload('taxonomy_full_sync', [
            'taxonomy' => $taxonomy,
            'hierarchical' => $taxonomyData['hierarchical'],
            'terms' => $taxonomyData['terms'],
            'name' => $taxonomyData['name'],
            'label' => $taxonomyData['label'],
            'public' => $taxonomyData['public'] ?? false
        ]);

        // Validate payload before sending
        if (!$this->validateSyncPayload($payload)) {
            throw new \InvalidArgumentException('Invalid sync payload for taxonomy: ' . $taxonomy);
        }

        // Use AI SDK TaxonomyService for sync operations
        // SDK throws exceptions on failure, so if we get here it succeeded
        $response = $this->syncTaxonomy($payload);

        // Log partial failures if any items failed
        if (!empty($response['failed_items'])) {
            $this->logger->warning('Some taxonomy terms failed to sync', [
                'taxonomy' => $taxonomy,
                'failed_count' => count($response['failed_items']),
                'failed_items' => array_slice($response['failed_items'], 0, 5),
            ]);
        }

        $this->logger->info('Taxonomy synced to cloud successfully', [
            'taxonomy' => $taxonomy,
            'terms_count' => count($taxonomyData['terms'] ?? []),
            'response_data' => $response
        ]);

        // Update last sync timestamp for this taxonomy
        update_option("intucart_last_{$taxonomy}_sync", time());
    }

    /**
     * Delete taxonomy term from cloud
     *
     * @param string $taxonomy Taxonomy slug
     * @param int    $termId   Term ID
     * @return void
     */
    private function deleteTaxonomyTermFromCloud(string $taxonomy, int $termId): void
    {
        $payload = $this->buildSyncPayload('taxonomy_delete', [
            'taxonomy' => $taxonomy,
            'term_id' => $termId
        ]);

        // SDK throws exceptions on failure, so if we get here it succeeded
        $this->syncTaxonomy($payload);

        $this->logger->info('Taxonomy term deleted from cloud successfully', [
            'taxonomy' => $taxonomy,
            'term_id' => $termId
        ]);
    }

    /**
     * Delete multiple taxonomy terms from cloud in bulk
     *
     * @param string $taxonomy Taxonomy slug
     * @param array  $termIds  Array of term IDs
     * @return void
     */
    private function deleteBulkTaxonomyTermsFromCloud(string $taxonomy, array $termIds): void
    {
        if (empty($termIds)) {
            $this->logger->warning('No term IDs to delete in bulk operation', [
                'taxonomy' => $taxonomy
            ]);
            return;
        }

        // Chunk term IDs to respect cloud API limits
        $termIdChunks = array_chunk($termIds, self::MAX_BULK_DELETE_SIZE);
        $totalChunks = count($termIdChunks);
        $successfulChunks = 0;
        $totalDeleted = 0;

        $this->logger->info('Starting chunked bulk taxonomy delete', [
            'taxonomy' => $taxonomy,
            'total_term_ids' => count($termIds),
            'total_chunks' => $totalChunks,
            'chunk_size' => self::MAX_BULK_DELETE_SIZE
        ]);

        foreach ($termIdChunks as $chunkIndex => $chunk) {
            try {
                $payload = $this->buildSyncPayload('taxonomy_bulk_delete', [
                    'taxonomy' => $taxonomy,
                    'term_ids' => $chunk
                ]);

                // SDK throws exceptions on failure, so if we get here it succeeded
                $this->syncTaxonomy($payload);

                $successfulChunks++;
                $totalDeleted += count($chunk);

                $this->logger->info('Bulk taxonomy delete chunk completed successfully', [
                    'taxonomy' => $taxonomy,
                    'chunk' => $chunkIndex + 1,
                    'total_chunks' => $totalChunks,
                    'chunk_size' => count($chunk),
                    'total_deleted' => $totalDeleted
                ]);
            } catch (\Exception $e) {
                $this->logger->error('Bulk taxonomy delete chunk failed', [
                    'taxonomy' => $taxonomy,
                    'chunk' => $chunkIndex + 1,
                    'total_chunks' => $totalChunks,
                    'chunk_size' => count($chunk),
                    'error' => $e->getMessage()
                ]);

                // Continue with other chunks instead of failing completely
                continue;
            }
        }

        if ($successfulChunks === 0) {
            throw new \Exception('All bulk taxonomy delete chunks failed for taxonomy: ' . $taxonomy);
        }

        // If we get here, at least some responses were successful
        $this->logger->info('Chunked bulk taxonomy terms deleted from cloud', [
            'taxonomy' => $taxonomy,
            'total_items' => count($termIds),
            'terms_deleted' => $totalDeleted,
            'successful_chunks' => $successfulChunks,
            'total_chunks' => $totalChunks,
            'success_rate' => round(($successfulChunks / $totalChunks) * 100, 2) . '%'
        ]);
    }

    /**
     * Sync single taxonomy term to cloud
     *
     * @param string $taxonomy Taxonomy slug
     * @param array  $termData Term data
     * @param array  $fullTaxonomyData Full taxonomy data for hierarchy
     * @return void
     */
    private function syncSingleTaxonomyTermToCloud(string $taxonomy, array $termData, array $fullTaxonomyData): void
    {
        // Validate inputs
        if (!$this->validateTaxonomySlug($taxonomy)) {
            throw new \InvalidArgumentException('Invalid taxonomy slug: ' . $taxonomy);
        }

        if (!$this->validateTermData($termData)) {
            throw new \InvalidArgumentException('Invalid term data for taxonomy: ' . $taxonomy);
        }

        if (!$this->validateTaxonomyData($fullTaxonomyData)) {
            throw new \InvalidArgumentException('Invalid full taxonomy data for: ' . $taxonomy);
        }

        $payload = $this->buildSyncPayload('taxonomy_update', [
            'taxonomy' => $taxonomy,
            'term' => $termData,
            'hierarchical' => $fullTaxonomyData['hierarchical'],
            'terms' => $fullTaxonomyData['terms'],
            'name' => $fullTaxonomyData['name'],
            'label' => $fullTaxonomyData['label'],
            'public' => $fullTaxonomyData['public'] ?? false
        ]);

        // Validate payload before sending
        if (!$this->validateSyncPayload($payload)) {
            throw new \InvalidArgumentException('Invalid sync payload for term: ' . $termData['id']);
        }

        // SDK throws exceptions on failure, so if we get here it succeeded
        $this->syncTaxonomy($payload);

        $this->logger->info('Single taxonomy term synced to cloud successfully', [
            'taxonomy' => $taxonomy,
            'term_id' => $termData['id']
        ]);
    }

    /**
     * Sync multiple taxonomy terms to cloud in bulk
     *
     * @param array $updateItems Array of update items
     * @return void
     */
    private function syncBulkTaxonomyTermsToCloud(array $updateItems): void
    {
        // Prepare items for bulk upsert
        $bulkItems = [];
        $validItems = [];

        foreach ($updateItems as $item) {
            try {
                $bulkItems[] = [
                    'taxonomy' => $item['data']['taxonomy'],
                    'hierarchical' => $item['data']['hierarchical'],
                    'terms' => [$item['data']['term']],
                    'name' => $item['data']['name'],
                    'label' => $item['data']['label'],
                    'public' => $item['data']['public']
                ];
                $validItems[] = $item;
            } catch (\Exception $e) {
                $this->logger->error('Failed to prepare taxonomy term for bulk sync', [
                    'error' => $e->getMessage()
                ]);
            }
        }

        if (empty($bulkItems)) {
            $this->logger->warning('No valid taxonomy terms to sync in bulk operation');
            return;
        }

        // Chunk items to respect cloud API limits
        $itemChunks = array_chunk($bulkItems, self::MAX_BULK_UPSERT_SIZE);
        $totalChunks = count($itemChunks);
        $successfulChunks = 0;
        $totalSynced = 0;

        $this->logger->info('Starting chunked bulk taxonomy terms sync', [
            'total_items' => count($bulkItems),
            'total_chunks' => $totalChunks,
            'chunk_size' => self::MAX_BULK_UPSERT_SIZE
        ]);

        foreach ($itemChunks as $chunkIndex => $chunk) {
            try {
                $payload = $this->buildSyncPayload('taxonomy_bulk_upsert', [
                    'items' => $chunk
                ]);

                // SDK throws exceptions on failure, so if we get here it succeeded
                $this->syncTaxonomy($payload);

                $successfulChunks++;
                $totalSynced += count($chunk);

                $this->logger->info('Bulk taxonomy sync chunk completed successfully', [
                    'chunk' => $chunkIndex + 1,
                    'total_chunks' => $totalChunks,
                    'chunk_size' => count($chunk),
                    'total_synced' => $totalSynced
                ]);
            } catch (\Exception $e) {
                $this->logger->error('Bulk taxonomy sync chunk failed', [
                    'chunk' => $chunkIndex + 1,
                    'total_chunks' => $totalChunks,
                    'chunk_size' => count($chunk),
                    'error' => $e->getMessage()
                ]);

                // Continue with other chunks instead of failing completely
                continue;
            }
        }

        if ($successfulChunks === 0) {
            throw new \Exception('All bulk taxonomy sync chunks failed');
        }

        // If we get here, at least some responses were successful
        $this->logger->info('Chunked bulk taxonomy terms synced to cloud', [
            'total_items' => count($updateItems),
            'valid_items' => count($validItems),
            'items_synced' => $totalSynced,
            'successful_chunks' => $successfulChunks,
            'total_chunks' => $totalChunks,
            'success_rate' => round(($successfulChunks / $totalChunks) * 100, 2) . '%'
        ]);
    }

    /**
     * Get all available taxonomies (both enabled and disabled)
     * Optionally filtered by syncable post types
     *
     * @param array $syncablePostTypes Optional array of syncable post types to filter by
     * @return array Taxonomy objects keyed by taxonomy name
     */
    public function getAllAvailableTaxonomies(array $syncablePostTypes = []): array
    {
        $allTaxonomies = get_taxonomies([], 'objects');
        $availableTaxonomies = [];

        foreach ($allTaxonomies as $taxonomy) {
            // Skip excluded taxonomies
            if (in_array($taxonomy->name, self::EXCLUDED_TAXONOMIES, true)) {
                continue;
            }

            // Only include public taxonomies or those that are queryable/show_ui
            if (!($taxonomy->public || $taxonomy->publicly_queryable || $taxonomy->show_ui)) {
                continue;
            }

            // If syncable post types are provided, only include taxonomies tied to those post types
            if (!empty($syncablePostTypes)) {
                // Check if this taxonomy is tied to any of the syncable post types
                $taxonomyPostTypes = $taxonomy->object_type ?? [];
                
                // If no object types are specified, it's a universal taxonomy (like categories/tags)
                // Include it if 'post' is in syncable post types
                if (empty($taxonomyPostTypes) && in_array('post', $syncablePostTypes, true)) {
                    $availableTaxonomies[$taxonomy->name] = $taxonomy;
                    continue;
                }

                // Check for intersection between taxonomy post types and syncable post types
                $hasIntersection = !empty(array_intersect($taxonomyPostTypes, $syncablePostTypes));
                
                if ($hasIntersection) {
                    $availableTaxonomies[$taxonomy->name] = $taxonomy;
                }
            } else {
                // No filtering by post types - include all that pass other criteria
                $availableTaxonomies[$taxonomy->name] = $taxonomy;
            }
        }

        return $availableTaxonomies;
    }

    /**
     * Get all available taxonomy names
     * Optionally filtered by syncable post types
     *
     * @param array $syncablePostTypes Optional array of syncable post types to filter by
     * @return array Array of taxonomy names
     */
    public function getAllAvailableTaxonomyNames(array $syncablePostTypes = []): array
    {
        return array_keys($this->getAllAvailableTaxonomies($syncablePostTypes));
    }

    /**
     * Get syncable taxonomies based on user preferences
     *
     * @return array Array of taxonomy names
     */
    public function getSyncableTaxonomyNames(): array
    {
        // Get all available taxonomies
        $allTaxonomies = $this->getAllAvailableTaxonomyNames();

        // Get user preferences for which taxonomies to include
        $userPreferences = get_option('intucart_syncable_taxonomies', []);

        // Filter taxonomies based on user preferences
        $syncableTaxonomies = [];
        foreach ($allTaxonomies as $taxonomy) {
            if (isset($userPreferences[$taxonomy]) && $userPreferences[$taxonomy] === true) {
                $syncableTaxonomies[] = $taxonomy;
            }
        }

        return $syncableTaxonomies;
    }

    /**
     * Get default taxonomy preferences
     *
     * @param array $taxonomyNames Array of taxonomy names
     * @return array Default preferences
     */
    public function getDefaultTaxonomyPreferences(array $taxonomyNames): array
    {
        $defaults = [];
        foreach ($taxonomyNames as $taxonomyName) {
            // Enable common taxonomies by default
            $isDefaultEnabled = in_array($taxonomyName, [
                // Core WordPress taxonomies
                'category', 'post_tag',
                // WooCommerce core taxonomies
                'product_cat', 'product_tag',
                // Common WooCommerce brand taxonomies (various plugins use these)
                'product_brand', 'pwb-brand', 'yith_product_brand', 'berocket_brand',
                // Common WooCommerce product attributes (pa_ prefix)
                'pa_color', 'pa_colour', 'pa_size', 'pa_brand'
            ], true);
            
            // Also enable any taxonomy that starts with 'pa_' (WooCommerce product attributes)
            // but only if it's a common attribute name
            if (!$isDefaultEnabled && strpos($taxonomyName, 'pa_') === 0) {
                $attributeName = substr($taxonomyName, 3); // Remove 'pa_' prefix
                $isDefaultEnabled = in_array($attributeName, [
                    'color', 'colour', 'size', 'brand', 'material', 'style'
                ], true);
            }
            
            $defaults[$taxonomyName] = $isDefaultEnabled;
        }
        return $defaults;
    }

    /**
     * Get synced taxonomies list (legacy method for backward compatibility)
     *
     * @return array
     */
    public static function getSyncedTaxonomies(): array
    {
        $allTaxonomies = get_taxonomies([], 'names');
        $syncedTaxonomies = [];

        foreach ($allTaxonomies as $taxonomy) {
            // Use a static instance to check if taxonomy should be synced
            $taxonomyObj = get_taxonomy($taxonomy);
            if (!$taxonomyObj) {
                continue;
            }

            // Exclude system/internal taxonomies
            if (in_array($taxonomy, self::EXCLUDED_TAXONOMIES, true)) {
                continue;
            }

            // Only include public taxonomies or those that are queryable
            if ($taxonomyObj->public || $taxonomyObj->publicly_queryable || $taxonomyObj->show_ui) {
                $syncedTaxonomies[] = $taxonomy;
            }
        }

        return $syncedTaxonomies;
    }

    /**
     * Perform taxonomy cleanup using cloud-based ID retrieval and batch processing
     *
     * @param string $taxonomy Taxonomy slug to clean up
     * @param array $validTermIds Array of valid WordPress term IDs
     * @return void
     */
    private function performTaxonomyCleanup(string $taxonomy, array $validTermIds): void
    {
        try {
            // Get all taxonomy term IDs from the cloud for this taxonomy
            $cloudTermIds = $this->getCloudTaxonomyIds($taxonomy);

            if (empty($cloudTermIds)) {
                return;
            }

            // Find IDs that exist in cloud but not in WordPress
            $validIdSet = array_flip(array_map('strval', $validTermIds));
            $staleIds = [];

            foreach ($cloudTermIds as $cloudId) {
                if (!isset($validIdSet[$cloudId])) {
                    $staleIds[] = $cloudId;
                }
            }

            if (empty($staleIds)) {
                return;
            }

            // Delete stale terms in batches
            $this->deleteStaleTaxonomyTerms($taxonomy, $staleIds);
        } catch (\Exception $e) {
            $this->logger->error('Taxonomy cleanup exception', [
                'taxonomy' => $taxonomy,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Get all taxonomy term IDs from the cloud for a specific taxonomy
     *
     * @param string $taxonomy Taxonomy slug to retrieve IDs for
     * @return array Array of taxonomy term IDs from the cloud
     */
    private function getCloudTaxonomyIds(string $taxonomy): array
    {
        $allIds = [];
        $offset = 0;
        $limit = 10000; // Use reasonable batch size
        $hasMore = true;

        while ($hasMore) {
            try {
                $response = $this->getTaxonomyIds([
                    'taxonomy_name' => $taxonomy
                ], $limit, $offset);

                // Check if response is valid (getIds returns data directly, not success wrapper)
                if (!is_array($response)) {
                    $this->logger->warning('Invalid response from taxonomy getIds API', [
                        'taxonomy' => $taxonomy,
                        'offset' => $offset,
                        'response_type' => gettype($response)
                    ]);
                    break;
                }

                $batchIds = $response['ids'] ?? [];
                $allIds = array_merge($allIds, $batchIds);

                $hasMore = $response['has_more'] ?? false;
                $offset += $limit;
            } catch (\Exception $e) {
                $this->logger->error('Exception while retrieving cloud taxonomy IDs', [
                    'taxonomy' => $taxonomy,
                    'offset' => $offset,
                    'error' => $e->getMessage()
                ]);
                break;
            }
        }

        return $allIds;
    }

    /**
     * Delete stale taxonomy terms in batches to respect OpenSearch terms query limits
     *
     * @param string $taxonomy Taxonomy slug
     * @param array $staleIds Array of stale taxonomy term IDs to delete
     * @return void
     */
    private function deleteStaleTaxonomyTerms(string $taxonomy, array $staleIds): void
    {
        if (empty($staleIds)) {
            return;
        }

        $batchSize = 5000; // Process in smaller batches for taxonomy terms
        $batches = array_chunk($staleIds, $batchSize);
        $totalBatches = count($batches);
        $processedCount = 0;

        foreach ($batches as $batchIndex => $batch) {
            try {
                // Build OpenSearch filters for this batch
                $filters = [
                    ['term' => ['taxonomy_name' => $taxonomy]],
                    ['bool' => [
                        'must_not' => [
                            ['terms' => ['external_id' => array_map('strval', $batch)]]
                        ]
                    ]]
                ];

                // Invert the logic: delete terms where external_id is NOT in the valid list
                // But we want to delete terms where external_id IS in the stale list
                $filters = [
                    ['term' => ['taxonomy_name' => $taxonomy]],
                    ['terms' => ['external_id' => array_map('strval', $batch)]]
                ];

                // SDK throws exceptions on failure, so if we get here it succeeded
                $response = $this->deleteByQuery($filters);
                $deletedCount = $response['deleted_count'] ?? 0;
                $processedCount += $deletedCount;
            } catch (\Exception $e) {
                $this->logger->error('Exception during taxonomy terms batch deletion', [
                    'taxonomy' => $taxonomy,
                    'batch' => $batchIndex + 1,
                    'error' => $e->getMessage()
                ]);
            }
        }
    }
}
