<?php

namespace Intucart\Services\Mcp;

use Intucart\Services\Logger;
use WP_REST_Request;

/**
 * MCP Authentication Handler
 * Handles authentication for MCP requests using existing Intucart auth system
 */
class McpAuthHandler
{
    private Logger $logger;
    private ?array $userContext = null;

    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
    }

    /**
     * Authenticate MCP request
     */
    public function authenticate(WP_REST_Request $request): bool
    {
        // First check for JWT authentication in headers (for cloud service requests)
        $jwtToken = $this->getJwtFromRequest($request);
        if ($jwtToken) {
            $jwtValid = $this->validateJwtToken($jwtToken);
            if ($jwtValid) {
                return true;
            }
            // If JWT is present but invalid, don't fall back to other methods
            $this->logger->warning('MCP request denied: Invalid JWT token provided');
            return false;
        }

        // Check for license key authentication
        $licenseKey = $this->getLicenseKeyFromRequest($request);
        if ($licenseKey) {
            return $this->validateLicenseKeyPayload($licenseKey);
        }

        // Fall back to WordPress user session authentication
        if (is_user_logged_in()) {
            return $this->checkWordPressCapabilities();
        }

        $this->logger->warning('MCP request denied: No valid authentication method found');
        return false;
    }

    /**
     * Get JWT token from request headers
     */
    private function getJwtFromRequest(WP_REST_Request $request): ?string
    {
        $authHeader = $request->get_header('Authorization');
        if ($authHeader && strpos($authHeader, 'Bearer ') === 0) {
            $token = substr($authHeader, 7);
            // Only log JWT presence at debug level to reduce noise
            $this->logger->info('JWT token found in Authorization header', [
                'token_length' => strlen($token),
                'token_prefix' => substr($token, 0, 20) . '...'
            ]);
            return $token;
        }

        // Only log missing JWT tokens when there's no auth header at all (actual issue)
        if (empty($authHeader)) {
            $this->logger->warning('No Authorization header found in MCP request');
        }
        return null;
    }

    /**
     * Get license key from request headers
     */
    private function getLicenseKeyFromRequest(WP_REST_Request $request): ?string
    {
        return $request->get_header('X-License-Key');
    }

    /**
     * Validate JWT token and extract user context
     */
    private function validateJwtToken(string $token): bool
    {
        try {
            // Try to decode and validate the token
            $parts = explode('.', $token);
            if (count($parts) !== 3) {
                $this->logger->warning('Invalid JWT format');
                return false;
            }

            // Validate header algorithm
            $header = json_decode(base64_decode(str_pad(strtr($parts[0], '-_', '+/'), strlen($parts[0]) % 4, '=', STR_PAD_RIGHT)), true);
            if (!$header || ($header['alg'] ?? '') !== 'HS256') {
                $this->logger->warning('Invalid JWT algorithm');
                return false;
            }

            $payload = json_decode(base64_decode(str_pad(strtr($parts[1], '-_', '+/'), strlen($parts[1]) % 4, '=', STR_PAD_RIGHT)), true);

            if (!$payload) {
                $this->logger->warning('Failed to decode JWT payload');
                return false;
            }

            // Check expiration
            if (isset($payload['exp']) && $payload['exp'] <= time()) {
                $this->logger->info('JWT token expired');
                return false;
            }

            // Check issued at time (prevent tokens from future)
            if (isset($payload['iat']) && $payload['iat'] > time() + 60) { // Allow 60s clock skew
                $this->logger->warning('JWT issued in future');
                return false;
            }

            // Verify signature using license key
            $licenseKey = get_option('intucart_license_key', '');
            if (!$this->verifyJwtSignature($token, $licenseKey)) {
                $this->logger->warning('JWT signature verification failed');
                return false;
            }

            // Store user context in instance with permissions included
            $this->userContext = [
                'id' => $payload['user_id'] ?? 0,
                'type' => $payload['user_type'] ?? 'guest',
                'email' => $payload['user_email'] ?? null,
                'roles' => $payload['user_roles'] ?? ['guest'],
                'permissions' => $payload['permissions'] ?? [],
            ];

            // Reduce successful auth logging to debug level to minimize noise
            $this->logger->info('JWT authentication successful', [
                'user_id' => $this->userContext['id'],
                'user_type' => $this->userContext['type']
            ]);

            return true;
        } catch (\Exception $e) {
            $this->logger->error('JWT validation failed', ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Verify JWT signature using license key
     */
    private function verifyJwtSignature(string $token, string $licenseKey): bool
    {
        $parts = explode('.', $token);
        if (count($parts) !== 3) {
            return false;
        }

        $header = $parts[0];
        $payload = $parts[1];
        $signature = $parts[2];

        $expectedSignature = hash_hmac('sha256', $header . '.' . $payload, $licenseKey, true);
        $expectedSignature = rtrim(strtr(base64_encode($expectedSignature), '+/', '-_'), '=');

        return hash_equals($signature, $expectedSignature);
    }

    /**
     * Validate license key
     */
    private function validateLicenseKeyPayload(string $licenseKey): bool
    {
        $storedLicenseKey = get_option('intucart_license_key', '');

        if (empty($storedLicenseKey)) {
            $this->logger->warning('No license key configured');
            return false;
        }

        if ($licenseKey !== $storedLicenseKey) {
            $this->logger->warning('Invalid license key provided');
            return false;
        }

        // Reduce successful auth logging to debug level to minimize noise
        $this->logger->info('License key authentication successful');
        return true;
    }

    /**
     * Check WordPress user capabilities
     */
    private function checkWordPressCapabilities(): bool
    {
        $currentUserId = get_current_user_id();

                // Allow any logged-in user - permissions will be checked at the tool level
        if ($currentUserId > 0) {
            $user = get_user_by('id', $currentUserId);
            $userRoles = $user ? $user->roles : ['guest'];

            // Store user context in instance for WordPress fallback authentication with permissions included
            $this->userContext = [
                'id' => $currentUserId,
                'type' => 'authenticated',
                'email' => $user ? $user->user_email : null,
                'roles' => $userRoles,
                'login' => $user ? $user->user_login : null,
                'permissions' => $this->calculateUserPermissions($currentUserId),
            ];

            // Reduce successful auth logging to debug level to minimize noise
            $this->logger->info('WordPress user authentication successful', [
                'user_id' => $currentUserId,
                'user_roles' => $userRoles
            ]);

            return true;
        }

        $this->logger->warning('User not logged in for MCP access');
        return false;
    }

    /**
     * Calculate user permissions based on WordPress user ID
     */
    private function calculateUserPermissions(int $userId): array
    {
        // Guest users - very limited permissions
        if ($userId === 0) {
            return [
                'can_view_own_orders' => false,
                'can_view_any_orders' => false,
                'can_update_orders' => false,
            ];
        }

        // Check user capabilities for logged-in users
        return [
            'can_view_own_orders' => true, // All logged-in users can view their own orders
            'can_view_any_orders' => current_user_can('manage_woocommerce') || current_user_can('view_woocommerce_reports'),
            'can_update_orders' => current_user_can('edit_shop_orders'),
        ];
    }

    /**
     * Get current user context for MCP operations
     */
    public function getCurrentUserContext(): array
    {
        if ($this->userContext !== null) {
            return $this->userContext;
        }

        // Fallback to current WordPress user
        $currentUserId = get_current_user_id();
        if ($currentUserId === 0) {
            return [
                'id' => 0,
                'type' => 'guest',
                'email' => null,
                'roles' => ['guest'],
                'permissions' => $this->calculateUserPermissions(0),
            ];
        }

        $user = get_user_by('id', $currentUserId);
        if (!$user) {
            return [
                'id' => 0,
                'type' => 'guest',
                'email' => null,
                'roles' => ['guest'],
                'permissions' => $this->calculateUserPermissions(0),
            ];
        }

        return [
            'id' => $currentUserId,
            'type' => 'authenticated',
            'email' => $user->user_email,
            'roles' => $user->roles,
            'login' => $user->user_login,
            'permissions' => $this->calculateUserPermissions($currentUserId),
        ];
    }
}
