<?php

namespace Intucart\Services\Mcp;

use Intucart\Services\Logger;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Intucart MCP Server
 * Custom MCP server designed for multi-agent, multi-tenant scenarios
 * 
 * Uses domain-based routing (/mcp/domain/{domain}) aligned with cloud architecture.
 * Available domains are derived from registered tool managers.
 */
class McpServer
{
    private Logger $logger;
    private McpToolsManager $toolsManager;
    private McpAuthHandler $authHandler;
    private static bool $initialized = false;

    public function __construct(Logger $logger, McpAuthHandler $authHandler, McpToolsManager $toolsManager)
    {
        $this->logger = $logger;
        $this->authHandler = $authHandler;
        $this->toolsManager = $toolsManager;

        $this->initialize();
    }

    /**
     * Initialize the MCP server
     */
    private function initialize(): void
    {
        if (self::$initialized) {
            return;
        }

        add_action('rest_api_init', [$this, 'registerRoutes']);

        // Initialize tools manager
        $this->toolsManager->initialize();

        self::$initialized = true;
    }

    /**
     * Register REST API routes
     */
    public function registerRoutes(): void
    {
        // Domain-based endpoints (e.g., /mcp/domain/orders)
        register_rest_route('intufind/v1', '/mcp/domain/(?P<domain>[a-zA-Z0-9_-]+)', [
            'methods' => WP_REST_Server::ALLMETHODS,
            'callback' => [$this, 'handleDomainRequest'],
            'permission_callback' => [$this, 'authenticate'],
        ]);

        // Generic endpoint (lists available domains)
        register_rest_route('intufind/v1', '/mcp', [
            'methods' => WP_REST_Server::ALLMETHODS,
            'callback' => [$this, 'handleGenericRequest'],
            'permission_callback' => [$this, 'authenticate'],
        ]);

        $this->logger->info('Intucart MCP routes registered', [
            'domains' => $this->toolsManager->getAvailableDomains()
        ]);
    }

    /**
     * Authenticate requests using existing auth system
     */
    public function authenticate(WP_REST_Request $request): bool
    {
        return $this->authHandler->authenticate($request);
    }

    /**
     * Handle MCP requests for specific domains
     */
    public function handleDomainRequest(WP_REST_Request $request): WP_REST_Response
    {
        $domain = $request->get_param('domain');
        
        // Validate domain exists
        if (!$this->toolsManager->hasDomain($domain)) {
            return $this->createErrorResponse(0, -32602, "Domain '$domain' not found");
        }
        
        return $this->processRequest($request, $domain);
    }

    /**
     * Handle generic MCP requests (returns available domains info)
     */
    public function handleGenericRequest(WP_REST_Request $request): WP_REST_Response
    {
        return $this->processRequest($request, null);
    }

    /**
     * Process MCP request
     * 
     * @param WP_REST_Request $request The REST request
     * @param string|null $domain Domain for domain-based routing (null for generic requests)
     */
    private function processRequest(WP_REST_Request $request, ?string $domain): WP_REST_Response
    {
        // Handle preflight requests
        if ($request->get_method() === 'OPTIONS') {
            return new WP_REST_Response(null, 204);
        }

        if ($request->get_method() !== 'POST') {
            return $this->createErrorResponse(0, -32601, 'Method not allowed');
        }

        try {
            // Validate content type and accept headers
            $validation = $this->validateHeaders($request);
            if ($validation !== true) {
                return $validation;
            }

            // Parse JSON-RPC request
            $body = $request->get_json_params();
            if ($body === null) {
                return $this->createErrorResponse(0, -32700, 'Parse error - Invalid JSON');
            }

            // Handle batch requests or single request
            $messages = is_array($body) && isset($body[0]) ? $body : [$body];
            $results = [];

            foreach ($messages as $message) {
                $validation = $this->validateJsonRpcMessage($message);
                if ($validation !== true) {
                    $results[] = $validation;
                    continue;
                }

                $results[] = $this->processMessage($message, $domain);
            }

            // Return single result or batch
            $responseBody = count($results) === 1 ? $results[0] : $results;

            return new WP_REST_Response($responseBody, 200, [
                'Content-Type' => 'application/json',
                'Access-Control-Allow-Origin' => '*',
                'Access-Control-Allow-Methods' => 'OPTIONS, POST',
            ]);
        } catch (\Throwable $exception) {
            $this->logger->error('MCP request processing failed', [
                'domain' => $domain,
                'error' => $exception->getMessage(),
                'trace' => $exception->getTraceAsString()
            ]);

            return $this->createErrorResponse(0, -32603, 'Internal error');
        }
    }

    /**
     * Process individual JSON-RPC message
     */
    private function processMessage(array $message, ?string $domain): array
    {
        $method = $message['method'];
        $params = $message['params'] ?? [];
        $id = $message['id'];

        $this->logger->info('Processing MCP message', [
            'method' => $method,
            'domain' => $domain,
            'id' => $id
        ]);

        try {
            $result = null;
            switch ($method) {
                case 'initialize':
                    $result = $this->handleInitialize($domain);
                    break;
                case 'tools/list':
                    $result = $domain !== null 
                        ? $this->handleDomainToolsList($domain, $params)
                        : $this->handleAllToolsList($params);
                    break;
                case 'tools/call':
                    if ($domain === null) {
                        throw new \Exception('Domain required for tools/call. Use /mcp/domain/{domain} endpoint.');
                    }
                    $result = $this->handleDomainToolsCall($domain, $params);
                    break;
                case 'ping':
                    $result = $this->handlePing();
                    break;
                default:
                    throw new \Exception("Method not found: $method");
            }

            return $this->createSuccessResponse($id, $result);
        } catch (\Throwable $exception) {
            $this->logger->error('MCP method execution failed', [
                'method' => $method,
                'domain' => $domain,
                'error' => $exception->getMessage()
            ]);

            return $this->createErrorResponse($id, -32601, $exception->getMessage());
        }
    }

    /**
     * Handle initialize method
     */
    private function handleInitialize(?string $domain): array
    {
        $context = $domain !== null 
            ? "domain: $domain" 
            : "generic (all domains)";
            
        return [
            'protocolVersion' => '2024-11-05',
            'capabilities' => [
                'tools' => [
                    'listChanged' => false
                ]
            ],
            'serverInfo' => [
                'name' => 'Intucart MCP Server',
                'version' => '1.0.0'
            ],
            'instructions' => "Initialized for $context",
            'availableDomains' => $this->toolsManager->getAvailableDomains()
        ];
    }

    /**
     * Handle tools/list for domain-based routing
     */
    private function handleDomainToolsList(string $domain, array $params): array
    {
        $domainTools = $this->toolsManager->getToolsForDomain($domain);
        $filteredTools = [];

        foreach ($domainTools as $tool) {
            // Permission check
            if ($this->toolsManager->checkToolPermission($tool['name'])) {
                // Remove internal properties before sending to client
                $clientTool = $this->prepareToolForClient($tool);
                $filteredTools[] = $clientTool;
            }
        }

        $this->logger->info('Domain tools list', [
            'domain' => $domain,
            'tool_count' => count($filteredTools),
            'tools' => array_column($filteredTools, 'name')
        ]);

        return ['tools' => $filteredTools];
    }

    /**
     * Handle tools/list for all domains (generic endpoint)
     */
    private function handleAllToolsList(array $params): array
    {
        $allTools = $this->toolsManager->getAllTools();
        $filteredTools = [];

        foreach ($allTools as $tool) {
            if ($this->toolsManager->checkToolPermission($tool['name'])) {
                $filteredTools[] = $this->prepareToolForClient($tool);
            }
        }

        $this->logger->info('All tools list', [
            'tool_count' => count($filteredTools),
            'tools' => array_column($filteredTools, 'name')
        ]);

        return ['tools' => $filteredTools];
    }

    /**
     * Handle tools/call for domain-based routing
     */
    private function handleDomainToolsCall(string $domain, array $params): array
    {
        $toolName = $params['name'] ?? '';
        $arguments = $params['arguments'] ?? [];

        // Verify tool belongs to this domain
        $domainTools = $this->toolsManager->getToolsForDomain($domain);
        
        if (!isset($domainTools[$toolName])) {
            throw new \Exception("Tool '$toolName' not available in domain '$domain'");
        }

        // Execute the tool
        $result = $this->toolsManager->executeTool($toolName, $arguments);

        return [
            'content' => [
                [
                    'type' => 'text',
                    'text' => json_encode($result)
                ]
            ]
        ];
    }

    /**
     * Prepare tool for client (remove internal properties)
     */
    private function prepareToolForClient(array $tool): array
    {
        // Remove callback and other internal properties
        return [
            'name' => $tool['name'],
            'description' => $tool['description'] ?? '',
            'inputSchema' => $tool['inputSchema'] ?? ['type' => 'object', 'properties' => []],
        ];
    }

    /**
     * Handle ping method
     */
    private function handlePing(): array
    {
        return ['status' => 'ok', 'timestamp' => time()];
    }

    /**
     * Validate request headers
     */
    private function validateHeaders(WP_REST_Request $request)
    {
        $accept = $request->get_header('accept');
        if (
            !$accept ||
            (strpos($accept, 'application/json') === false &&
             strpos($accept, 'text/event-stream') === false)
        ) {
            return $this->createErrorResponse(0, -32600, 'Invalid Accept header');
        }

        $contentType = $request->get_header('content-type');
        if ($contentType && strpos($contentType, 'application/json') === false) {
            return $this->createErrorResponse(0, -32600, 'Invalid Content-Type header');
        }

        return true;
    }

    /**
     * Validate JSON-RPC message
     */
    private function validateJsonRpcMessage(array $message)
    {
        if (!isset($message['jsonrpc']) || $message['jsonrpc'] !== '2.0') {
            return $this->createErrorResponse(0, -32600, 'Invalid JSON-RPC version');
        }

        if (!isset($message['method'])) {
            return $this->createErrorResponse(0, -32600, 'Missing method');
        }

        if (!isset($message['id'])) {
            return $this->createErrorResponse(0, -32600, 'Missing id');
        }

        return true;
    }

    /**
     * Create success response
     */
    private function createSuccessResponse(int $id, array $result): array
    {
        return [
            'jsonrpc' => '2.0',
            'id' => $id,
            'result' => $result
        ];
    }

    /**
     * Create error response
     */
    private function createErrorResponse(int $id, int $code, string $message)
    {
        $error = [
            'jsonrpc' => '2.0',
            'id' => $id,
            'error' => [
                'code' => $code,
                'message' => $message
            ]
        ];

        return is_int($id) && $id === 0 ? new WP_REST_Response($error, 400) : $error;
    }

    /**
     * Get available domains
     */
    public function getAvailableDomains(): array
    {
        return $this->toolsManager->getAvailableDomains();
    }

    /**
     * Get tools for specific domain
     */
    public function getToolsForDomain(string $domain): array
    {
        return array_keys($this->toolsManager->getToolsForDomain($domain));
    }

    /**
     * Get MCP server status
     */
    public function getStatus(): array
    {
        if (!self::$initialized) {
            return [
                'status' => 'not_initialized',
                'message' => 'MCP server not initialized'
            ];
        }

        $domains = $this->getAvailableDomains();
        $domainMappings = $this->toolsManager->getDomainToolMappings();

        return [
            'status' => 'active',
            'message' => 'MCP server is active and ready',
            'domains' => $domains,
            'domain_tools' => $domainMappings,
            'endpoints' => [
                'base' => '/wp-json/intufind/v1/mcp',
                'domain_based' => '/wp-json/intufind/v1/mcp/domain/{domain}',
            ]
        ];
    }

    /**
     * Create MCP client configuration for a domain
     */
    public function createClientConfig(string $domain): array
    {
        if (!self::$initialized) {
            throw new \Exception('MCP server not initialized');
        }

        if (!$this->toolsManager->hasDomain($domain)) {
            throw new \Exception("Domain '$domain' not found");
        }

        $baseUrl = get_site_url();

        return [
            'type' => 'http',
            'name' => "intucart-mcp-$domain",
            'baseUrl' => $baseUrl,
            'endpoints' => [
                'mcp' => "/wp-json/intufind/v1/mcp/domain/$domain"
            ],
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json, text/event-stream'
            ],
            'transport' => 'http-post',
            'domain' => $domain,
            'available_tools' => $this->getToolsForDomain($domain)
        ];
    }

    /**
     * Get available endpoints for debugging
     */
    public function getEndpoints(): array
    {
        $baseUrl = get_site_url();
        
        // Build domain endpoints dynamically from registered domains
        $domainEndpoints = [];
        foreach ($this->getAvailableDomains() as $domain) {
            $domainEndpoints[$domain] = "$baseUrl/wp-json/intufind/v1/mcp/domain/$domain";
        }

        $endpoints = [
            'base_endpoint' => "$baseUrl/wp-json/intufind/v1/mcp",
        ];

        // Only add domain endpoints if we have any
        if (!empty($domainEndpoints)) {
            $endpoints['domain_endpoints'] = $domainEndpoints;
        }

        return $endpoints;
    }

    /**
     * Test MCP server connection for a domain
     */
    public function testDomainConnection(string $domain): array
    {
        if (!self::$initialized) {
            return [
                'success' => false,
                'error' => 'MCP server not initialized'
            ];
        }

        if (!$this->toolsManager->hasDomain($domain)) {
            return [
                'success' => false,
                'error' => "Domain '$domain' not found"
            ];
        }

        try {
            $tools = $this->toolsManager->getToolsForDomain($domain);
            $permittedTools = [];
            
            foreach ($tools as $toolName => $tool) {
                if ($this->toolsManager->checkToolPermission($toolName)) {
                    $permittedTools[] = $toolName;
                }
            }

            return [
                'success' => true,
                'domain' => $domain,
                'tools_available' => count($permittedTools),
                'tools' => $permittedTools,
                'server_status' => 'active'
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Check if MCP server is initialized
     */
    public static function isInitialized(): bool
    {
        return self::$initialized;
    }
}
