<?php

namespace Intucart\Services\Mcp;

use Intucart\Services\Logger;
use Intucart\Services\Mcp\Tools\OrdersMcpTools;
use Intucart\Services\Mcp\Tools\BaseMcpTools;

/**
 * MCP Tools Manager
 * Manages tool registration, discovery, and execution for the Intucart MCP server
 * 
 * Extensibility Hooks:
 * - Filter 'intufind_mcp_tool_managers': Register custom tool managers by domain
 * - Filter 'intufind_mcp_tools': Modify registered tools (add, remove, override)
 * - Filter 'intufind_mcp_domains': Register custom domains
 * - Action 'intufind_mcp_initialized': After MCP initialization
 */
class McpToolsManager
{
    private Logger $logger;
    private McpAuthHandler $authHandler;
    private array $registeredTools = [];
    private array $toolManagers = [];
    private array $toolsByDomain = [];

    public function __construct(Logger $logger, McpAuthHandler $authHandler)
    {
        $this->logger = $logger;
        $this->authHandler = $authHandler;
    }

    /**
     * Initialize the tools manager
     */
    public function initialize(): void
    {
        $this->initializeToolManagers();
        $this->registerAllTools();

        // Action: After MCP initialization (for custom setup)
        do_action('intufind_mcp_initialized', $this);

        $this->logger->info('MCP Tools Manager initialized', [
            'total_tools' => count($this->registeredTools),
            'domains' => array_keys($this->toolsByDomain)
        ]);
    }

    /**
     * Initialize tool managers
     * Uses WordPress filter to allow plugins to register custom tool managers
     */
    private function initializeToolManagers(): void
    {
        // Default built-in tool managers (only if WooCommerce is active)
        $defaultManagers = [];
        if (class_exists('WooCommerce')) {
            $defaultManagers = [
                'orders' => new OrdersMcpTools($this->logger, $this->authHandler),
            ];
        }

        /**
         * Filter: Register custom tool providers by domain
         * 
         * @param array<string, BaseMcpTools> $toolManagers Tool managers keyed by domain
         * @param Logger $logger Logger instance for tool managers
         * @param McpAuthHandler $authHandler Auth handler for tool managers
         * @return array<string, BaseMcpTools> Modified tool managers
         * 
         * Example:
         * add_filter('intufind_mcp_tool_managers', function($managers, $logger, $authHandler) {
         *     $managers['inventory'] = new MyInventoryMcpTools($logger, $authHandler);
         *     return $managers;
         * }, 10, 3);
         */
        $this->toolManagers = apply_filters(
            'intufind_mcp_tool_managers',
            $defaultManagers,
            $this->logger,
            $this->authHandler
        );
    }

    /**
     * Register all tools from managers
     */
    private function registerAllTools(): void
    {
        // Register built-in tools
        $this->registerBuiltInTools();

        // Register tools from each manager
        foreach ($this->toolManagers as $domain => $manager) {
            if (!($manager instanceof BaseMcpTools)) {
                $this->logger->warning("Invalid tool manager for domain: {$domain}");
                continue;
            }

            $tools = $manager->registerTools();
            foreach ($tools as $toolName => $toolConfig) {
                $this->registeredTools[$toolName] = $toolConfig;
                
                // Index tools by domain for domain-based routing
                $toolDomain = $toolConfig['domain'] ?? $domain;
                if (!isset($this->toolsByDomain[$toolDomain])) {
                    $this->toolsByDomain[$toolDomain] = [];
            }
                $this->toolsByDomain[$toolDomain][$toolName] = $toolConfig;
            }
            
            $this->logger->info("Registered tools from {$domain} manager", [
                'tool_count' => count($tools),
                'domain' => $manager->getDomain()
            ]);
        }

        /**
         * Filter: Modify registered tools (add, remove, override)
         * 
         * @param array $registeredTools All registered tools
         * @return array Modified tools
         * 
         * Example:
         * add_filter('intufind_mcp_tools', function($tools) {
         *     // Add a custom tool
         *     $tools['my_custom_tool'] = [
         *         'name' => 'my_custom_tool',
         *         'domain' => 'custom',
         *         'description' => 'My custom tool',
         *         'inputSchema' => [...],
         *         'callback' => [MyClass::class, 'myMethod']
         *     ];
         *     return $tools;
         * });
         */
        $this->registeredTools = apply_filters('intufind_mcp_tools', $this->registeredTools);

        // Rebuild domain index after filter
        $this->rebuildDomainIndex();
    }

    /**
     * Rebuild the domain index after tools have been modified
     */
    private function rebuildDomainIndex(): void
    {
        $this->toolsByDomain = [];
        foreach ($this->registeredTools as $toolName => $toolConfig) {
            $domain = $toolConfig['domain'] ?? 'default';
            if (!isset($this->toolsByDomain[$domain])) {
                $this->toolsByDomain[$domain] = [];
            }
            $this->toolsByDomain[$domain][$toolName] = $toolConfig;
        }
    }

    /**
     * Register built-in tools
     */
    private function registerBuiltInTools(): void
    {
        // Reserved for future built-in tools
        // Currently no built-in tools are registered
    }

    /**
     * Get all registered tools
     */
    public function getAllTools(): array
    {
        return array_values($this->registeredTools);
    }

    /**
     * Get tool by name
     */
    public function getTool(string $name): ?array
    {
        return $this->registeredTools[$name] ?? null;
    }

    /**
     * Execute a tool
     */
    public function executeTool(string $toolName, array $arguments): array
    {
        $tool = $this->getTool($toolName);
        if (!$tool) {
            throw new \Exception("Tool '$toolName' not found");
        }

        $this->logger->info('Executing MCP tool', [
            'tool_name' => $toolName,
            'arguments' => $arguments
        ]);

        try {
            $callback = $tool['callback'];

            if (is_array($callback) && count($callback) === 2) {
                $result = call_user_func($callback, $arguments);
            } elseif (is_callable($callback)) {
                $result = $callback($arguments);
            } else {
                throw new \Exception("Invalid callback for tool '$toolName'");
            }

            $this->logger->info('MCP tool executed successfully', [
                'tool_name' => $toolName
            ]);

            return $result;
        } catch (\Exception $e) {
            $this->logger->error('MCP tool execution failed', [
                'tool_name' => $toolName,
                'error' => $e->getMessage(),
                'arguments' => $arguments
            ]);

            return [
                'error' => "Tool execution failed: " . $e->getMessage()
            ];
        }
    }

    /**
     * Check if user has permission to use a tool
     */
    public function checkToolPermission(string $toolName): bool
    {
        // Get user context with permissions from auth handler
        $userContext = $this->authHandler->getCurrentUserContext();

        // Define tool permission mappings
        $toolPermissions = [
            'get_orders' => ($userContext['permissions']['can_view_own_orders'] ?? false) || ($userContext['permissions']['can_view_any_orders'] ?? false),
            'get_order_details' => ($userContext['permissions']['can_view_own_orders'] ?? false) || ($userContext['permissions']['can_view_any_orders'] ?? false),
            'update_order_status' => $userContext['permissions']['can_update_orders'] ?? false,
        ];

        return $toolPermissions[$toolName] ?? true; // Default to allow if not specified
    }

    /**
     * Get tools for a specific domain
     * 
     * @param string $domain The domain identifier (e.g., 'orders')
     * @return array Tools for the specified domain
     */
    public function getToolsForDomain(string $domain): array
    {
        return $this->toolsByDomain[$domain] ?? [];
    }

    /**
     * Get all available domains
     * 
     * @return array List of domain identifiers
     */
    public function getAvailableDomains(): array
    {
        /**
         * Filter: Register custom domains
         * 
         * @param array $domains Available domain identifiers
         * @return array Modified domains
         */
        return apply_filters(
            'intufind_mcp_domains',
            array_keys($this->toolsByDomain)
        );
    }

    /**
     * Check if a domain exists
     * 
     * @param string $domain The domain identifier
     * @return bool True if domain exists
     */
    public function hasDomain(string $domain): bool
    {
        return isset($this->toolsByDomain[$domain]);
    }

    /**
     * Get domain for a specific tool
     * 
     * @param string $toolName The tool name
     * @return string|null The domain or null if not found
     */
    public function getToolDomain(string $toolName): ?string
    {
        $tool = $this->getTool($toolName);
        return $tool['domain'] ?? null;
    }

    /**
     * Get tool manager for a domain
     * 
     * @param string $domain The domain identifier
     * @return BaseMcpTools|null The tool manager or null
     */
    public function getToolManager(string $domain): ?BaseMcpTools
    {
        return $this->toolManagers[$domain] ?? null;
    }

    /**
     * Get domain-to-tools mapping (useful for debugging/introspection)
     * 
     * @return array<string, array<string, array>> Domain to tools mapping
     */
    public function getDomainToolMappings(): array
    {
        $mappings = [];
        foreach ($this->toolsByDomain as $domain => $tools) {
            $mappings[$domain] = array_keys($tools);
        }
        return $mappings;
    }
}
