<?php

namespace Intucart\Services\Mcp\Tools;

use Intucart\Services\Logger;
use Intucart\Services\Mcp\McpAuthHandler;

/**
 * Base class for MCP tools with shared functionality
 * Updated to work with custom Intucart MCP server
 * 
 * Each tool class belongs to a domain (e.g., 'orders')
 * which enables domain-based routing and agent separation.
 */
abstract class BaseMcpTools
{
    protected Logger $logger;
    protected McpAuthHandler $authHandler;
    protected array $registeredTools = [];

    public function __construct(Logger $logger, McpAuthHandler $authHandler)
    {
        $this->logger = $logger;
        $this->authHandler = $authHandler;
    }

    /**
     * Abstract method that each tool class must implement to register its tools
     */
    abstract public function registerTools(): array;

    /**
     * Abstract method that each tool class must implement to specify its domain
     * 
     * @return string The domain identifier (e.g., 'orders')
     */
    abstract public function getDomain(): string;

    /**
     * Register a tool with this tool manager
     * Automatically includes domain metadata from getDomain()
     */
    protected function registerTool(array $toolConfig): void
    {
        $toolName = $toolConfig['name'];
        // Automatically add domain metadata to each tool
        $toolConfig['domain'] = $this->getDomain();
        $this->registeredTools[$toolName] = $toolConfig;
    }

    /**
     * Get all registered tools from this manager
     */
    public function getRegisteredTools(): array
    {
        return $this->registeredTools;
    }

    /**
     * Validate if user can access a specific order
     */
    protected function canAccessOrder(int $orderId, array $userContext, array $permissions): bool
    {
        // Admins can access all orders
        if ($permissions['can_view_any_orders']) {
            return true;
        }

        // Users can only access their own orders
        if ($permissions['can_view_own_orders'] && $userContext['id'] > 0) {
            $order = wc_get_order($orderId);
            if ($order && $order->get_customer_id() == $userContext['id']) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if WooCommerce is active
     */
    protected function isWooCommerceActive(): bool
    {
        return class_exists('WooCommerce');
    }

    /**
     * Validate request arguments against schema
     */
    protected function validateArguments(array $args, array $schema): array
    {
        $errors = [];
        $validated = [];

        // Check required fields
        if (isset($schema['required'])) {
            foreach ($schema['required'] as $field) {
                if (!isset($args[$field]) || empty($args[$field])) {
                    $errors[] = "Missing required field: $field";
                }
            }
        }

        // Validate field types and constraints
        if (isset($schema['properties'])) {
            foreach ($schema['properties'] as $field => $fieldSchema) {
                if (isset($args[$field])) {
                    $validated[$field] = $this->validateField($args[$field], $fieldSchema, $field, $errors);
                } elseif (isset($fieldSchema['default'])) {
                    $validated[$field] = $fieldSchema['default'];
                }
            }
        }

        if (!empty($errors)) {
            throw new \InvalidArgumentException('Validation errors: ' . implode(', ', $errors));
        }

        return $validated;
    }

    /**
     * Validate individual field
     */
    private function validateField($value, array $schema, string $fieldName, array &$errors)
    {
        $type = $schema['type'] ?? 'string';

        switch ($type) {
            case 'integer':
                if (!is_numeric($value)) {
                    $errors[] = "$fieldName must be an integer";
                    return $value;
                }
                $value = (int)$value;

                if (isset($schema['minimum']) && $value < $schema['minimum']) {
                    $errors[] = "$fieldName must be at least {$schema['minimum']}";
                }
                if (isset($schema['maximum']) && $value > $schema['maximum']) {
                    $errors[] = "$fieldName must be at most {$schema['maximum']}";
                }
                break;

            case 'string':
                if (!is_string($value)) {
                    $errors[] = "$fieldName must be a string";
                    return $value;
                }

                if (isset($schema['enum']) && !in_array($value, $schema['enum'])) {
                    $errors[] = "$fieldName must be one of: " . implode(', ', $schema['enum']);
                }
                break;

            case 'array':
                if (!is_array($value)) {
                    $errors[] = "$fieldName must be an array";
                }
                break;
        }

        return $value;
    }

    /**
     * Create standardized error response
     */
    protected function createErrorResponse(string $message, string $code = 'error'): array
    {
        return [
            'error' => [
                'code' => $code,
                'message' => $message
            ]
        ];
    }

    /**
     * Create standardized success response
     */
    protected function createSuccessResponse(array $data): array
    {
        return [
            'success' => true,
            'data' => $data
        ];
    }

    /**
     * Log tool execution
     */
    protected function logToolExecution(string $toolName, array $args, array $userContext): void
    {
        $this->logger->info('MCP tool executed', [
            'tool' => $toolName,
            'user_id' => $userContext['id'] ?? 0,
            'user_type' => $userContext['type'] ?? 'unknown',
            'args_count' => count($args)
        ]);
    }

    /**
     * Handle tool exceptions
     */
    protected function handleToolException(string $toolName, \Exception $exception): array
    {
        $this->logger->error('MCP tool execution failed', [
            'tool' => $toolName,
            'error' => $exception->getMessage(),
            'trace' => $exception->getTraceAsString()
        ]);

        return $this->createErrorResponse(
            'Tool execution failed: ' . $exception->getMessage(),
            'execution_error'
        );
    }
}
