<?php

namespace Intucart\Services\Mcp\Tools;

/**
 * MCP tools for WooCommerce order management
 * Domain: orders
 */
class OrdersMcpTools extends BaseMcpTools
{
    /**
     * Get the domain identifier for this tool manager
     */
    public function getDomain(): string
    {
        return 'orders';
    }

    /**
     * Register order-related MCP tools
     */
    public function registerTools(): array
    {
        $this->registerTool([
            'name' => 'get_orders',
            'description' => 'Get orders with flexible filtering and sorting options',
            'inputSchema' => [
                'type' => 'object',
                'properties' => [
                    'customer_id' => [
                        'type' => 'integer',
                        'description' => 'Filter by specific customer ID'
                    ],
                    'email' => [
                        'type' => 'string',
                        'format' => 'email',
                        'description' => 'Filter by customer email (alternative to customer_id)'
                    ],
                    'status' => [
                        'type' => 'array',
                        'items' => [
                            'type' => 'string',
                            'enum' => ['pending', 'processing', 'on-hold', 'completed', 'cancelled', 'refunded', 'failed']
                        ],
                        'description' => 'Filter by order status(es)'
                    ],
                    'limit' => [
                        'type' => 'integer',
                        'default' => 20,
                        'minimum' => 1,
                        'maximum' => 100,
                        'description' => 'Number of orders to retrieve'
                    ],
                    'orderby' => [
                        'type' => 'string',
                        'enum' => ['date', 'id', 'total', 'modified'],
                        'default' => 'date',
                        'description' => 'Sort orders by: date (creation date), id (order ID), total (order amount), or modified (last modified date)'
                    ],
                    'order' => [
                        'type' => 'string',
                        'enum' => ['ASC', 'DESC'],
                        'default' => 'DESC',
                        'description' => 'Sort direction: ASC (ascending/oldest first) or DESC (descending/newest first)'
                    ],
                    'date_after' => [
                        'type' => 'string',
                        'format' => 'date',
                        'description' => 'Get orders created after this date (YYYY-MM-DD format)'
                    ],
                    'date_before' => [
                        'type' => 'string',
                        'format' => 'date',
                        'description' => 'Get orders created before this date (YYYY-MM-DD format)'
                    ]
                ]
            ],
            'callback' => [$this, 'getOrders'],
        ]);

        $this->registerTool([
            'name' => 'get_order_details',
            'description' => 'Get detailed information about a specific WooCommerce order',
            'inputSchema' => [
                'type' => 'object',
                'properties' => [
                    'order_id' => [
                        'type' => 'integer',
                        'description' => 'Order ID to retrieve details for'
                    ]
                ],
                'required' => ['order_id']
            ],
            'callback' => [$this, 'getOrderDetails'],
        ]);

        $this->registerTool([
            'name' => 'update_order_status',
            'description' => 'Update the status of an order',
            'inputSchema' => [
                'type' => 'object',
                'properties' => [
                    'order_id' => [
                        'type' => 'integer',
                        'description' => 'Order ID to update'
                    ],
                    'status' => [
                        'type' => 'string',
                        'enum' => ['pending', 'processing', 'on-hold', 'completed', 'cancelled', 'refunded', 'failed'],
                        'description' => 'New order status'
                    ],
                    'note' => [
                        'type' => 'string',
                        'description' => 'Optional note to add to the order'
                    ]
                ],
                'required' => ['order_id', 'status']
            ],
            'callback' => [$this, 'updateOrderStatus'],
        ]);

        return $this->getRegisteredTools();
    }

    /**
     * Get orders
     */
    public function getOrders(array $args): array
    {
        try {
            if (!$this->isWooCommerceActive()) {
                return $this->createErrorResponse('WooCommerce is not active');
            }

            // Get user context and check permissions
            $userContext = $this->authHandler->getCurrentUserContext();
            $permissions = $userContext['permissions'];

            if (!$permissions['can_view_own_orders'] && !$permissions['can_view_any_orders']) {
                return $this->createErrorResponse('Permission denied: Cannot view orders');
            }

            // Use validateArguments to validate and set defaults - reuse the same schema as registration
            $validatedArgs = $this->validateArguments($args, [
                'type' => 'object',
                'properties' => [
                    'customer_id' => [
                        'type' => 'integer',
                        'description' => 'Filter by specific customer ID'
                    ],
                    'email' => [
                        'type' => 'string',
                        'format' => 'email',
                        'description' => 'Filter by customer email (alternative to customer_id)'
                    ],
                    'status' => [
                        'type' => 'array',
                        'items' => [
                            'type' => 'string',
                            'enum' => ['pending', 'processing', 'on-hold', 'completed', 'cancelled', 'refunded', 'failed']
                        ],
                        'description' => 'Filter by order status(es)'
                    ],
                    'limit' => [
                        'type' => 'integer',
                        'default' => 20,
                        'minimum' => 1,
                        'maximum' => 100,
                        'description' => 'Number of orders to retrieve'
                    ],
                    'orderby' => [
                        'type' => 'string',
                        'enum' => ['date', 'id', 'total', 'modified'],
                        'default' => 'date',
                        'description' => 'Sort orders by'
                    ],
                    'order' => [
                        'type' => 'string',
                        'enum' => ['ASC', 'DESC'],
                        'default' => 'DESC',
                        'description' => 'Sort direction'
                    ],
                    'date_after' => [
                        'type' => 'string',
                        'format' => 'date',
                        'description' => 'Get orders created after this date (YYYY-MM-DD format)'
                    ],
                    'date_before' => [
                        'type' => 'string',
                        'format' => 'date',
                        'description' => 'Get orders created before this date (YYYY-MM-DD format)'
                    ]
                ]
            ]);

            // Build query arguments for WooCommerce
            $queryArgs = [
                'limit' => $validatedArgs['limit'],
                'orderby' => $validatedArgs['orderby'],
                'order' => $validatedArgs['order']
            ];

            // Handle status filtering
            if (!empty($validatedArgs['status'])) {
                $queryArgs['status'] = $validatedArgs['status'];
            }

            // Handle customer filtering by ID or email
            if (!empty($validatedArgs['customer_id'])) {
                $queryArgs['customer_id'] = $validatedArgs['customer_id'];
            } elseif (!empty($validatedArgs['email'])) {
                $queryArgs['billing_email'] = $validatedArgs['email'];
            }

            // Handle date filtering
            if (!empty($validatedArgs['date_after'])) {
                $queryArgs['date_created'] = '>=' . $validatedArgs['date_after'];
            }
            if (!empty($validatedArgs['date_before'])) {
                // If we already have date_after, combine them
                if (!empty($validatedArgs['date_after'])) {
                    $queryArgs['date_created'] = $validatedArgs['date_after'] . '...' . $validatedArgs['date_before'];
                } else {
                    $queryArgs['date_created'] = '<=' . $validatedArgs['date_before'];
                }
            }

            // Restrict to current user's orders if they can't view all orders
            if (!$permissions['can_view_any_orders'] && $userContext['id'] > 0) {
                $queryArgs['customer_id'] = $userContext['id'];
            }

            $orders = wc_get_orders($queryArgs);
            $formatted_orders = [];

            foreach ($orders as $order) {
                $formatted_orders[] = [
                    'id' => $order->get_id(),
                    'status' => $order->get_status(),
                    'total' => $order->get_total(),
                    'currency' => $order->get_currency(),
                    'date_created' => $order->get_date_created()->date('c'),
                    'date_modified' => $order->get_date_modified()->date('c'),
                    'customer_id' => $order->get_customer_id(),
                    'billing_email' => $order->get_billing_email(),
                    'billing_name' => trim($order->get_billing_first_name() . ' ' . $order->get_billing_last_name()),
                    'item_count' => $order->get_item_count(),
                    'payment_method' => $order->get_payment_method_title(),
                ];
            }

            return $this->createSuccessResponse([
                'orders' => $formatted_orders,
                'count' => count($formatted_orders),
                'filters_applied' => [
                    'customer_id' => $queryArgs['customer_id'] ?? null,
                    'email' => $queryArgs['billing_email'] ?? null,
                    'status' => $queryArgs['status'] ?? null,
                    'date_created' => $queryArgs['date_created'] ?? null,
                    'limit' => $queryArgs['limit'],
                    'orderby' => $queryArgs['orderby'],
                    'order' => $queryArgs['order']
                ]
            ]);
        } catch (\Exception $e) {
            return $this->handleToolException('get_orders', $e);
        }
    }

    /**
     * Get order details
     */
    public function getOrderDetails(array $args): array
    {
        try {
            // Validate input
            $validatedArgs = $this->validateArguments($args, [
                'type' => 'object',
                'properties' => [
                    'order_id' => [
                        'type' => 'integer',
                        'description' => 'Order ID to retrieve details for'
                    ]
                ],
                'required' => ['order_id']
            ]);

            if (!$this->isWooCommerceActive()) {
                return $this->createErrorResponse('WooCommerce is not active');
            }

            // Authorization check
            $userContext = $this->authHandler->getCurrentUserContext();
            $permissions = $userContext['permissions'];
            $order_id = $validatedArgs['order_id'];

            if (!$this->canAccessOrder($order_id, $userContext, $permissions)) {
                return $this->createErrorResponse('Permission denied: Cannot access this order');
            }

            $order = wc_get_order($order_id);
            if (!$order) {
                return $this->createErrorResponse('Order not found');
            }

            $items = [];
            foreach ($order->get_items() as $item) {
                /** @var \WC_Order_Item_Product $item */
                $product = $item->get_product();
                $items[] = [
                    'name' => $item->get_name(),
                    'quantity' => $item->get_quantity(),
                    'total' => $item->get_total(),
                    'product_id' => $product ? $product->get_id() : null,
                    'sku' => $product ? $product->get_sku() : null,
                ];
            }

            return $this->createSuccessResponse([
                'order' => [
                    'id' => $order->get_id(),
                    'status' => $order->get_status(),
                    'total' => $order->get_total(),
                    'currency' => $order->get_currency(),
                    'date_created' => $order->get_date_created()->date('c'),
                    'customer_id' => $order->get_customer_id(),
                    'billing' => [
                        'first_name' => $order->get_billing_first_name(),
                        'last_name' => $order->get_billing_last_name(),
                        'email' => $order->get_billing_email(),
                        'phone' => $order->get_billing_phone(),
                        'address' => $order->get_formatted_billing_address(),
                    ],
                    'shipping' => [
                        'address' => $order->get_formatted_shipping_address(),
                        'method' => $order->get_shipping_method(),
                    ],
                    'payment_method' => $order->get_payment_method_title(),
                    'items' => $items,
                    'notes' => array_map(function ($note) {
                        return [
                            'content' => $note->comment_content,
                            'date' => $note->comment_date,
                            'author' => $note->comment_author,
                        ];
                    }, $order->get_customer_order_notes()),
                ]
            ]);
        } catch (\Exception $e) {
            return $this->handleToolException('get_order_details', $e);
        }
    }

    /**
     * Update order status
     */
    public function updateOrderStatus(array $args): array
    {
        try {
            // Validate input
            $validatedArgs = $this->validateArguments($args, [
                'type' => 'object',
                'properties' => [
                    'order_id' => [
                        'type' => 'integer',
                        'description' => 'Order ID to update'
                    ],
                    'status' => [
                        'type' => 'string',
                        'enum' => ['pending', 'processing', 'on-hold', 'completed', 'cancelled', 'refunded', 'failed'],
                        'description' => 'New order status'
                    ],
                    'note' => [
                        'type' => 'string',
                        'description' => 'Optional note to add to the order'
                    ]
                ],
                'required' => ['order_id', 'status']
            ]);

            if (!$this->isWooCommerceActive()) {
                return $this->createErrorResponse('WooCommerce is not active');
            }

            // Authorization check
            $userContext = $this->authHandler->getCurrentUserContext();
            $permissions = $userContext['permissions'];

            if (!$permissions['can_update_orders']) {
                return $this->createErrorResponse('Permission denied: Cannot update orders');
            }

            $order = wc_get_order($validatedArgs['order_id']);
            if (!$order) {
                return $this->createErrorResponse('Order not found');
            }

            $old_status = $order->get_status();
            $order->update_status($validatedArgs['status'], $validatedArgs['note'] ?? '');

            return $this->createSuccessResponse([
                'order_id' => $order->get_id(),
                'old_status' => $old_status,
                'new_status' => $validatedArgs['status'],
                'message' => "Order status updated from {$old_status} to {$validatedArgs['status']}"
            ]);
        } catch (\Exception $e) {
            return $this->handleToolException('update_order_status', $e);
        }
    }
}
