<?php

namespace Intucart\Services;

use Intucart\Services\Constants;

/**
 * Service for managing third-party page builder integrations
 * Handles cache busting, compatibility, and other page builder specific tasks
 */
class PageBuilderService
{
    private $logger;
    
    /**
     * Constructor
     *
     * @param Logger $logger Logger service
     */
    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
    }

    /**
     * Initialize page builder hooks and integrations
     *
     * @return void
     */
    public function initialize(): void
    {
        // Hook into settings updates to clear page builder caches
        add_action('update_option', [$this, 'handleSettingsUpdate'], 10, 3);
    }

    /**
     * Handle settings updates and clear page builder caches when needed
     *
     * @param string $option Option name that was updated
     * @param mixed $old_value Previous value
     * @param mixed $new_value New value
     * @return void
     */
    public function handleSettingsUpdate(string $option, $old_value, $new_value): void
    {
        // All chat styling options now use the standardized itf_chat_ prefix
        if (strpos($option, Constants::CHAT_OPTION_PREFIX) === 0) {
            $this->logger->debug('Chat styling option updated, clearing page builder caches', [
                'option' => $option,
                'prefix' => Constants::CHAT_OPTION_PREFIX,
                'old_value' => $old_value,
                'new_value' => $new_value
            ]);
            
            $this->clearAllPageBuilderCaches();
        }
    }

    /**
     * Clear caches for all detected page builders
     *
     * @return array Results of cache clearing attempts
     */
    public function clearAllPageBuilderCaches(): array
    {
        $results = [];
        
        // Clear Elementor cache
        $results['elementor'] = $this->clearElementorCache();
        
        // Clear Beaver Builder cache
        $results['beaver_builder'] = $this->clearBeaverBuilderCache();
        
        // Clear Divi cache
        $results['divi'] = $this->clearDiviCache();
        
        // Clear Oxygen cache
        $results['oxygen'] = $this->clearOxygenCache();
        
        // Clear WPBakery cache
        $results['wpbakery'] = $this->clearWPBakeryCache();
        
        // Log results
        $cleared = array_filter($results, function($result) {
            return $result['cleared'] === true;
        });
        
        if (!empty($cleared)) {
            $this->logger->info('Page builder caches cleared', [
                'cleared_builders' => array_keys($cleared),
                'total_cleared' => count($cleared)
            ]);
        }
        
        return $results;
    }

    /**
     * Clear Elementor cache
     *
     * @return array Result of cache clearing attempt
     */
    public function clearElementorCache(): array
    {
        if (!$this->isElementorActive()) {
            return ['cleared' => false, 'reason' => 'not_active'];
        }

        try {
            // Clear Elementor CSS cache
            if (class_exists('\Elementor\Plugin')) {
                \Elementor\Plugin::$instance->files_manager->clear_cache();
                
                // Also clear Elementor page cache if available
                if (method_exists(\Elementor\Plugin::$instance->files_manager, 'clear_all_cache')) {
                    \Elementor\Plugin::$instance->files_manager->clear_all_cache();
                }
                
                return ['cleared' => true, 'method' => 'files_manager'];
            }
            
            return ['cleared' => false, 'reason' => 'plugin_class_not_found'];
        } catch (\Exception $e) {
            $this->logger->error('Failed to clear Elementor cache', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return ['cleared' => false, 'reason' => 'exception', 'error' => $e->getMessage()];
        }
    }

    /**
     * Clear Beaver Builder cache
     *
     * @return array Result of cache clearing attempt
     */
    public function clearBeaverBuilderCache(): array
    {
        if (!$this->isBeaverBuilderActive()) {
            return ['cleared' => false, 'reason' => 'not_active'];
        }

        try {
            if (class_exists('FLBuilderModel')) {
                \FLBuilderModel::delete_asset_cache_for_all_posts();
                return ['cleared' => true, 'method' => 'FLBuilderModel'];
            }
            
            return ['cleared' => false, 'reason' => 'model_class_not_found'];
        } catch (\Exception $e) {
            $this->logger->error('Failed to clear Beaver Builder cache', [
                'error' => $e->getMessage()
            ]);
            
            return ['cleared' => false, 'reason' => 'exception', 'error' => $e->getMessage()];
        }
    }

    /**
     * Clear Divi cache
     *
     * @return array Result of cache clearing attempt
     */
    public function clearDiviCache(): array
    {
        if (!$this->isDiviActive()) {
            return ['cleared' => false, 'reason' => 'not_active'];
        }

        try {
            // Clear Divi static CSS cache
            if (function_exists('et_core_clear_wp_cache')) {
                et_core_clear_wp_cache();
                return ['cleared' => true, 'method' => 'et_core_clear_wp_cache'];
            }
            
            // Alternative method for older Divi versions
            if (class_exists('ET_Core_PageResource')) {
                \ET_Core_PageResource::remove_static_resources('all', 'all');
                return ['cleared' => true, 'method' => 'ET_Core_PageResource'];
            }
            
            return ['cleared' => false, 'reason' => 'no_clear_method_found'];
        } catch (\Exception $e) {
            $this->logger->error('Failed to clear Divi cache', [
                'error' => $e->getMessage()
            ]);
            
            return ['cleared' => false, 'reason' => 'exception', 'error' => $e->getMessage()];
        }
    }

    /**
     * Clear Oxygen cache
     *
     * @return array Result of cache clearing attempt
     */
    public function clearOxygenCache(): array
    {
        if (!$this->isOxygenActive()) {
            return ['cleared' => false, 'reason' => 'not_active'];
        }

        try {
            if (function_exists('oxygen_vsb_cache_universal_css')) {
                delete_option('oxygen_vsb_universal_css_cache');
                oxygen_vsb_cache_universal_css();
                return ['cleared' => true, 'method' => 'oxygen_vsb_cache_universal_css'];
            }
            
            return ['cleared' => false, 'reason' => 'no_clear_method_found'];
        } catch (\Exception $e) {
            $this->logger->error('Failed to clear Oxygen cache', [
                'error' => $e->getMessage()
            ]);
            
            return ['cleared' => false, 'reason' => 'exception', 'error' => $e->getMessage()];
        }
    }

    /**
     * Clear WPBakery (Visual Composer) cache
     *
     * @return array Result of cache clearing attempt
     */
    public function clearWPBakeryCache(): array
    {
        if (!$this->isWPBakeryActive()) {
            return ['cleared' => false, 'reason' => 'not_active'];
        }

        try {
            if (function_exists('vc_clear_cache')) {
                vc_clear_cache();
                return ['cleared' => true, 'method' => 'vc_clear_cache'];
            }
            
            return ['cleared' => false, 'reason' => 'no_clear_method_found'];
        } catch (\Exception $e) {
            $this->logger->error('Failed to clear WPBakery cache', [
                'error' => $e->getMessage()
            ]);
            
            return ['cleared' => false, 'reason' => 'exception', 'error' => $e->getMessage()];
        }
    }

    /**
     * Check if Elementor is active
     *
     * @return bool
     */
    public function isElementorActive(): bool
    {
        return class_exists('\Elementor\Plugin');
    }

    /**
     * Check if Beaver Builder is active
     *
     * @return bool
     */
    public function isBeaverBuilderActive(): bool
    {
        return class_exists('FLBuilder');
    }

    /**
     * Check if Divi is active
     *
     * @return bool
     */
    public function isDiviActive(): bool
    {
        return function_exists('et_divi_fonts_url') || defined('ET_BUILDER_VERSION');
    }

    /**
     * Check if Oxygen is active
     *
     * @return bool
     */
    public function isOxygenActive(): bool
    {
        return defined('CT_VERSION');
    }

    /**
     * Check if WPBakery is active
     *
     * @return bool
     */
    public function isWPBakeryActive(): bool
    {
        return defined('WPB_VC_VERSION');
    }

    /**
     * Get list of active page builders
     *
     * @return array Array of active page builder names
     */
    public function getActivePageBuilders(): array
    {
        $active = [];
        
        if ($this->isElementorActive()) {
            $active[] = 'elementor';
        }
        
        if ($this->isBeaverBuilderActive()) {
            $active[] = 'beaver_builder';
        }
        
        if ($this->isDiviActive()) {
            $active[] = 'divi';
        }
        
        if ($this->isOxygenActive()) {
            $active[] = 'oxygen';
        }
        
        if ($this->isWPBakeryActive()) {
            $active[] = 'wpbakery';
        }
        
        return $active;
    }

    /**
     * Get post types that should be excluded from sync because they're page builder templates/system types
     *
     * @return array Array of post type names that should be excluded
     */
    public function getExcludedPostTypes(): array
    {
        $excludedTypes = [];

        // Elementor post types
        if ($this->isElementorActive()) {
            $excludedTypes = array_merge($excludedTypes, [
                'elementor_library',   // Elementor templates, headers, footers, popups
                'e-floating-buttons',  // Elementor floating buttons
            ]);
        }

        // Beaver Builder post types
        if ($this->isBeaverBuilderActive()) {
            $excludedTypes = array_merge($excludedTypes, [
                'fl-builder-template', // Beaver Builder templates
                'fl-theme-layout',     // Beaver Builder theme layouts
            ]);
        }

        // Divi/Elegant Themes post types
        if ($this->isDiviActive()) {
            $excludedTypes = array_merge($excludedTypes, [
                'et_pb_layout',        // Divi layouts
                'et_header_layout',    // Divi header layouts
                'et_footer_layout',    // Divi footer layouts
                'et_body_layout',      // Divi body layouts
            ]);
        }

        // Oxygen post types
        if ($this->isOxygenActive()) {
            $excludedTypes = array_merge($excludedTypes, [
                'ct_template',         // Oxygen Builder templates
                'oxy_user_library',    // Oxygen user library items
            ]);
        }

        // WPBakery/Visual Composer post types
        if ($this->isWPBakeryActive()) {
            $excludedTypes = array_merge($excludedTypes, [
                'vc_grid_item',        // Visual Composer grid items
                'templatera',          // Templatera templates (WPBakery addon)
            ]);
        }

        // Other popular page builder post types (check if they exist rather than if plugin is active)
        $otherPageBuilderTypes = [
            // ElementsKit (Elementor addon)
            'elementskit_content',   // ElementsKit content templates
            'elementskit_template',  // ElementsKit templates
            'elementskit_widget',    // ElementsKit widgets
            'elementskit_items',     // ElementsKit items (alternative naming)
            'ekit_content',          // ElementsKit shortened naming
            'ekit_template',         // ElementsKit shortened template naming
            'floating_elements',     // Floating elements (alternative naming)
            'floating-elements',     // Floating elements (with dashes)
            'eael_content',          // Essential Addons for Elementor content
            'eael_template',         // Essential Addons for Elementor templates

            // Gutenberg/Block Editor
            'wp_block',              // WordPress reusable blocks (design components, not content)
            'wp_template',           // WordPress block theme templates
            'wp_template_part',      // WordPress block theme template parts
            'wp_navigation',         // WordPress navigation blocks
            'wp_global_styles',      // WordPress global styles

            // Other popular builders
            'brizy_template',        // Brizy templates
            'thrive_template',       // Thrive Architect templates
            'fusion_template',       // Avada Fusion Builder templates
            'kadence_template',      // Kadence blocks templates
        ];

        // Only include types that actually exist
        foreach ($otherPageBuilderTypes as $type) {
            if (post_type_exists($type)) {
                $excludedTypes[] = $type;
            }
        }

        return array_unique($excludedTypes);
    }

    /**
     * Check if a post type is a page builder template/system type
     *
     * @param string $postType Post type to check
     * @return bool True if it's a page builder type that should be excluded
     */
    public function isPageBuilderPostType(string $postType): bool
    {
        return in_array($postType, $this->getExcludedPostTypes(), true);
    }

    /**
     * Force clear all page builder caches (for manual/admin use)
     * This is more aggressive than the automatic clearing
     *
     * @return array Results of cache clearing attempts
     */
    public function forceClearAllCaches(): array
    {
        $this->logger->info('Force clearing all page builder caches (manual trigger)');
        
        $results = $this->clearAllPageBuilderCaches();
        
        // Additional aggressive clearing for stubborn caches
        if ($this->isElementorActive()) {
            // Delete Elementor CSS files from filesystem
            $upload_dir = wp_upload_dir();
            $elementor_css_path = $upload_dir['basedir'] . '/elementor/css/';
            
            if (is_dir($elementor_css_path)) {
                $files = glob($elementor_css_path . '*.css');
                foreach ($files as $file) {
                    if (is_file($file)) {
                        @unlink($file);
                    }
                }
                $results['elementor']['aggressive_file_cleanup'] = count($files) . ' files deleted';
            }
        }
        
        return $results;
    }
}
