<?php

namespace Intucart\Services\Providers\Events;

use Intucart\Services\Events\Purchase;
use Intucart\Services\Managers\EventManager;
use Intucart\Services\Events\Data\ProductDTO;
use Intucart\Services\Events\Data\CartItemDTO;
use Intucart\Services\Events\Data\OrderDTO;

/**
 * WooCommerce event provider implementation
 */
class WooCommerceEventProvider implements EventProviderInterface
{
    /**
     * Event manager instance
     *
     * @var EventManager
     */
    private EventManager $eventManager;

    /**
     * Constructor
     *
     * @param EventManager $eventManager Event manager instance
     */
    public function __construct(EventManager $eventManager)
    {
        $this->eventManager = $eventManager;
    }

    /**
     * Get provider name
     *
     * @return string
     */
    public function getName(): string
    {
        return 'woocommerce';
    }

    /**
     * Check if WooCommerce is active
     *
     * @return bool
     */
    public function isActive(): bool
    {
        return class_exists('WooCommerce');
    }

    /**
     * Initialize WooCommerce event hooks
     *
     * @return void
     */
    public function initialize(): void
    {
        if (!$this->isActive()) {
            return;
        }

        // Track purchases - only on order completion
        // Used for product-to-product relationships (frequently bought together)
        add_action(
            'woocommerce_order_status_completed',
            [$this, 'trackPurchase']
        );
    }

    /**
     * Create product data DTO for events
     *
     * @param \WC_Product $product WooCommerce product
     * @param float|null  $price   Optional override price
     *
     * @return ProductDTO
     */
    private function createProductDTO(\WC_Product $product, ?float $price = null): ProductDTO
    {
        return new ProductDTO(
            $product->get_id(),
            $product->get_name(),
            $price ?? (float) $product->get_price(),
            array_map(
                fn($term) => $term->name,
                wc_get_product_terms($product->get_id(), 'product_cat')
            ),
            array_map(
                fn($term) => $term->name,
                wc_get_product_terms($product->get_id(), 'product_cat', ['parent' => 0])
            )
        );
    }

    /**
     * Track purchase event
     *
     * @param int $orderId Order ID
     *
     * @return void
     */
    public function trackPurchase(int $orderId): void
    {
        $order = wc_get_order($orderId);
        if (!$order) {
            return;
        }

        // Check if we've already tracked this purchase
        if (get_post_meta($orderId, '_intucart_purchase_tracked', true)) {
            return;
        }

        $cartItemDTOs = [];
        foreach ($order->get_items() as $item) {
            if (!($item instanceof \WC_Order_Item_Product)) {
                continue;
            }

            $product = $item->get_product();
            if (!$product) {
                continue;
            }

            $productDTO = $this->createProductDTO($product, (float) $item->get_subtotal());
            $cartItemDTO = new CartItemDTO(
                $productDTO,
                $item->get_quantity()
            );

            $cartItemDTOs[] = $cartItemDTO;
        }

        $orderDTO = new OrderDTO(
            (string) $orderId,
            (float) $order->get_total(),
            $order->get_currency(),
            $cartItemDTOs
        );

        $event = new Purchase(
            (int) $order->get_customer_id(),
            [
                'order' => $orderDTO->toArray()
            ]
        );

        $this->eventManager->track($event);

        // Mark this purchase as tracked
        update_post_meta($orderId, '_intucart_purchase_tracked', true);
    }
}
