<?php

namespace Intucart\Services\Providers\PostTypes;

use Intucart\Services\Logger;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\AcfAutoMapper;
use Intucart\Utils\StringUtils;
use Intucart\Utils\ElementorUtils;
use Intucart\Utils\AttributeUtils;

/**
 * Provider for WordPress posts, pages, and custom post types.
 * Optimized for flat OpenSearch mapping structure with root-level fields.
 *
 * Features:
 * - Native Gutenberg block support (via WordPress core)
 * - Full Elementor page builder support with JSON content extraction
 * - ACF field integration and content extraction
 * - StringUtils text cleaning and standardization
 * - Comprehensive taxonomy and metadata handling
 */
class PostTypeProvider implements PostTypeProviderInterface
{
    protected Logger $logger;
    protected CacheService $cacheService;
    protected AcfAutoMapper $acfAutoMapper;

    public function __construct(Logger $logger, CacheService $cacheService, AcfAutoMapper $acfAutoMapper)
    {
        $this->logger = $logger;
        $this->cacheService = $cacheService;
        $this->acfAutoMapper = $acfAutoMapper;
    }

    /**
     * Get a post by ID and type.
     *
     * @param int $postId
     * @param string $postType
     * @return \WP_Post|null
     */
    public function getPost(int $postId, string $postType)
    {
        $post = get_post($postId);
        if (!$post || $post->post_type !== $postType) {
            $this->logger->debug('Post not found or type mismatch', [
                'post_id' => $postId,
                'expected_type' => $postType,
                'actual_type' => $post ? $post->post_type : null
            ]);
            return null;
        }
        return $post;
    }

    /**
     * Get multiple posts by type and args.
     *
     * @param array $args
     * @param string $postType
     * @return array
     */
    public function getPosts(array $args = [], string $postType = 'post'): array
    {
        $default_args = [
            'post_type' => $postType,
            'post_status' => ['publish', 'draft', 'private', 'pending', 'future'], // Get all potentially syncable posts
            'numberposts' => -1
        ];
        $query_args = array_merge($default_args, $args);
        $posts = get_posts($query_args);
        if (empty($posts)) {
            $this->logger->debug('No posts found for type', [
                'post_type' => $postType,
                'args' => $query_args
            ]);
            return [];
        }
        return $posts;
    }

    /**
     * Serialize a post for API responses with optimized flat structure.
     *
     * @param mixed $post
     * @param string $postType
     * @param array $additionalData
     * @return array
     */
    public function serializePost(mixed $post, string $postType, array $additionalData = []): array
    {
        if (!($post instanceof \WP_Post)) {
            $post = get_post($post);
            if (!$post) {
                $this->logger->warning('serializePost: Invalid post', [
                    'type' => gettype($post)
                ]);
                return [];
            }
        }

        $title = get_the_title($post);
        $content = $this->extractPostContent($post);
        $excerpt = StringUtils::cleanText($post->post_excerpt);
        $author = get_the_author_meta('display_name', $post->post_author);

        // Post-specific fields for root-level mapping
        $post_status = $post->post_status;
        $post_format = get_post_format($post) ?: 'standard';
        $word_count = str_word_count(strip_tags($content));
        $reading_time = max(1, ceil($word_count / 200));
        $featured_image = get_the_post_thumbnail_url($post, 'thumbnail') ?: '';
        $comment_count = (int) $post->comment_count;
        $view_count = (int) get_post_meta($post->ID, 'post_views_count', true) ?: 0;
        $last_modified = get_post_modified_time('c', true, $post);

        // Categories and tags (root-level arrays)
        $categories = [];
        $tags = [];

        // Get categories and tags for any post type that supports them
        $taxonomies = get_object_taxonomies($postType);

        // Handle categories
        if (in_array('category', $taxonomies)) {
            $post_categories = get_the_category($post->ID);
            if (!empty($post_categories)) {
                $categories = array_map(fn($cat) => $cat->name, $post_categories);
            }
        }

        // Handle tags
        if (in_array('post_tag', $taxonomies)) {
            $post_tags = get_the_tags($post->ID);
            if (!empty($post_tags)) {
                $tags = array_map(fn($tag) => $tag->name, $post_tags);
            }
        }

        // Topic extraction
        $topic = $this->extractTopic($title, $content);

        // Extract ACF fields for enhanced search capabilities
        $acfData = $this->acfAutoMapper->extractAcfFields($post->ID, $postType);

        // If no meaningful HTML/Elementor content, fall back to ACF search content for main content
        if ($content === '' && !empty($acfData['search_content'])) {
            $content = $acfData['search_content'];
        }

        // Custom attributes (only truly dynamic fields)
        $attributesData = [];

        // Get custom meta fields (excluding private ones and root-level fields)
        $meta = get_post_meta($post->ID);
        foreach ($meta as $key => $value) {
            if (strpos($key, '_') === 0) {
                continue; // Skip private/meta keys
            }
            if (in_array($key, ['word_count', 'reading_time', 'view_count', 'topic'])) {
                continue; // Skip fields that are now at root level
            }
            $attributesData[$key] = (is_array($value) && count($value) === 1) ? $value[0] : $value;
        }

        // Get custom taxonomies (excluding standard ones)
        $taxonomies = get_object_taxonomies($postType);
        foreach ($taxonomies as $taxonomy) {
            if (in_array($taxonomy, ['category', 'post_tag'])) {
                continue; // Skip standard taxonomies that go to root level
            }

            $terms = get_the_terms($post->ID, $taxonomy);
            if (!is_wp_error($terms) && !empty($terms)) {
                $term_names = array_map(fn($term) => $term->name, $terms);
                $attributesData[$taxonomy] = $term_names;
            }
        }

        // Integrate ACF fields with existing attributes for cloud compatibility
        if (!empty($acfData['fields'])) {
            // Add ACF values to existing attributes for cloud compatibility
            foreach ($acfData['fields'] as $acfField) {
                if ($acfField['filterable'] ?? false) {
                    $fieldKey = str_replace('acf_', '', $acfField['key']); // Remove acf_ prefix for cleaner keys
                    $fieldValues = $acfField['value'] ?? [];

                    // Add values to attributes (posts support flexible types)
                    if (!empty($fieldValues)) {
                        if (count($fieldValues) === 1) {
                            $attributesData[$fieldKey] = $fieldValues[0]; // Single value
                        } else {
                            $attributesData[$fieldKey] = $fieldValues; // Multiple values
                        }
                    }
                }
            }
        }

        $data = [
            'id' => (string) $post->ID,
            'post_type' => $postType,        // Fixed: was 'postType', cloud expects 'post_type'
            'title' => $title,
            'content' => $content,
            'excerpt' => $excerpt,
            'author_name' => $author,        // Fixed: was 'author', cloud expects 'author_name'
            'created_at' => get_post_time('c', true, $post),
            'published_at' => $post->post_status === 'publish' ? get_post_time('c', true, $post) : null,
            'categories' => $categories,
            'tags' => $tags,

            // Post-specific root-level fields
            'status' => $post_status,        // Fixed: was 'post_status', cloud expects 'status'
            'post_format' => $post_format,
            'topic' => $topic,
            'word_count' => $word_count,
            'reading_time' => $reading_time,
            'featured_image' => $featured_image,
            'comment_count' => $comment_count,
            'view_count' => $view_count,
            'last_modified' => $last_modified,
            'source' => 'wordpress',

            'attributes' => AttributeUtils::processAttributes($attributesData),
        ];

        // Add ACF search content as separate searchableText field only if it adds beyond main content
        if (!empty($acfData['search_content']) && $content !== $acfData['search_content']) {
            $data['searchableText'] = $acfData['search_content'];
        }

        // Merge any additional data
        if (!empty($additionalData)) {
            $data = array_merge($data, $additionalData);
        }

        return $data;
    }

    /**
     * Extract topic from title and content using simple keyword analysis.
     *
     * @param string $title
     * @param string $content
     * @return string
     */
    private function extractTopic(string $title, string $content): string
    {
        // Simple topic extraction - could be enhanced with NLP
        $text = $title . ' ' . substr($content, 0, 500);
        $words = str_word_count(strtolower($text), 1);

        // Remove common stop words
        $stopWords = ['the', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by', 'is', 'are', 'was', 'were', 'be', 'been', 'have', 'has', 'had', 'do', 'does', 'did', 'will', 'would', 'could', 'should', 'may', 'might', 'must', 'can', 'this', 'that', 'these', 'those', 'a', 'an'];
        $words = array_diff($words, $stopWords);

        // Get word frequency
        $wordCount = array_count_values($words);
        arsort($wordCount);

        // Return most frequent word as topic
        return !empty($wordCount) ? array_key_first($wordCount) : '';
    }

    /**
     * Extract content from a post, with special handling for Elementor
     *
     * @param \WP_Post $post Post object
     * @return string Extracted content
     */
    private function extractPostContent(\WP_Post $post): string
    {
        // 1) Standard WordPress content processing (works with Gutenberg)
        $content = apply_filters('the_content', $post->post_content);
        $plain   = trim(wp_strip_all_tags($content));

        // 2) If content is empty/very short, enhance with Elementor output when available
        if (strlen($plain) < 50) {
            // Centralized rendering (builder HTML or Theme Builder template)
            $rendered = ElementorUtils::renderPostHtml($post);
            if ($rendered !== '') {
                return $rendered;
            }

            // Fallback to cleaned Elementor JSON if post is edited with Elementor
            if (ElementorUtils::isElementorPost($post->ID)) {
                $elementorJson = $this->extractElementorContent($post->ID);
                if ($elementorJson !== '') {
                    return $elementorJson;
                }
            }
        }

        return $content;
    }

    /**
     * Extract content from Elementor page builder data using clean JSON approach
     *
     * @param int $postId Post ID
     * @return string Extracted content as JSON
     */
    private function extractElementorContent(int $postId): string
    {
        return ElementorUtils::extractContentAsJson($postId);
    }

    /**
     * Build the flat metadata payload that Intucart Cloud expects.
     * No embedding/text generation – the cloud service now handles that.
     */
    public function getCloudPayload(int $postId, string $postType, int $maxDescLength = 50000): array
    {
        $post = $this->getPost($postId, $postType);
        if (!$post) {
            $this->logger->debug('Post not found for cloud payload', [
                'post_id'   => $postId,
                'post_type' => $postType,
            ]);
            return [];
        }

        $title    = get_the_title($post);
        // Extract full HTML, then provide both markdown and plain text for different use cases
        $htmlContent = $this->extractPostContent($post);
        $contentMarkdown = StringUtils::htmlToMarkdown($htmlContent);
        // Prefer markdown for content field; only compute plain text if needed
        $content  = $contentMarkdown !== ''
            ? $contentMarkdown
            : StringUtils::extractPlainText($htmlContent);
        $excerpt  = StringUtils::cleanText($post->post_excerpt);
        $author   = get_the_author_meta('display_name', $post->post_author);

        // Post-specific root-level fields
        $post_status    = $post->post_status;
        $post_format    = get_post_format($post) ?: 'standard';
        $word_count     = str_word_count(strip_tags($content));
        $reading_time   = max(1, ceil($word_count / 200));
        $featured_image = get_the_post_thumbnail_url($post, 'thumbnail') ?: '';
        $comment_count  = (int) $post->comment_count;
        $view_count     = (int) get_post_meta($postId, 'post_views_count', true) ?: 0;
        $last_modified  = get_post_modified_time('c', true, $post);

        // Categories and tags
        $categories = [];
        $tags       = [];

        $taxonomies = get_object_taxonomies($postType);
        if (in_array('category', $taxonomies, true)) {
            $post_categories = get_the_category($postId);
            if (!empty($post_categories)) {
                $categories = array_map(static fn($cat) => $cat->name, $post_categories);
            }
        }
        if (in_array('post_tag', $taxonomies, true)) {
            $post_tags = get_the_tags($postId);
            if (!empty($post_tags)) {
                $tags = array_map(static fn($tag) => $tag->name, $post_tags);
            }
        }

        // Topic extraction
        $topic = $this->extractTopic($title, $content);

        // Extract ACF fields for enhanced search capabilities
        $acfData = $this->acfAutoMapper->extractAcfFields($post->ID, $postType);

        // Custom meta & taxonomy attributes (dynamic only)
        $attributesData = [];

        $meta = get_post_meta($postId);
        foreach ($meta as $key => $value) {
            if (strpos($key, '_') === 0) {
                continue; // private keys
            }
            if (in_array($key, ['word_count', 'reading_time', 'view_count', 'topic'], true)) {
                continue; // now root-level
            }
            $raw = (is_array($value) && count($value) === 1) ? $value[0] : $value;
            $attributesData[$key] = StringUtils::cleanAttributeValue($raw);
        }

        foreach ($taxonomies as $taxonomy) {
            if (in_array($taxonomy, ['category', 'post_tag'], true)) {
                continue; // handled above
            }
            $terms = get_the_terms($postId, $taxonomy);
            if (!is_wp_error($terms) && !empty($terms)) {
                $attributesData[$taxonomy] = array_map(static fn($term) => $term->name, $terms);
            }
        }

        // Integrate ACF fields with existing attributes for cloud compatibility
        if (!empty($acfData['fields'])) {
            // Add ACF values to existing attributes for cloud compatibility
            foreach ($acfData['fields'] as $acfField) {
                if ($acfField['filterable'] ?? false) {
                    $fieldKey = str_replace('acf_', '', $acfField['key']); // Remove acf_ prefix for cleaner keys
                    $fieldValues = $acfField['value'] ?? [];

                    // Add values to attributes (posts support flexible types)
                    if (!empty($fieldValues)) {
                        if (count($fieldValues) === 1) {
                            $attributesData[$fieldKey] = $fieldValues[0]; // Single value
                        } else {
                            $attributesData[$fieldKey] = $fieldValues; // Multiple values
                        }
                    }
                }
            }
        }

        // Get searchable status
        // Private posts are always non-searchable (chatbot-only)
        // Published posts default to searchable unless explicitly disabled
        $searchableMeta = get_post_meta($postId, '_intucart_searchable', true);
        if ($post->post_status === 'private') {
            $searchable = false; // Private posts are never searchable
        } else {
            $searchable = ($searchableMeta !== 'no'); // Default true unless explicitly set to 'no'
        }

        $metadata = [
            'id'           => (string) $postId,
            'post_type'    => $postType,              // Fixed: was 'postType', cloud expects 'post_type'
            'title'        => $title,
            'content'      => $content,
            'excerpt'      => $excerpt,
            'author_name'  => $author,                // Fixed: was 'author', cloud expects 'author_name'
            'created_at'   => get_post_time('c', true, $post),
            'published_at' => $post->post_status === 'publish' ? get_post_time('c', true, $post) : null,
            'categories'   => $categories,
            'tags'         => $tags,

            'status'       => $post_status,           // Fixed: was 'post_status', cloud expects 'status'
            'post_format'  => $post_format,
            'topic'        => $topic,
            'word_count'   => $word_count,
            'reading_time' => $reading_time,
            'featured_image' => $featured_image,
            'comment_count'  => $comment_count,
            'view_count'     => $view_count,
            'last_modified'  => $last_modified,
            'source'       => 'wordpress',            // Document source for read-only tracking
            'searchable'   => $searchable,            // Controls visibility in search results (true by default)

            'attributes'   => AttributeUtils::processAttributes($attributesData),
        ];

        // Add ACF search content as separate searchableText field (keeps content clean)
        if (!empty($acfData['search_content'])) {
            $metadata['searchableText'] = $acfData['search_content'];
        }

        return apply_filters('intucart_post_embedding_metadata', $metadata, $postId, $post);
    }
}
