<?php

namespace Intucart\Services\Providers\PostTypes;

use Intucart\Services\Logger;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\AcfAutoMapper;
use Intucart\Utils\StringUtils;
use Intucart\Utils\AttributeUtils;

/**
 * Specialized provider for WooCommerce products.
 * Provides product-specific logic for WooCommerce products with enhanced
 * content processing and StringUtils integration.
 */
class WooCommerceProductProvider implements PostTypeProviderInterface
{
    protected Logger $logger;
    protected CacheService $cacheService;
    protected AcfAutoMapper $acfAutoMapper;

    public function __construct(
        Logger $logger,
        CacheService $cacheService,
        AcfAutoMapper $acfAutoMapper
    ) {
        $this->logger = $logger;
        $this->cacheService = $cacheService;
        $this->acfAutoMapper = $acfAutoMapper;
    }

    /**
     * Get a post by ID and type
     *
     * @param int $postId
     * @param string $postType
     * @return \WP_Post|null
     */
    public function getPost(int $postId, string $postType)
    {
        $post = get_post($postId);
        if (!$post || $post->post_type !== $postType) {
            $this->logger->debug('Post not found or type mismatch', [
                'post_id' => $postId,
                'expected_type' => $postType,
                'actual_type' => $post ? $post->post_type : null
            ]);
            return null;
        }
        return $post;
    }

    /**
     * Get multiple posts by type and args
     *
     * @param array $args
     * @param string $postType
     * @return array
     */
    public function getPosts(array $args = [], string $postType = 'post'): array
    {
        $default_args = [
            'post_type' => $postType,
            'post_status' => 'publish',
            'numberposts' => -1
        ];
        $query_args = array_merge($default_args, $args);
        $posts = get_posts($query_args);
        if (empty($posts)) {
            $this->logger->debug('No posts found for type', [
                'post_type' => $postType,
                'args' => $query_args
            ]);
            return [];
        }
        return $posts;
    }

    /**
     * Return flat payload for cloud upsert (no text field)
     */
    public function getCloudPayload(int $postId, string $postType = 'product', int $maxDescLength = 50000): array
    {
        $wcProduct = wc_get_product($postId);
        if (!$wcProduct) {
            $this->logger->debug('Product not found for cloud payload', ['product_id' => $postId]);
            return [];
        }

        return $this->serializePost($wcProduct, $postType, ['_max_desc_length' => $maxDescLength]);
    }

    /**
     * Serialize a WooCommerce product for API responses and OpenSearch mapping.
     *
     * @param mixed $post
     * @param string $postType
     * @param array $additionalData
     * @return array
     */
    public function serializePost(mixed $post, string $postType = 'product', array $additionalData = []): array
    {
        // Allow max description length override via $additionalData
        $maxDescLength = $additionalData['_max_desc_length'] ?? 50000;
        unset($additionalData['_max_desc_length']);

        // Normalize to WC_Product
        if ($post instanceof \WP_Post) {
            $product = wc_get_product($post->ID);
        } elseif ($post instanceof \WC_Product) {
            $product = $post;
        } else {
            $product = wc_get_product($post);
        }

        if (!$product) {
            $this->logger->warning('serializePost: Unable to resolve product', ['input_type' => gettype($post)]);
            return [];
        }

        // Core properties
        $postId        = $product->get_id();
        $name          = StringUtils::cleanText($product->get_name());
        $sku           = StringUtils::cleanText($product->get_sku());
        $type          = $product->get_type();
        $brands        = array_map(function ($brand) {
            return StringUtils::cleanText($brand);
        }, wp_get_post_terms($postId, 'product_brand', ['fields' => 'names']));
        $categories    = array_map(function ($id) {
            return StringUtils::cleanText(get_term($id)->name);
        }, $product->get_category_ids());
        $tags          = array_map(function ($tag) {
            return StringUtils::cleanText($tag);
        }, wp_get_post_terms($postId, 'product_tag', ['fields' => 'names']));
        $short_desc    = StringUtils::cleanText($product->get_short_description());
        $full_desc     = StringUtils::cleanText($product->get_description());

        // Truncate description if it's significantly longer than the limit
        // The cloud service uses intelligent chunking, so we can be more generous
        if (!empty($full_desc) && $full_desc !== $short_desc && strlen($full_desc) > $maxDescLength) {
            $full_desc = StringUtils::truncate($full_desc, $maxDescLength, '...', true);
        }

        $price          = $product->get_price();
        $regular_price  = $product->get_regular_price();
        $sale_price     = $product->get_sale_price();
        $on_sale        = $product->is_on_sale();
        $stock_status   = $product->get_stock_status() ?: 'instock';
        $stock_quantity = $product->get_stock_quantity() ?? 0;
        $total_sales    = $product->get_total_sales();
        $rating         = $product->get_average_rating();
        $review_count   = $product->get_rating_count();
        $weight         = $product->get_weight();
        $weight_unit    = get_option('woocommerce_weight_unit', 'kg');
        $length         = $product->get_length();
        $width          = $product->get_width();
        $height         = $product->get_height();
        $dimension_unit = get_option('woocommerce_dimension_unit', 'cm');
        $image          = wp_get_attachment_image_url($product->get_image_id(), 'woocommerce_thumbnail');
        $url            = $product->get_permalink();

        // Review summary / insights
        $reviewData = $this->getReviewSummaryAndInsights($postId, $review_count);

        // Extract ACF fields for enhanced search capabilities
        $acfData = $this->acfAutoMapper->extractAcfFields($postId, $postType);

        // Attributes
        $attributesData = [];
        foreach ($product->get_attributes() as $attribute) {
            if (!is_object($attribute) || !method_exists($attribute, 'get_visible') || !$attribute->get_visible()) {
                continue;
            }
            $slug = $attribute->get_name();
            $key  = (strpos($slug, 'pa_') === 0) ? substr($slug, 3) : $slug;
            $values = [];
            try {
                if ($attribute->is_taxonomy() && method_exists($attribute, 'get_terms')) {
                    $terms = $attribute->get_terms();
                    if (!empty($terms) && !is_wp_error($terms)) {
                        $values = array_map(function ($term) {
                            return StringUtils::cleanText($term->name);
                        }, $terms);
                    }
                } elseif (method_exists($attribute, 'get_options')) {
                    $rawValues = $attribute->get_options();
                    $values = array_map(function ($value) {
                        return StringUtils::cleanText($value);
                    }, $rawValues);
                }
            } catch (\Exception $e) {
                continue;
            }
            if (!empty($values)) {
                $attributesData[$key] = $values;
            }
        }

        // Integrate ACF fields with existing attributes for cloud compatibility
        if (!empty($acfData['fields'])) {
            // Add ACF values to existing attributes array for cloud filtering/aggregation
            foreach ($acfData['fields'] as $acfField) {
                if ($acfField['filterable'] ?? false) {
                    $fieldKey = str_replace('acf_', '', $acfField['key']); // Remove acf_ prefix for cleaner keys
                    $fieldValues = $acfField['value'] ?? [];

                    // Add each value as a separate attribute (cloud format)
                    foreach ((array) $fieldValues as $value) {
                        if (!empty($value)) {
                            $attributesData[$fieldKey] = (array) ($attributesData[$fieldKey] ?? []);
                            // Clean text values but preserve numeric values
                            $cleanValue = is_string($value) ? StringUtils::cleanText($value) : $value;
                            $attributesData[$fieldKey][] = (string) $cleanValue;
                        }
                    }
                }
            }
        }

        // Variations
        $variationsData = [];
        if ($product->is_type('variable')) {
            $variations = $product instanceof \WC_Product_Variable ? $product->get_available_variations() : [];
            foreach ($variations as $variation) {
                $variationId  = $variation['variation_id'];
                $variationObj = wc_get_product($variationId);
                if (!$variationObj) {
                    continue;
                }
                $variationAttributes = [];
                foreach ($variation['attributes'] as $key => $value) {
                    $attributeName                    = StringUtils::cleanText(wc_attribute_label(str_replace('attribute_', '', $key)));
                    $cleanValue                       = StringUtils::cleanText($value);
                    $variationAttributes[$attributeName] = $cleanValue;
                }
                $variationsData[] = [
                    'id'            => $variationId,
                    'attributes'    => $variationAttributes,
                    'price'         => $variation['display_price'] ?? '',
                    'regular_price' => $variation['display_regular_price'] ?? '',
                    'is_in_stock'   => $variation['is_in_stock'] ?? false,
                    'sku'           => StringUtils::cleanText($variationObj->get_sku()),
                ];
            }
        }

        // Get searchable status
        // Private products are always non-searchable (chatbot-only)
        // Published products default to searchable unless explicitly disabled
        $searchableMeta = get_post_meta($postId, '_intucart_searchable', true);
        $productPost = get_post($postId);
        if ($productPost && $productPost->post_status === 'private') {
            $searchable = false; // Private products are never searchable
        } else {
            $searchable = ($searchableMeta !== 'no'); // Default true unless explicitly set to 'no'
        }

        $metadata = [
            'id'            => (string) $postId,
            'postType'      => $postType,
            'name'          => $name,
            'sku'           => $sku,
            'type'          => $type,
            'content'       => $full_desc,
            'excerpt'       => $short_desc,
            'short_desc'    => $short_desc,
            'full_desc'     => $full_desc,
            'price'         => is_numeric($price) ? (float) $price : null,
            'regular_price' => is_numeric($regular_price) ? (float) $regular_price : null,
            'sale_price'    => is_numeric($sale_price) ? (float) $sale_price : null,
            'on_sale'       => $on_sale,
            'stock_status'  => $stock_status,
            'stock_quantity' => (int) $stock_quantity,
            'total_sales'   => (int) $total_sales,
            'rating'        => is_numeric($rating) ? (float) $rating : null,
            'review_count'  => (int) $review_count,
            'created_at'    => get_post_time('c', true, $postId),
            'categories'    => $categories,
            'tags'          => array_values((array) $tags),
            'image'         => $image,
            'url'           => $url,
            'source'        => 'wordpress',
            'searchable'    => $searchable,
            'variations'    => $variationsData,
            'brands'        => $brands,
            'weight'        => is_numeric($weight) ? (float) $weight : null,
            'weight_unit'   => $weight_unit,
            'dimensions'    => [
                'length' => is_numeric($length) ? (float) $length : null,
                'width'  => is_numeric($width) ? (float) $width : null,
                'height' => is_numeric($height) ? (float) $height : null,
            ],
            'dimension_unit' => $dimension_unit,
            'min_price'      => is_numeric($price) ? (float) $price : null,
            'max_price'      => is_numeric($price) ? (float) $price : null,
            'product_flags'  => array_filter([
                $product->is_downloadable() ? 'downloadable' : null,
                $product->is_virtual() ? 'virtual' : null,
                $product->is_featured() ? 'featured' : null,
                $product->is_on_sale() ? 'on_sale' : null,
                $product->is_purchasable() ? 'purchasable' : null,
                $product->is_sold_individually() ? 'sold_individually' : null,
            ]),

            // Review insights
            'review_summary'            => $reviewData['summary'] ?? null,
            'review_sentiment'          => $reviewData['sentiment'] ?? null,
            'review_key_features'       => $reviewData['key_features'] ?? [],
            'review_pros'               => $reviewData['pros'] ?? [],
            'review_cons'               => $reviewData['cons'] ?? [],
            'review_summary_generated_at' => $reviewData['generated_at'] ?? null,
        ];

        // Add attributes (null if empty, otherwise the data)
        $metadata['attributes'] = AttributeUtils::processAttributes($attributesData);

        // Add ACF search content as separate searchableText field (keeps content clean)
        if (!empty($acfData['search_content'])) {
            $metadata['searchableText'] = $acfData['search_content'];
        }

        // Merge any additional data (after removing control flag)
        if (!empty($additionalData)) {
            $metadata = array_merge($metadata, $additionalData);
        }

        return apply_filters('intucart_product_embedding_metadata', $metadata, $postId, $product);
    }

    /**
     * Get review summary and insights using AI summarization
     *
     * @param int $productId Product ID
     * @param int $reviewCount Number of reviews
     * @return array Review data with summary and insights
     */
    private function getReviewSummaryAndInsights(int $productId, int $reviewCount): array
    {
        // Return empty data if no reviews
        if ($reviewCount === 0) {
            return [
                'summary' => null,
                'sentiment' => null,
                'key_features' => [],
                'pros' => [],
                'cons' => [],
                'generated_at' => null,
            ];
        }

        // TODO: Implement cloud review summarization service
        // For now, return empty data to avoid circular dependency
        // The review summarization calls the cloud API which creates a circular dependency
        // This should be implemented after the cloud review service is available
        return [
            'summary' => null,
            'sentiment' => null,
            'key_features' => [],
            'pros' => [],
            'cons' => [],
            'generated_at' => null,
        ];

        // DISABLED: Cloud review summarization (circular dependency)
        /*
        // Check cache first
        $cacheKey = "intucart_review_summary_{$productId}";
        $cached = get_transient($cacheKey);

        if ($cached !== false) {
            $this->logger->debug('Using cached review summary', [
                'product_id' => $productId,
                'cache_key' => $cacheKey
            ]);
            return $cached;
        }

        try {
            // Get recent reviews (last 50 or all if fewer)
            $reviews = get_comments([
                'post_id' => $productId,
                'status' => 'approve',
                'type' => 'review',
                'number' => 50,
                'orderby' => 'comment_date',
                'order' => 'DESC',
                'meta_query' => [
                    [
                        'key' => 'rating',
                        'value' => 1,
                        'compare' => '>='
                    ]
                ]
            ]);

            if (empty($reviews)) {
                return [
                    'summary' => null,
                    'sentiment' => null,
                    'key_features' => [],
                    'pros' => [],
                    'cons' => [],
                    'generated_at' => null,
                ];
            }

            // Prepare review text for AI summarization
            $reviewTexts = [];
            $ratings = [];

            foreach ($reviews as $review) {
                $rating = get_comment_meta($review->comment_ID, 'rating', true);
                $content = trim($review->comment_content);

                if (!empty($content) && strlen($content) > 10) {
                    $reviewTexts[] = "Rating: {$rating}/5 - " . $content;
                    $ratings[] = (int)$rating;
                }
            }

            if (empty($reviewTexts)) {
                return [
                    'summary' => null,
                    'sentiment' => null,
                    'key_features' => [],
                    'pros' => [],
                    'cons' => [],
                    'generated_at' => null,
                ];
            }

            // Generate AI summary
            $reviewData = $this->generateReviewSummary($reviewTexts, $ratings, $productId);

            // Cache for 24 hours (reviews don't change frequently)
            set_transient($cacheKey, $reviewData, 24 * HOUR_IN_SECONDS);

            $this->logger->info('Generated new review summary', [
                'product_id' => $productId,
                'review_count' => count($reviewTexts),
                'summary_length' => strlen($reviewData['summary'] ?? ''),
                'sentiment' => $reviewData['sentiment'] ?? 'unknown'
            ]);

            return $reviewData;
        } catch (\Exception $e) {
            $this->logger->error('Failed to generate review summary', [
                'product_id' => $productId,
                'error' => $e->getMessage()
            ]);

            // Return empty data on error
            return [
                'summary' => null,
                'sentiment' => null,
                'key_features' => [],
                'pros' => [],
                'cons' => [],
                'generated_at' => null,
            ];
        }
        */
    }

    /**
     * Generate review summary using cloud AI service
     *
     * DISABLED: This method creates a circular dependency where product sync
     * calls the cloud API for review summarization. This should be implemented
     * as a separate service after the cloud review endpoint is available.
     *
     * @param array $reviewTexts Array of review texts with ratings
     * @param array $ratings Array of ratings
     * @param int $productId Product ID for logging
     * @return array Generated review data
     */
    private function generateReviewSummary(array $reviewTexts, array $ratings, int $productId): array
    {
        // DISABLED: Circular dependency - commenting out implementation
        /*
        // Get license key for cloud API
        $licenseKey = get_option('intucart_license_key', '');
        if (empty($licenseKey)) {
            throw new \Exception('License key not configured');
        }

        // Get cloud endpoint
        $endpoint = get_option('intucart_cloud_endpoint', '');
        if (empty($endpoint)) {
            throw new \Exception('Cloud endpoint not configured');
        }

        // Prepare reviews data for cloud service
        $reviews = [];
        for ($i = 0; $i < count($reviewTexts) && $i < count($ratings); $i++) {
            // Extract content from formatted review text (remove "Rating: X/5 - " prefix)
            $content = preg_replace('/^Rating: \d\/5 - /', '', $reviewTexts[$i]);

            $reviews[] = [
                'rating' => (int)$ratings[$i],
                'content' => $content,
                'date' => current_time('c')
            ];
        }

        // Make API call to cloud reviews service
        $response = wp_remote_post($endpoint . '/reviews/summarize', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $licenseKey
            ],
            'body' => json_encode([
                'product_id' => $productId,
                'reviews' => $reviews,
                'force_regenerate' => false
            ]),
            'timeout' => 30
        ]);

        if (is_wp_error($response)) {
            throw new \Exception('API request failed: ' . $response->get_error_message());
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($statusCode !== 200) {
            $errorMessage = $data['message'] ?? $data['error'] ?? 'Unknown error';
            throw new \Exception("API returned error (HTTP {$statusCode}): {$errorMessage}");
        }

        if (!$data || !isset($data['success']) || !$data['success']) {
            throw new \Exception('API returned error: ' . ($data['error'] ?? 'Unknown error'));
        }

        $this->logger->info('Review summary generated via cloud service', [
            'product_id' => $productId,
            'review_count' => count($reviews),
            'cached' => $data['cached'] ?? false,
            'processing_time' => $data['processing_time'] ?? 0,
            'sentiment' => $data['sentiment'] ?? 'unknown'
        ]);

        // Return the cloud service response in our expected format
        return [
            'summary' => $data['summary'] ?? null,
            'sentiment' => $data['sentiment'] ?? 'neutral',
            'key_features' => $data['key_features'] ?? [],
            'pros' => $data['pros'] ?? [],
            'cons' => $data['cons'] ?? [],
            'generated_at' => $data['generated_at'] ?? current_time('c'),
        ];
        */

        // Return empty data to avoid circular dependency
        return [
            'summary' => null,
            'sentiment' => null,
            'key_features' => [],
            'pros' => [],
            'cons' => [],
            'generated_at' => null,
        ];
    }
}
