<?php

namespace Intucart\Services;

use Intucart\Services\Managers\EventManager;
use Intucart\Services\Managers\PostTypeManager;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\Logger;
use Intucart\Services\Licensing\License;
use Intucart\Services\Constants;
use Intucart\Services\AIClientManager;
use Intufind\AI\Exceptions\ApiException;
use Intufind\AI\Exceptions\RateLimitException;
use Intufind\AI\Exceptions\TrialExpiredException;

/**
 * Recommendations service
 * php version 8.1
 *
 * @category Services
 * @package  Intucart
 * @author   Intucart <support@intufind.com>
 * @license  GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://intufind.com
 */
class Recommendations
{
    private PostTypeManager $postTypeManager;
    private EventManager $eventManager;
    private CacheService $cacheService;
    private Logger $logger;
    private License $license;
    private AIClientManager $aiClientManager;

    private const CACHE_GROUP = 'intucart_recommendations';
    private const CACHE_EXPIRATION = 3600; // 1 hour

    /**
     * Constructor
     *
     * @param PostTypeManager $postTypeManager Post type manager instance
     * @param EventManager   $eventManager   Event manager instance
     * @param CacheService   $cacheService   Cache service instance
     * @param Logger         $logger         Logger instance
     * @param License        $license        License service instance
     * @param AIClientManager $aiClientManager AI client manager instance
     */
    public function __construct(
        PostTypeManager $postTypeManager,
        EventManager $eventManager,
        CacheService $cacheService,
        Logger $logger,
        License $license,
        AIClientManager $aiClientManager
    ) {
        $this->postTypeManager = $postTypeManager;
        $this->eventManager = $eventManager;
        $this->cacheService = $cacheService;
        $this->logger = $logger;
        $this->license = $license;
        $this->aiClientManager = $aiClientManager;
    }


    /**
     * Get trending product recommendations using AI SDK
     *
     * @param int $maxRecommendations Maximum number of recommendations
     * @param array $filters Additional filters
     * @return array Trending recommendations
     */
    public function getTrendingRecommendations(int $maxRecommendations = 10, array $filters = []): array
    {
        try {
            $aiClient = $this->aiClientManager->getClient();

            $params = [
                'limit' => $maxRecommendations,
            ];

            if (!empty($filters)) {
                $params['filters'] = $filters;
            }

            $recommendations = $aiClient->recommendations()->getTrending($params);

            return $this->convertSDKRecommendations($recommendations);

        } catch (\Exception $e) {
            $this->logger->warning('Failed to get trending recommendations', [
                'error' => $e->getMessage()
            ]);

            return ['fallback_needed' => true, 'reason' => 'technical_error'];
        }
    }

    /**
     * Get popular product recommendations using AI SDK
     *
     * @param int $maxRecommendations Maximum number of recommendations
     * @param array $filters Additional filters
     * @return array Popular recommendations
     */
    public function getPopularRecommendations(int $maxRecommendations = 10, array $filters = []): array
    {
        try {
            $aiClient = $this->aiClientManager->getClient();

            $params = [
                'limit' => $maxRecommendations,
            ];

            if (!empty($filters)) {
                $params['filters'] = $filters;
            }

            $recommendations = $aiClient->recommendations()->getPopular($params);

            return $this->convertSDKRecommendations($recommendations);

        } catch (\Exception $e) {
            $this->logger->warning('Failed to get popular recommendations', [
                'error' => $e->getMessage()
            ]);

            return ['fallback_needed' => true, 'reason' => 'technical_error'];
        }
    }

    /**
     * Get personalized recommendations for a user using AI SDK
     *
     * @param int $userId User ID
     * @param int $maxRecommendations Maximum number of recommendations
     * @param array $filters Additional filters
     * @return array Personalized recommendations
     */
    public function getPersonalizedRecommendations(int $userId, int $maxRecommendations = 10, array $filters = []): array
    {
        if ($userId <= 0) {
            return ['fallback_needed' => true, 'reason' => 'invalid_user'];
        }

        try {
            $aiClient = $this->aiClientManager->getClient();

            $params = [
                'limit' => $maxRecommendations,
            ];

            if (!empty($filters)) {
                $params['filters'] = $filters;
            }

            $recommendations = $aiClient->recommendations()->getPersonalized((string)$userId, $params);

            return $this->convertSDKRecommendations($recommendations);

        } catch (\Exception $e) {
            $this->logger->warning('Failed to get personalized recommendations', [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);

            return ['fallback_needed' => true, 'reason' => 'technical_error'];
        }
    }

    /**
     * Convert SDK recommendation format to plugin format
     *
     * @param array $sdkRecommendations Recommendations from SDK
     * @return array Converted recommendations
     */
    private function convertSDKRecommendations(array $sdkRecommendations): array
    {
        if (!isset($sdkRecommendations['recommendations']) || !is_array($sdkRecommendations['recommendations'])) {
            return [];
        }

        $convertedRecommendations = [];
        foreach ($sdkRecommendations['recommendations'] as $rec) {
            if (isset($rec['product'])) {
                $product = $rec['product'];
                $convertedRecommendations[] = [
                    'id' => $product->id,
                    'score' => $rec['score'] ?? 0.0,
                    'reason' => $rec['reason'] ?? '',
                    'type' => $rec['type'] ?? 'recommendation',
                    'product' => $product
                ];
            }
        }

        return $convertedRecommendations;
    }

    /**
     * Get product recommendations
     *
     * @param int|array $productId          Product ID or array of Product IDs
     * @param int       $userId             User ID
     * @param float     $similarWeight      Weight for similar products (0-1)
     * @param float     $basketWeight       Weight for basket affinity (0-1)
     * @param float     $userWeight         Weight for user similarity (0-1)
     * @param int       $maxRecommendations Maximum number of recommendations
     * @param bool      $bypassCache        Whether to bypass the cache
     * @param float|null $halfLifeDays      Optional override for half-life days
     *
     * @return array Returns recommendations array, or array with 'fallback_needed' key if usage limits exceeded
     * @throws \Exception If recommendations are not allowed
     */
    public function getRecommendations(
        $productId,
        int $userId,
        float $similarWeight = Constants::DEFAULT_SIMILAR_PRODUCTS_WEIGHT,
        float $basketWeight = Constants::DEFAULT_BASKET_AFFINITY_WEIGHT,
        float $userWeight = Constants::DEFAULT_USER_SIMILARITY_WEIGHT,
        int $maxRecommendations = 10,
        bool $bypassCache = false,
        ?float $halfLifeDays = null
    ): array {
        $this->logger->debug('Starting getRecommendations', [
            'product_id' => $productId,
            'user_id' => $userId,
            'similar_weight' => $similarWeight,
            'basket_weight' => $basketWeight,
            'user_weight' => $userWeight,
            'max_recommendations' => $maxRecommendations,
            'bypass_cache' => $bypassCache,
            'half_life_days' => $halfLifeDays,
        ]);

        // Input validation
        if ($maxRecommendations <= 0) {
            $this->logger->warning('Invalid maxRecommendations value', ['value' => $maxRecommendations]);
            return [];
        }

        // Handle array of product IDs
        $productIds = is_array($productId) ? $productId : [$productId];

        $this->logger->debug('Processing product IDs', [
            'original_product_id' => $productId,
            'is_array' => is_array($productId),
            'product_ids' => $productIds,
        ]);

        // Filter out invalid product IDs
        $validProductIds = array_filter($productIds, function ($id) {
            return is_numeric($id) && $id > 0;
        });

        if (empty($validProductIds)) {
            $this->logger->warning('No valid product IDs provided', [
                'original_product_ids' => $productIds,
                'filtered_product_ids' => $validProductIds,
            ]);
            return [];
        }

        $this->logger->debug('Valid product IDs identified', [
            'valid_product_ids' => $validProductIds,
            'count' => count($validProductIds),
        ]);

        // Validate weights sum to 1.0 (with small float precision tolerance)
        $weightSum = $similarWeight + $basketWeight + $userWeight;
        if (abs($weightSum - 1.0) > 0.0001) {
            throw new \InvalidArgumentException(
                'Weights must sum to 1.0, got: ' . esc_html($weightSum)
            );
        }

        // Get half-life from settings if not explicitly provided
        if ($halfLifeDays === null) {
            $halfLifeDays = $this->getHalfLifeDays();
        }

        // Validate half-life days
        if ($halfLifeDays <= 0) {
            $this->logger->warning('Invalid halfLifeDays value, using default', ['value' => $halfLifeDays]);
            $halfLifeDays = 30.0;
        }

        // Allow third parties to provide a context key (e.g., retailer ID, user role, etc.)
        $contextKey = apply_filters('intucart_context_key', '');

        // Generate cache key based on parameters
        $cache_key = sprintf(
            'recommendations_%s_%d_%.2f_%.2f_%.2f_%d_%.1f%s',
            implode('_', $validProductIds),
            $userId,
            $similarWeight,
            $basketWeight,
            $userWeight,
            $maxRecommendations,
            $halfLifeDays,
            $contextKey
        );

        // If bypassing cache, get fresh recommendations
        if ($bypassCache) {
            return $this->getFreshRecommendations(
                $validProductIds,
                $userId,
                $similarWeight,
                $basketWeight,
                $userWeight,
                $maxRecommendations,
                $halfLifeDays
            );
        }

        // Use stale-while-revalidate pattern
        $recommendations = $this->cacheService->getWithRevalidate(
            $cache_key,
            self::CACHE_GROUP,
            self::CACHE_EXPIRATION,
            [$this, 'getFreshRecommendations'],
            [
                $validProductIds,
                $userId,
                $similarWeight,
                $basketWeight,
                $userWeight,
                $maxRecommendations,
                $halfLifeDays
            ]
        );

        // Check if fallback is needed
        if (isset($recommendations['fallback_needed']) && $recommendations['fallback_needed']) {
            $this->updateFallbackStats($recommendations['reason'] ?? 'unknown');
            return $recommendations;
        }

        // Track successful recommendation
        $this->trackSuccessfulRecommendation();

        $this->logger->info(
            'SWR product recommendations',
            [
                'product_ids' => $validProductIds,
                'recommendations' => array_map(function ($item) {
                    return [
                        'id' => $item['id'],
                        'score' => $item['score']
                    ];
                }, $recommendations)
            ]
        );

        return $recommendations;
    }

    /**
     * Track successful recommendation (not fallback)
     */
    private function trackSuccessfulRecommendation(): void
    {
        // Get current fallback statistics
        $stats = get_transient('intucart_recommendations_fallback_stats');
        if (!$stats) {
            $stats = [
                'total_recommendations' => 0,
                'fallback_recommendations' => 0,
                'usage_limit_fallbacks' => 0,
                'recommendations_disabled_fallbacks' => 0,
                'technical_error_fallbacks' => 0,
                'last_reset' => time()
            ];
        }

        // Reset daily if needed
        if (time() - $stats['last_reset'] > DAY_IN_SECONDS) {
            $stats = [
                'total_recommendations' => 0,
                'fallback_recommendations' => 0,
                'usage_limit_fallbacks' => 0,
                'recommendations_disabled_fallbacks' => 0,
                'technical_error_fallbacks' => 0,
                'last_reset' => time()
            ];
        }

        // Update total counter only
        $stats['total_recommendations']++;

        // Save updated stats
        set_transient('intucart_recommendations_fallback_stats', $stats, DAY_IN_SECONDS + 3600);
    }

    /**
     * Validate product IDs and return valid product objects for reuse
     *
     * @param array $productIds Product IDs to validate
     * @return array Array of valid WC_Product objects keyed by product ID
     */
    private function validateProductIds(array $productIds): array
    {
        $validProducts = [];

        foreach ($productIds as $productId) {
            if (!is_numeric($productId) || $productId <= 0) {
                continue;
            }

            $product = wc_get_product($productId);
            if ($product && $product->exists() && $product->get_status() === 'publish') {
                $validProducts[$productId] = $product;
            }
        }

        if (empty($validProducts)) {
            $this->logger->info('No valid products found for recommendations', [
                'product_ids' => $productIds
            ]);
        }

        return $validProducts;
    }

    /**
     * Get fresh recommendations without caching
     *
     * @param array $productIds         Product IDs
     * @param int   $userId             User ID
     * @param float $similarWeight      Weight for similar products
     * @param float $basketWeight       Weight for basket affinity
     * @param float $userWeight         Weight for user similarity
     * @param int   $maxRecommendations Maximum number of recommendations
     * @param float $halfLifeDays       Half-life in days
     *
     * @return array Returns recommendations array, or array with 'fallback_needed' key if usage limits exceeded
     */
    public function getFreshRecommendations(
        array $productIds,
        int $userId,
        float $similarWeight,
        float $basketWeight,
        float $userWeight,
        int $maxRecommendations,
        float $halfLifeDays
    ): array {
        // Validate product IDs once and get product objects for reuse
        $validProducts = $this->validateProductIds($productIds);

        if (empty($validProducts)) {
            return [];
        }

        // For now, only support single product recommendations via cloud service
        // TODO: Extend cloud service to support multiple products
        if (count($validProducts) === 1) {
            $productId = array_keys($validProducts)[0];

            try {
                // Try cloud service first
                $cloudRecommendations = $this->getCloudRecommendations(
                    $productId,
                    $userId,
                    $similarWeight,
                    $basketWeight,
                    $userWeight,
                    $maxRecommendations,
                    array_keys($validProducts) // Exclude the original product
                );

                // Check if cloud service returned fallback needed
                if (isset($cloudRecommendations['fallback_needed']) && $cloudRecommendations['fallback_needed']) {
                    $this->logger->info('Cloud recommendations unavailable, no fallback available', [
                        'product_id' => $productId,
                        'reason' => $cloudRecommendations['reason'] ?? 'unknown'
                    ]);

                    // Return fallback needed response
                    return $cloudRecommendations;
                }

                // Cloud service succeeded
                $this->logger->info('Cloud recommendations successful', [
                    'product_id' => $productId,
                    'recommendations_count' => count($cloudRecommendations)
                ]);

                return $cloudRecommendations;
            } catch (\Exception $e) {
                $this->logger->warning('Cloud recommendations failed, no fallback available', [
                    'product_id' => $productId,
                    'error' => $e->getMessage()
                ]);

                // Return fallback needed response
                return ['fallback_needed' => true, 'reason' => 'technical_error'];
            }
        }

        // Multiple products - not supported without local computation
        $this->logger->info('Multiple products requested, not supported without local computation', [
            'product_ids' => array_keys($validProducts)
        ]);

        return ['fallback_needed' => true, 'reason' => 'multiple_products_not_supported'];
    }

    /**
     * Get recommendations from cloud service using AI SDK
     *
     * @param string $productId Product ID
     * @param int $userId User ID
     * @param float $similarWeight Weight for similar products
     * @param float $basketWeight Weight for basket affinity
     * @param float $userWeight Weight for user similarity
     * @param int $maxRecommendations Maximum number of recommendations
     * @param array $excludeProductIds Product IDs to exclude
     * @return array Cloud recommendations or fallback indicator
     */
    private function getCloudRecommendations(
        string $productId,
        int $userId,
        float $similarWeight,
        float $basketWeight,
        float $userWeight,
        int $maxRecommendations,
        array $excludeProductIds = []
    ): array {
        try {
            $aiClient = $this->aiClientManager->getClient();

            // Get basket affinity data from local analysis
            $basketAffinity = $this->getBasketAffinityData($productId);

            // Get complementary products from local analysis
            $complimentaryProducts = $this->getComplimentaryProducts($productId);

            // Prepare recommendation parameters
            $params = [
                'product_id' => $productId,
                'limit' => $maxRecommendations,
            ];

            // Add user ID if provided
            if ($userId > 0) {
                $params['user_id'] = (string)$userId;
            }

            // Add exclusion filters (cloud will map 'id' to 'external_id' internally)
            if (!empty($excludeProductIds)) {
                $params['excludeProductIds'] = array_map('strval', $excludeProductIds);
            }

            // Add strategy weights as metadata
            $params['weights'] = [
                'similarProducts' => $similarWeight,
                'basketAffinity' => $basketWeight,
                'memoryBased' => $userWeight,
            ];

            // Add local analysis data
            if (!empty($basketAffinity)) {
                $params['basketAffinity'] = $basketAffinity;
            }

            if (!empty($complimentaryProducts)) {
                $params['complimentaryProducts'] = $complimentaryProducts;
            }

            // Get recommendations using the SDK
            $recommendations = $aiClient->recommendations()->getForProduct($productId, $params);

            // Check if we got valid recommendations
            if (isset($recommendations['recommendations']) && is_array($recommendations['recommendations'])) {
                // Convert SDK format to expected format
                $convertedRecommendations = [];
                foreach ($recommendations['recommendations'] as $rec) {
                    if (isset($rec['product'])) {
                        $product = $rec['product'];
                        $convertedRecommendations[] = [
                            'id' => $product->id,
                            'score' => $rec['score'] ?? 0.0,
                            'reason' => $rec['reason'] ?? '',
                            'product' => $product
                        ];
                    } else {
                        $this->logger->warning('Recommendation missing product data', [
                            'product_id' => $productId,
                            'rec_keys' => array_keys($rec),
                        ]);
                    }
                }

                return $convertedRecommendations;
            }

            $this->logger->warning('AI SDK returned no valid recommendations', [
                'product_id' => $productId,
            ]);

            return ['fallback_needed' => true, 'reason' => 'no_recommendations'];

        } catch (RateLimitException $e) {
            $this->logger->warning('AI SDK rate limit exceeded', [
                'product_id' => $productId,
                'error' => $e->getMessage()
            ]);

            return ['fallback_needed' => true, 'reason' => 'usage_limit_exceeded'];

        } catch (TrialExpiredException $e) {
            $this->logger->warning('AI SDK trial expired', [
                'product_id' => $productId,
                'error' => $e->getMessage(),
                'trial_ended_at' => $e->getTrialEndedAt()
            ]);

            // Cache the expired status so subsequent requests don't hit the API
            $this->license->markTrialExpired($e);

            return ['fallback_needed' => true, 'reason' => 'trial_expired'];

        } catch (ApiException $e) {
            $this->logger->warning('AI SDK API error', [
                'product_id' => $productId,
                'status_code' => $e->getStatusCode(),
                'error' => $e->getMessage(),
                'error_details' => $e->getErrorDetails(),
                'exception_class' => get_class($e)
            ]);

            // Check if this is a tier limitation
            if ($e->getStatusCode() === 403) {
                return ['fallback_needed' => true, 'reason' => 'recommendations_disabled'];
            }

            return ['fallback_needed' => true, 'reason' => 'technical_error'];

        } catch (\Exception $e) {
            $this->logger->error('AI SDK unexpected error', [
                'product_id' => $productId,
                'error' => $e->getMessage(),
                'exception_class' => get_class($e),
                'trace' => $e->getTraceAsString()
            ]);

            return ['fallback_needed' => true, 'reason' => 'technical_error'];
        }
    }

    /**
     * Get basket affinity data for cloud service
     *
     * @param string $productId Product ID
     * @return array Basket affinity data with frequency
     */
    private function getBasketAffinityData(string $productId): array
    {
        try {
            // Get product associations with frequency data
            $associations = $this->postTypeManager->getProductAssociations(
                (int)$productId,
                $this->eventManager,
                $this->getHalfLifeDays()
            );

            // Convert to cloud service format
            $basketAffinity = [];
            foreach ($associations as $association) {
                if (isset($association['id']) && isset($association['score'])) {
                    $basketAffinity[] = [
                        'productId' => (string)$association['id'],
                        'frequency' => (float)$association['score'] * 100 // Convert score to frequency
                    ];
                }
            }

            return $basketAffinity;
        } catch (\Exception $e) {
            $this->logger->warning('Failed to get basket affinity data', [
                'product_id' => $productId,
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    /**
     * Get complementary products for cloud service
     *
     * @param string $productId Product ID
     * @return array Complementary product IDs
     */
    private function getComplimentaryProducts(string $productId): array
    {
        // TODO: Implement proper complementary products logic
        // For now, return empty array.
        return [];
    }

    /**
     * Get the current half-life setting in days
     *
     * @return float The half-life in days
     */
    public function getHalfLifeDays(): float
    {
        return (float) get_option(Constants::RECOMMENDATION_HALF_LIFE_OPTION, 30.0);
    }

    /**
     * Update fallback statistics for usage limits.
     * This method is called when a fallback is needed.
     *
     * @param string $reason The reason for the fallback.
     */
    private function updateFallbackStats(string $reason): void
    {
        // Get current fallback statistics
        $stats = get_transient('intucart_recommendations_fallback_stats');
        if (!$stats) {
            $stats = [
                'total_recommendations' => 0,
                'fallback_recommendations' => 0,
                'usage_limit_fallbacks' => 0,
                'recommendations_disabled_fallbacks' => 0,
                'technical_error_fallbacks' => 0,
                'last_reset' => time()
            ];
        }

        // Reset daily if needed
        if (time() - $stats['last_reset'] > DAY_IN_SECONDS) {
            $stats = [
                'total_recommendations' => 0,
                'fallback_recommendations' => 0,
                'usage_limit_fallbacks' => 0,
                'recommendations_disabled_fallbacks' => 0,
                'technical_error_fallbacks' => 0,
                'last_reset' => time()
            ];
        }

        // Update counters
        $stats['total_recommendations']++;
        $stats['fallback_recommendations']++;

        // Update specific reason counters
        switch ($reason) {
            case 'usage_limit_exceeded':
                $stats['usage_limit_fallbacks']++;
                break;
            case 'recommendations_disabled':
                $stats['recommendations_disabled_fallbacks']++;
                break;
            case 'technical_error':
            default:
                $stats['technical_error_fallbacks']++;
                break;
        }

        // Save updated stats
        set_transient('intucart_recommendations_fallback_stats', $stats, DAY_IN_SECONDS + 3600);

        // Log the fallback
        $this->logger->warning('Recommendations fallback triggered', [
            'reason' => $reason,
            'fallback_percentage' => $stats['total_recommendations'] > 0 ?
                round(($stats['fallback_recommendations'] / $stats['total_recommendations']) * 100, 1) : 0
        ]);

        // Check if we should show admin notice
        $this->checkAdminNotice($stats);
    }

    /**
     * Check if admin notice should be shown for high fallback usage
     *
     * @param array $stats Current fallback statistics
     */
    private function checkAdminNotice(array $stats): void
    {
        // Only show notice if we have significant usage and high fallback rate
        if ($stats['total_recommendations'] < 10) {
            return;
        }

        $fallback_percentage = ($stats['fallback_recommendations'] / $stats['total_recommendations']) * 100;

        // Show notice if more than 50% of recommendations are falling back
        if ($fallback_percentage > 50) {
            // Check if we've already shown notice today
            $notice_shown = get_transient('intucart_recommendations_fallback_notice_shown');
            if (!$notice_shown) {
                // Schedule admin notice
                add_action('admin_notices', function () use ($stats, $fallback_percentage) {
                    $this->showFallbackAdminNotice($stats, $fallback_percentage);
                });

                // Mark notice as shown for today
                set_transient('intucart_recommendations_fallback_notice_shown', true, DAY_IN_SECONDS);
            }
        }
    }

    /**
     * Show admin notice for high fallback usage
     *
     * @param array $stats Current fallback statistics
     * @param float $fallback_percentage Percentage of fallbacks
     */
    private function showFallbackAdminNotice(array $stats, float $fallback_percentage): void
    {
        $message = sprintf(
            'Intucart Recommendations: %.1f%% of recommendations are using fallback today (%d/%d). ',
            $fallback_percentage,
            $stats['fallback_recommendations'],
            $stats['total_recommendations']
        );

        $details = [];
        if ($stats['usage_limit_fallbacks'] > 0) {
            $details[] = sprintf('%d due to usage limits', $stats['usage_limit_fallbacks']);
        }
        if ($stats['recommendations_disabled_fallbacks'] > 0) {
            $details[] = sprintf('%d due to recommendations being disabled', $stats['recommendations_disabled_fallbacks']);
        }
        if ($stats['technical_error_fallbacks'] > 0) {
            $details[] = sprintf('%d due to technical errors', $stats['technical_error_fallbacks']);
        }

        if (!empty($details)) {
            $message .= 'Breakdown: ' . implode(', ', $details) . '.';
        }

        echo '<div class="notice notice-warning"><p>' . esc_html($message) . '</p></div>';
    }

    /**
     * Get fallback statistics for recommendations
     *
     * @return array Statistics about fallback usage
     */
    public function getFallbackStats(): array
    {
        $stats = get_transient('intucart_recommendations_fallback_stats');
        if (!$stats) {
            $stats = [
                'total_recommendations' => 0,
                'fallback_recommendations' => 0,
                'usage_limit_fallbacks' => 0,
                'recommendations_disabled_fallbacks' => 0,
                'technical_error_fallbacks' => 0,
                'last_reset' => time()
            ];
        }
        return $stats;
    }

    /**
     * Get the appropriate product provider for similarity search
     *
     * @param \WC_Product $product Product object
     * @return mixed Product provider instance
     */
    private function getProductProvider(\WC_Product $product)
    {
        // Get the WooCommerce product provider from the post type manager
        // This assumes PostTypeManager has a method to get providers
        return $this->postTypeManager->getProviderForType('product');
    }
}
