<?php

namespace Intucart\Services;

use Intucart\Services\AIClientManager;
use Intucart\Services\Logger;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\Analytics;
use Intucart\Services\Constants;
use Intucart\Services\Licensing\License;
use Intufind\AI\Exceptions\TrialExpiredException;

/**
 * Search service class
 */
class Search
{
    private AIClientManager $aiClientManager;
    private Logger $logger;
    private CacheService $cache;
    private Analytics $analytics;
    private License $license;
    private const CACHE_GROUP = 'intucart_search';

    /**
     * Get cache expiration time from settings
     */
    private function getCacheExpiration(): int
    {
        return (int) get_option(Constants::SEARCH_CACHE_DURATION_OPTION, 3600);
    }


    /**
     * Initialize hooks
     *
     * @return void
     */
    public function initialize(): void
    {
        add_action(Constants::SEMANTIC_SEARCH_ACTION, [$this, 'handleSearchHook'], 10, 5);
    }

    /**
     * Handle the search hook
     *
     * @param string $query Search query
     * @param int    $limit Maximum number of results (optional)
     * @param bool   $bypass_cache Whether to bypass cache (optional)
     * @param string $postType Post type (optional, default 'product')
     * @param bool   $useAgent Whether to use AI agent for complex queries (optional, default false)
     *
     * @return array Array of search results
     */
    public function handleSearchHook(string $query, int $limit = 10, bool $bypass_cache = false, string $postType = 'product', bool $useAgent = false): array
    {
        try {
            $results = $this->searchPostTypes($query, $limit, [], $bypass_cache, $postType, $useAgent);
            return apply_filters('intucart_semantic_search_results', $results, $query, $limit, $postType);
        } catch (\Exception $e) {
            $this->logger->error('Intucart semantic search failed: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Constructor
     *
     * @param AIClientManager $aiClientManager AI Client Manager service
     * @param Logger       $logger Logger service
     * @param CacheService $cache Cache service
     * @param Analytics    $analytics Analytics service
     */
    public function __construct(
        AIClientManager $aiClientManager,
        Logger $logger,
        CacheService $cache,
        Analytics $analytics,
        License $license
    ) {
        $this->aiClientManager = $aiClientManager;
        $this->logger = $logger;
        $this->cache = $cache;
        $this->analytics = $analytics;
        $this->license = $license;
    }

    /**
     * Get fallback search statistics
     *
     * @return array Statistics about fallback usage
     */
    public function getFallbackStats(): array
    {
        $stats = get_transient('intucart_fallback_stats');
        if (!$stats) {
            $stats = [
                'total_searches' => 0,
                'fallback_searches' => 0,
                'usage_limit_fallbacks' => 0,
                'semantic_disabled_fallbacks' => 0,
                'exception_fallbacks' => 0,
                'no_api_key_fallbacks' => 0,
                'last_reset' => time()
            ];
        }
        return $stats;
    }

    /**
     * Update fallback statistics
     *
     * @param string $fallback_reason The reason for fallback
     */
    private function updateFallbackStats(string $fallback_reason = ''): void
    {
        $stats = $this->getFallbackStats();
        $stats['total_searches']++;

        if (!empty($fallback_reason)) {
            $stats['fallback_searches']++;

            switch ($fallback_reason) {
                case 'usage_limit_exceeded':
                    $stats['usage_limit_fallbacks']++;
                    break;
                case 'semantic_search_disabled':
                    $stats['semantic_disabled_fallbacks']++;
                    break;
                case 'exception':
                    $stats['exception_fallbacks']++;
                    break;
                case 'no_api_key':
                    $stats['no_api_key_fallbacks'] = ($stats['no_api_key_fallbacks'] ?? 0) + 1;
                    break;
            }
        }

        // Reset stats daily
        if (time() - $stats['last_reset'] > DAY_IN_SECONDS) {
            $stats = [
                'total_searches' => 1,
                'fallback_searches' => !empty($fallback_reason) ? 1 : 0,
                'usage_limit_fallbacks' => $fallback_reason === 'usage_limit_exceeded' ? 1 : 0,
                'semantic_disabled_fallbacks' => $fallback_reason === 'semantic_search_disabled' ? 1 : 0,
                'exception_fallbacks' => $fallback_reason === 'exception' ? 1 : 0,
                'no_api_key_fallbacks' => $fallback_reason === 'no_api_key' ? 1 : 0,
                'last_reset' => time()
            ];
        }

        set_transient('intucart_fallback_stats', $stats, DAY_IN_SECONDS);

        // Show admin notice if fallback usage is high
        if ($stats['total_searches'] >= 10 && ($stats['fallback_searches'] / $stats['total_searches']) > 0.5) {
            $this->scheduleAdminNotice($fallback_reason, $stats);
        }
    }

    /**
     * Schedule an admin notice about high fallback usage
     *
     * @param string $fallback_reason Primary reason for fallbacks
     * @param array $stats Current fallback statistics
     */
    private function scheduleAdminNotice(string $fallback_reason, array $stats): void
    {
        // Only show notice once per day
        if (get_transient('intucart_fallback_notice_shown')) {
            return;
        }

        $notice_data = [
            'fallback_reason' => $fallback_reason,
            'fallback_percentage' => round(($stats['fallback_searches'] / $stats['total_searches']) * 100, 1),
            'usage_limit_fallbacks' => $stats['usage_limit_fallbacks'],
            'semantic_disabled_fallbacks' => $stats['semantic_disabled_fallbacks'],
            'exception_fallbacks' => $stats['exception_fallbacks'],
            'no_api_key_fallbacks' => $stats['no_api_key_fallbacks'] ?? 0
        ];

        set_transient('intucart_fallback_notice_data', $notice_data, DAY_IN_SECONDS);
        set_transient('intucart_fallback_notice_shown', true, DAY_IN_SECONDS);

        // Hook to show admin notice
        add_action('admin_notices', function () use ($notice_data) {
            $this->showFallbackAdminNotice($notice_data);
        });
    }

    /**
     * Show admin notice about fallback usage
     *
     * @param array $notice_data Notice data
     */
    private function showFallbackAdminNotice(array $notice_data): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        $message = sprintf(
            __('Intucart Search Notice: %s%% of searches are using fallback to native WordPress search.', 'intufind'),
            $notice_data['fallback_percentage']
        );

        $details = [];
        if ($notice_data['usage_limit_fallbacks'] > 0) {
            $details[] = sprintf(__('%d due to usage limits', 'intufind'), $notice_data['usage_limit_fallbacks']);
        }
        if ($notice_data['semantic_disabled_fallbacks'] > 0) {
            $details[] = sprintf(__('%d due to semantic search being disabled', 'intufind'), $notice_data['semantic_disabled_fallbacks']);
        }
        if ($notice_data['exception_fallbacks'] > 0) {
            $details[] = sprintf(__('%d due to technical errors', 'intufind'), $notice_data['exception_fallbacks']);
        }
        if (($notice_data['no_api_key_fallbacks'] ?? 0) > 0) {
            $details[] = sprintf(__('%d due to missing/invalid API key', 'intufind'), $notice_data['no_api_key_fallbacks']);
        }

        if (!empty($details)) {
            $message .= ' ' . sprintf(__('Breakdown: %s.', 'intufind'), implode(', ', $details));
        }

        echo '<div class="notice notice-warning is-dismissible">';
        echo '<p>' . $message . '</p>';
        echo '<p>' . sprintf(
            __('Consider <a href="%s">upgrading your plan</a> or <a href="%s">checking your settings</a> to restore full semantic search functionality.', 'intufind'),
            admin_url('admin.php?page=intucart-settings'),
            admin_url('admin.php?page=intucart-settings')
        ) . '</p>';
        echo '</div>';
    }

    /**
     * Search products using natural language query (cloud-based)
     *
     * @param string $query Search query in natural language
     * @param int    $limit Maximum number of results to return
     * @param array<string, mixed>  $filters Optional filters as associative array (e.g., ['categories' => ['electronics'], 'stock_status' => 'instock'])
     * @param bool   $bypassCache Whether to bypass the cache
     * @param string $postType Post type (optional, default 'product')
     * @param bool   $useAgent Whether to use AI agent for complex queries (optional, default false)
     * @param string $searchContext Search context: 'native' for WordPress native search, 'widget' for search widget (optional, default 'native')
     *
     * @return array Array of matching products with scores, or empty array if fallback needed
     */
    public function searchPostTypes(
        string $query,
        int $limit = 10,
        array $filters = [],
        bool $bypassCache = false,
        string $postType = 'product',
        bool $useAgent = false,
        string $searchContext = 'native'
    ): array {
        // Early check: Skip cloud search if API key is not configured
        if (!$this->aiClientManager->hasApiKey()) {
            $this->logger->debug('Search service: No API key configured, using native search fallback');
            $this->updateFallbackStats('no_api_key');
            return [
                'fallback_needed' => true,
                'fallback_reason' => 'no_api_key'
            ];
        }

        try {
            $this->logger->info('Search service: searchPostTypes called', [
                'query' => $query,
                'limit' => $limit,
                'postType' => $postType,
                'filters' => $filters,
                'bypassCache' => $bypassCache,
                'useAgent' => $useAgent
            ]);

            $contextKey = apply_filters('intucart_context_key', '');
            $cacheKey = md5($query . $limit . $postType . wp_json_encode($filters) . $contextKey . ($useAgent ? 'agent' : 'basic'));
            if (!$bypassCache) {
                $cached = $this->cache->get($cacheKey, self::CACHE_GROUP);
                if ($cached !== false) {
                    $this->logger->info('Search service: returning cached results', ['count' => count($cached)]);
                    $this->updateFallbackStats(); // Track as successful search
                    return $cached;
                }
            }
            $baseFilters = $this->getSearchFilters();
            $mergedFilters = array_merge($baseFilters, $filters);
            
            if (empty($mergedFilters)) {
                // Send empty object instead of empty array
                $searchFilters = new \stdClass();
            } else {
                // Ensure we have an associative array
                $searchFilters = $mergedFilters;
            }

            $aiClient = $this->aiClientManager->getClient();
            
            $searchParams = [
                'text' => $query,
                'limit' => $limit,
                'filters' => $searchFilters,
                'searchContext' => $searchContext
            ];

            // Choose the appropriate service based on post type
            if ($postType === 'product') {
                $service = $aiClient->products();
            } else {
                $service = $aiClient->posts();
            }
            
            // Use agent-enhanced search if requested
            if ($useAgent) {
                $searchResult = $service->searchWithAgent($searchParams);
            } else {
                $searchResult = $service->search($searchParams);
            }

            $results = $searchResult->getResults();

            // Update statistics for successful search
            $this->updateFallbackStats();

            $this->analytics->trackSearch($query, count($results));
            $this->cache->set(
                $cacheKey,
                $results,
                self::CACHE_GROUP,
                $this->getCacheExpiration()
            );

            return $results;
        } catch (TrialExpiredException $e) {
            $this->logger->warning(
                'Trial expired, falling back to native search',
                [
                    'error' => $e->getMessage(),
                    'trial_ended_at' => $e->getTrialEndedAt(),
                    'query' => $query
                ]
            );

            // Cache the expired status so subsequent requests don't hit the API
            $this->license->markTrialExpired($e);

            // Update fallback statistics
            $this->updateFallbackStats('trial_expired');

            // Track the fallback event
            $this->analytics->trackSearch($query, 0);

            return [
                'fallback_needed' => true,
                'fallback_reason' => 'trial_expired',
                'error_message' => 'Your free trial has ended. Subscribe to continue using AI-powered search.'
            ];
        } catch (\Exception $e) {
            $this->logger->error(
                'Product search failed, letting native search handle it',
                [
                    'error' => $e->getMessage(),
                    'query' => $query,
                    'post_type' => $postType
                ]
            );

            // Update fallback statistics
            $this->updateFallbackStats('exception');

            // Track the fallback event (0 results indicates a failure)
            $this->analytics->trackSearch($query, 0);

            // Return fallback indicator - SearchOverride will not modify the query
            return [
                'fallback_needed' => true,
                'fallback_reason' => 'exception',
                'error_message' => $e->getMessage()
            ];
        }
    }

    /**
     * Get search filters based on context key
     *
     * @return array Array of filters for cloud query
     */
    private function getSearchFilters(): array
    {
        $default_filters = [];
        if (get_option('woocommerce_hide_out_of_stock_items') === 'yes') {
            $default_filters = ['stock_status' => 'instock'];
        }
        
        return apply_filters('intucart_product_stock_filters', $default_filters, '');
    }

}
