<?php

namespace Intucart\Services;

use Intucart\Services\Logger;
use Intucart\Services\AIClientManager;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\Licensing\License;
use InvalidArgumentException;

/**
 * ThreadService - WordPress service for managing conversation threads
 *
 * Provides admin functionality for viewing, managing, and analyzing
 * conversations stored in the cloud. Follows the same patterns as
 * PostTypeManager and integrates with AIClientManager service.
 *
 * Key Features:
 * - Search conversations with advanced filtering
 * - Get detailed conversation content
 * - Delete conversations
 * - Analytics and reporting
 * - Export conversation data
 * - Caching for performance
 * - Error handling and logging
 */
class ThreadService
{
    /**
     * Cache configuration constants
     */
    private const CACHE_TTL_THREAD_SEARCH = 300;    // 5 minutes cache for search results
    private const CACHE_TTL_THREAD_DETAIL = 600;    // 10 minutes cache for thread details
    private const CACHE_TTL_ANALYTICS = 900;        // 15 minutes cache for analytics

    /**
     * API configuration constants
     */
    private const DEFAULT_SEARCH_LIMIT = 10;
    private const MAX_SEARCH_LIMIT = 100;
    private const DEFAULT_EXPORT_LIMIT = 1000;
    private const MAX_EXPORT_LIMIT = 10000;

    private Logger $logger;
    private AIClientManager $aiClientManager;
    private CacheService $cache;
    private License $license;

    /**
     * Constructor
     *
     * @param Logger           $logger           Logger service
     * @param AIClientManager  $aiClientManager  AIClientManager service
     * @param CacheService     $cache            Cache service
     * @param License          $license          License service
     */
    public function __construct(
        Logger $logger,
        AIClientManager $aiClientManager,
        CacheService $cache,
        License $license
    ) {
        $this->logger = $logger;
        $this->aiClientManager = $aiClientManager;
        $this->cache = $cache;
        $this->license = $license;
    }

    /**
     * Search conversations with advanced filtering and pagination
     *
     * @param array $request Search request parameters
     *   - query: Optional text search across conversation content
     *   - limit: Number of results to return (default 10, max 100)
     *   - offset: Pagination offset (default 0)
     *   - userId: Filter by specific user ID
     *   - dateFrom: ISO date string - conversations after this date
     *   - dateTo: ISO date string - conversations before this date
     *   - minMessages: Minimum message count
     *   - maxMessages: Maximum message count
     *   - minTokens: Minimum token count
     *   - maxTokens: Maximum token count
     *   - sortBy: Sort field (timestamp, message_count, token_count)
     *   - sortOrder: Sort direction (asc, desc)
     *
     * @return array Search results with pagination info
     * @throws InvalidArgumentException If request parameters are invalid
     */
    public function searchThreads(array $request = []): array
    {
        // Validate and normalize request
        $normalizedRequest = $this->validateSearchRequest($request);

        // Generate cache key based on request parameters
        $cacheKey = 'thread_search_' . md5(serialize($normalizedRequest));
        $cached = $this->cache->get($cacheKey, 'threads');
        if ($cached !== false && $cached !== null) {
            return $cached;
        }

        try {
            // Use AI SDK thread service for search
            // License validation happens at the cloud API level (threadHandler.ts)
            $results = $this->aiClientManager->getClient()->threads()->search($normalizedRequest);

            // Cache successful results
            $this->cache->set($cacheKey, $results, 'threads', self::CACHE_TTL_THREAD_SEARCH);

            return $results;
        } catch (\Exception $e) {
            $this->logger->error('Thread search failed', [
                'error' => $e->getMessage(),
                'request_params' => $normalizedRequest
            ]);
            throw $e;
        }
    }

    /**
     * Get detailed conversation by thread ID
     *
     * @param string $threadId Thread ID (external_id from cloud)
     * @return array|null Thread details or null if not found
     * @throws InvalidArgumentException If thread ID is invalid
     */
    public function getThreadById(string $threadId): ?array
    {
        if (empty($threadId)) {
            throw new InvalidArgumentException('Thread ID is required');
        }

        // Check cache first
        $cacheKey = "thread_detail_{$threadId}";
        $cached = $this->cache->get($cacheKey, 'threads');
        if ($cached !== false && $cached !== null) {
            return $cached;
        }

        try {
            // Use AI SDK thread service to get thread details
            // License validation happens at the cloud API level (threadHandler.ts)
            $threadDetail = $this->aiClientManager->getClient()->threads()->get($threadId);

            // Cache successful result
            if ($threadDetail) {
                $this->cache->set($cacheKey, $threadDetail, 'threads', self::CACHE_TTL_THREAD_DETAIL);
            }

            return $threadDetail;
        } catch (\Exception $e) {
            $this->logger->error('Failed to fetch thread details', [
                'thread_id' => $threadId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Get message history for a thread (for polling during live agent handoff)
     *
     * Returns formatted message history including:
     * - User messages
     * - AI assistant messages
     * - Human agent messages (with agent metadata)
     *
     * Messages are cached with a short TTL (30 seconds) to support
     * polling while minimizing API calls during active conversations.
     *
     * @param string $threadId Thread ID (external_id from cloud)
     * @return array Message history with metadata
     *   - threadId: The thread ID
     *   - messages: Array of message objects with role, content, timestamp, source, agentInfo
     *   - count: Total number of messages
     * @throws InvalidArgumentException If thread ID is invalid
     */
    public function getMessages(string $threadId): array
    {
        if (empty($threadId)) {
            throw new InvalidArgumentException('Thread ID is required');
        }

        // No caching for live agent messages - need real-time updates
        // The backend (SAM/Express) already has Redis caching, so this is just
        // adding unnecessary latency for live agent conversations
        try {
            // Use AI SDK thread service to get message history
            // License validation happens at the cloud API level (threadHandler.ts)
            $messages = $this->aiClientManager->getClient()->threads()->getMessages($threadId);

            return $messages;
        } catch (\Exception $e) {
            $this->logger->error('Failed to fetch thread messages', [
                'thread_id' => $threadId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * End an active live agent handoff session
     *
     * @param string $threadId Thread ID
     * @return array Response with {success: bool, message: string}
     * @throws InvalidArgumentException If thread ID is empty
     * @throws \Exception If API call fails
     */
    public function endHandoff(string $threadId): array
    {
        if (empty($threadId)) {
            throw new InvalidArgumentException('Thread ID is required');
        }

        try {
            $this->logger->info('Ending live agent handoff', ['thread_id' => $threadId]);

            // Use AI SDK thread service to end the handoff
            // This will clear Redis session, update OpenSearch, and stop polling
            $result = $this->aiClientManager->getClient()->threads()->endHandoff($threadId);

            $this->logger->info('Live agent handoff ended successfully', ['thread_id' => $threadId]);

            return $result;
        } catch (\Exception $e) {
            $this->logger->error('Failed to end live agent handoff', [
                'thread_id' => $threadId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Delete a conversation thread
     *
     * @param string $threadId Thread ID to delete
     * @return array Response from cloud service with 'id' and 'message'
     * @throws InvalidArgumentException If thread ID is invalid
     */
    public function deleteThread(string $threadId): array
    {
        if (empty($threadId)) {
            throw new InvalidArgumentException('Thread ID is required');
        }

        try {
            // Use AI SDK thread service to delete thread
            // License validation happens at the cloud API level (threadHandler.ts)
            $result = $this->aiClientManager->getClient()->threads()->delete($threadId);

            // Clear cached data for this thread
            $this->invalidateThreadCache($threadId);

            return $result;
        } catch (\Exception $e) {
            $this->logger->error('Failed to delete thread', [
                'thread_id' => $threadId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Get conversation analytics and insights
     *
     * @return array Analytics data
     */
    public function getThreadAnalytics(): array
    {
        // Check cache first
        $cacheKey = 'thread_analytics';
        $cached = $this->cache->get($cacheKey, 'threads');
        if ($cached !== false && $cached !== null) {
            return $cached;
        }

        try {
            // Use AI SDK thread service to get analytics
            // License validation happens at the cloud API level (threadHandler.ts)
            $analytics = $this->aiClientManager->getClient()->threads()->getAnalytics();

            // Cache successful result
            $this->cache->set($cacheKey, $analytics, 'threads', self::CACHE_TTL_ANALYTICS);

            return $analytics;
        } catch (\Exception $e) {
            $this->logger->error('Failed to fetch thread analytics', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Export conversation data
     *
     * @param array $request Export request parameters
     *   - format: Export format (json, csv) - default: json
     *   - dateFrom: ISO date string - conversations after this date
     *   - dateTo: ISO date string - conversations before this date
     *   - userId: Filter by specific user ID
     *   - includeContent: Include full conversation content - default: false
     *
     * @return array Export data with download information
     * @throws InvalidArgumentException If request parameters are invalid
     */
    public function exportThreads(array $request = []): array
    {
        // Validate and normalize request
        $normalizedRequest = $this->validateExportRequest($request);

        try {
            // Use AI SDK thread service to export threads
            // License validation happens at the cloud API level (threadHandler.ts)
            $exportData = $this->aiClientManager->getClient()->threads()->export($normalizedRequest);

            return $exportData;
        } catch (\Exception $e) {
            $this->logger->error('Thread export failed', [
                'error' => $e->getMessage(),
                'request_params' => $normalizedRequest
            ]);
            throw $e;
        }
    }

    /**
     * Get thread statistics for admin dashboard
     *
     * @return array Basic thread statistics
     */
    public function getThreadStats(): array
    {
        try {
            $analytics = $this->getThreadAnalytics();

            return [
                'total_conversations' => $analytics['totalConversations'] ?? 0,
                'active_users' => $analytics['activeUsers'] ?? 0,
                'average_messages_per_conversation' => $analytics['averageMessagesPerConversation'] ?? 0,
                'conversations_today' => $analytics['conversationsToday'] ?? 0,
                'conversations_this_week' => $analytics['conversationsThisWeek'] ?? 0,
                'conversations_this_month' => $analytics['conversationsThisMonth'] ?? 0,
            ];
        } catch (\Exception $e) {
            $this->logger->error('Failed to fetch thread stats', [
                'error' => $e->getMessage()
            ]);

            // Return empty stats on error
            return [
                'total_conversations' => 0,
                'active_users' => 0,
                'average_messages_per_conversation' => 0,
                'conversations_today' => 0,
                'conversations_this_week' => 0,
                'conversations_this_month' => 0,
            ];
        }
    }

    /**
     * Validate search request parameters
     *
     * @param array $request Raw request parameters
     * @return array Normalized and validated parameters
     * @throws InvalidArgumentException If parameters are invalid
     */
    private function validateSearchRequest(array $request): array
    {
        $validated = [];

        // Validate and normalize limit
        $limit = $request['limit'] ?? self::DEFAULT_SEARCH_LIMIT;
        if (!is_numeric($limit) || $limit < 1 || $limit > self::MAX_SEARCH_LIMIT) {
            throw new InvalidArgumentException("Limit must be between 1 and " . self::MAX_SEARCH_LIMIT);
        }
        $validated['limit'] = (int) $limit;

        // Validate and normalize offset
        $offset = $request['offset'] ?? 0;
        if (!is_numeric($offset) || $offset < 0) {
            throw new InvalidArgumentException("Offset must be a non-negative number");
        }
        $validated['offset'] = (int) $offset;

        // Optional text query
        if (!empty($request['query'])) {
            $validated['query'] = trim((string) $request['query']);
        }

        // Optional user ID filter
        if (!empty($request['userId'])) {
            $validated['userId'] = trim((string) $request['userId']);
        }

        // Date range filters
        if (!empty($request['dateFrom'])) {
            $dateFrom = trim((string) $request['dateFrom']);
            if (!$this->isValidIsoDate($dateFrom)) {
                throw new InvalidArgumentException("dateFrom must be a valid ISO date string");
            }
            $validated['dateFrom'] = $dateFrom;
        }

        if (!empty($request['dateTo'])) {
            $dateTo = trim((string) $request['dateTo']);
            if (!$this->isValidIsoDate($dateTo)) {
                throw new InvalidArgumentException("dateTo must be a valid ISO date string");
            }
            $validated['dateTo'] = $dateTo;
        }

        // Numeric filters
        $numericFilters = ['minMessages', 'maxMessages', 'minTokens', 'maxTokens'];
        foreach ($numericFilters as $filter) {
            if (isset($request[$filter])) {
                $value = $request[$filter];
                if (!is_numeric($value) || $value < 0) {
                    throw new InvalidArgumentException("{$filter} must be a non-negative number");
                }
                $validated[$filter] = (int) $value;
            }
        }

        // Sort options
        $sortBy = $request['sortBy'] ?? 'timestamp';
        if (!in_array($sortBy, ['timestamp', 'message_count', 'token_count'], true)) {
            throw new InvalidArgumentException("sortBy must be one of: timestamp, message_count, token_count");
        }
        $validated['sortBy'] = $sortBy;

        $sortOrder = $request['sortOrder'] ?? 'desc';
        if (!in_array($sortOrder, ['asc', 'desc'], true)) {
            throw new InvalidArgumentException("sortOrder must be either 'asc' or 'desc'");
        }
        $validated['sortOrder'] = $sortOrder;

        return $validated;
    }

    /**
     * Validate export request parameters
     *
     * @param array $request Raw request parameters
     * @return array Normalized and validated parameters
     * @throws InvalidArgumentException If parameters are invalid
     */
    private function validateExportRequest(array $request): array
    {
        $validated = [];

        // Export format
        $format = $request['format'] ?? 'json';
        if (!in_array($format, ['json', 'csv'], true)) {
            throw new InvalidArgumentException("Format must be either 'json' or 'csv'");
        }
        $validated['format'] = $format;

        // Date range filters (optional)
        if (!empty($request['dateFrom'])) {
            $dateFrom = trim((string) $request['dateFrom']);
            if (!$this->isValidIsoDate($dateFrom)) {
                throw new InvalidArgumentException("dateFrom must be a valid ISO date string");
            }
            $validated['dateFrom'] = $dateFrom;
        }

        if (!empty($request['dateTo'])) {
            $dateTo = trim((string) $request['dateTo']);
            if (!$this->isValidIsoDate($dateTo)) {
                throw new InvalidArgumentException("dateTo must be a valid ISO date string");
            }
            $validated['dateTo'] = $dateTo;
        }

        // Optional user ID filter
        if (!empty($request['userId'])) {
            $validated['userId'] = trim((string) $request['userId']);
        }

        // Include content flag
        $includeContent = $request['includeContent'] ?? false;
        $validated['includeContent'] = (bool) $includeContent;

        return $validated;
    }

    /**
     * Validate ISO date string
     *
     * @param string $date Date string to validate
     * @return bool True if valid ISO date
     */
    private function isValidIsoDate(string $date): bool
    {
        $dateTime = \DateTime::createFromFormat('Y-m-d\TH:i:s\Z', $date);
        if ($dateTime !== false) {
            return true;
        }

        $dateTime = \DateTime::createFromFormat('Y-m-d\TH:i:s.u\Z', $date);
        if ($dateTime !== false) {
            return true;
        }

        $dateTime = \DateTime::createFromFormat('Y-m-d', $date);
        return $dateTime !== false;
    }

    /**
     * Invalidate cached data for a specific thread
     *
     * @param string $threadId Thread ID
     * @return void
     */
    private function invalidateThreadCache(string $threadId): void
    {
        // Clear specific thread detail cache
        $detailCacheKey = "thread_detail_{$threadId}";
        $this->cache->delete($detailCacheKey, 'threads');

        // Clear search caches (since they might contain this thread)
        $this->cache->clearCacheGroup('threads');
    }

    /**
     * Clear all thread-related caches
     *
     * @return void
     */
    public function clearAllThreadCaches(): void
    {
        $this->cache->clearCacheGroup('threads');
        $this->logger->debug('All thread caches cleared');
    }
}
