<?php

namespace Intucart\Services;

use Intucart\Services\Cache\CacheService;
use Intucart\Services\AIClientManager;
use Intucart\Services\Logger;
use Intucart\Services\Constants;

/**
 * Usage tracking and limit enforcement service
 */
class UsageService
{
    private Logger $logger;
    private CacheService $cacheService;
    private AIClientManager $aiClientManager;

    public function __construct(
        Logger $logger,
        CacheService $cacheService,
        AIClientManager $aiClientManager
    ) {
        $this->logger = $logger;
        $this->cacheService = $cacheService;
        $this->aiClientManager = $aiClientManager;

        // Listen for license changes to invalidate usage cache
        add_action('intucart_license_updated', [$this, 'invalidateUsageCache']);
        add_action('intucart_license_activated', [$this, 'invalidateUsageCache']);

        // Listen for successful sync operations to invalidate usage cache
        add_action(Constants::POST_SYNC_COMPLETED_ACTION, [$this, 'invalidateUsageCache']);
        add_action(Constants::TAXONOMY_SYNC_COMPLETED_ACTION, [$this, 'invalidateUsageCache']);
    }

    /**
     * Get current usage data from cloud or cache
     *
     * @return array {success: bool, usage?: array, reason?: string}
     */
    public function getCurrentUsage(): array
    {
        // Don't make API calls if no license key is set
        $license_key = get_option(Constants::LICENSE_KEY_OPTION, '');
        if (empty($license_key)) {
            return [
                'success' => false,
                'reason' => 'No license key configured'
            ];
        }

        // Try cache first (5 minute TTL)
        $cacheKey = Constants::USAGE_CACHE_KEY;
        $cached = $this->cacheService->get($cacheKey, Constants::USAGE_CACHE_GROUP);

        if ($cached !== false && is_array($cached)) {
            return ['success' => true, 'usage' => $cached];
        }

        // Get fresh usage data from cloud using AI SDK
        try {
            $aiClient = $this->aiClientManager->getClient();
            // Use tenant().getUsage() for usage metrics and feature limits
            $usageData = $aiClient->tenant()->getUsage();

            if (empty($usageData) || !isset($usageData['features'])) {
                return [
                    'success' => false,
                    'reason' => 'Could not retrieve usage data from cloud'
                ];
            }

            // Cache for 5 minutes
            $this->cacheService->set($cacheKey, $usageData, Constants::USAGE_CACHE_GROUP, 300);

            return ['success' => true, 'usage' => $usageData];
        } catch (\Exception $e) {
            $this->logger->error('Failed to retrieve usage data', [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'reason' => 'Exception while retrieving usage data: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get feature status for admin display purposes only
     * Note: This is for status page display only, not for real-time feature enforcement
     *
     * @return array
     */
    public function getFeatureStatus(): array
    {
        $usageData = $this->getCurrentUsage();

        if (!$usageData['success']) {
            return [
                'chat' => ['success' => false, 'reason' => 'Unable to retrieve usage data'],
                'search' => ['success' => false, 'reason' => 'Unable to retrieve usage data'],
                'recommendations' => ['success' => false, 'reason' => 'Unable to retrieve usage data']
            ];
        }

        $usage = $usageData['usage'];
        $status = [];

        // Use standardized format (new health handler always returns this)
        if (isset($usage['features']) && is_array($usage['features'])) {
            foreach ($usage['features'] as $feature) {
                $type = $feature['type'] ?? '';
                $featureUsage = $feature['usage'] ?? [];

                // Skip if no valid feature type
                if (empty($type)) {
                    continue;
                }

                $current = $featureUsage['current'] ?? 0;
                $limit = $featureUsage['limit'] ?? PHP_INT_MAX;
                $available = $featureUsage['available'] ?? true;

                if (!$available || $current >= $limit) {
                    $status[$type] = [
                        'success' => false,
                        'reason' => ucfirst($type) . ' limit exceeded',
                        'usage' => [
                            'used' => $current,
                            'limit' => $limit,
                            'resetTime' => $usage['resetTime'] ?? null
                        ]
                    ];
                } else {
                    $status[$type] = [
                        'success' => true,
                        'usage' => [
                            'used' => $current,
                            'limit' => $limit,
                            'remaining' => $limit - $current
                        ]
                    ];
                }
            }
        }

        // Ensure all expected features are present
        // Note: 'users' is excluded as user sync is currently disabled
        // See Status.php:1065 - TODO: Users sync is temporarily disabled
        $expectedFeatures = ['chat', 'search', 'recommendations', 'products', 'posts', 'taxonomies', 'prompts', 'webhooks', 'webhook_deliveries'];
        foreach ($expectedFeatures as $feature) {
            if (!isset($status[$feature])) {
                $status[$feature] = ['success' => false, 'reason' => 'No data available'];
            }
        }

        return $status;
    }

    /**
     * Invalidate usage cache to force fresh data on next request
     * Call this after manual syncs, license changes, etc.
     *
     * @return void
     */
    public function invalidateUsageCache(): void
    {
        $cacheKey = Constants::USAGE_CACHE_KEY;
        $this->cacheService->delete($cacheKey, Constants::USAGE_CACHE_GROUP);

        $this->logger->debug('Usage cache invalidated - fresh data will be fetched on next request');
    }

    /**
     * Get formatted usage summary for display
     *
     * @return array
     */
    public function getUsageSummary(): array
    {
        $usageData = $this->getCurrentUsage();

        if (!$usageData['success']) {
            return [
                'success' => false,
                'message' => $usageData['reason'] ?? 'Could not retrieve usage data'
            ];
        }

        $usage = $usageData['usage'];

        // Check if we have new standardized format
        if (isset($usage['features']) && is_array($usage['features'])) {
            return $this->getUsageSummaryFromStandardFormat($usage);
        }

        // Fallback to legacy format
        return $this->getUsageSummaryFromLegacyFormat($usage);
    }

    /**
     * Get usage summary from new standardized format
     *
     * @param array $usage Usage data in new format
     * @return array Usage summary
     */
    private function getUsageSummaryFromStandardFormat(array $usage): array
    {
        $summary = [
            'success' => true,
            'tier' => $usage['tier'] ?? 'Unknown',
            'resetTime' => $usage['resetTime'] ?? null,
            'features' => []
        ];

        // Pass through debug mode flag if present
        if (isset($usage['debug_mode'])) {
            $summary['debug_mode'] = $usage['debug_mode'];
        }

        foreach ($usage['features'] as $feature) {
            $type = $feature['type'] ?? '';
            $name = $feature['name'] ?? ucfirst($type);
            $featureUsage = $feature['usage'] ?? [];

            $current = $featureUsage['current'] ?? 0;
            $limit = $featureUsage['limit'] ?? 0;
            $percentage = $featureUsage['percentage'] ?? 0;
            $available = $featureUsage['available'] ?? true;

            $summary['features'][$type] = [
                'name' => $name,
                'used' => $current,
                'limit' => $limit,
                'percentage' => $percentage,
                'available' => $available
            ];
        }

        return $summary;
    }

    /**
     * Get usage summary from legacy format (backward compatibility)
     *
     * @param array $usage Usage data in legacy format
     * @return array Usage summary
     */
    private function getUsageSummaryFromLegacyFormat(array $usage): array
    {
        $quotas = $usage['quotas'] ?? [];
        $tierLimits = $usage['tierLimits'] ?? null;
        $resetTime = $usage['resetTimes']['quotas'] ?? null;

        $summary = [
            'success' => true,
            'tier' => $tierLimits['name'] ?? 'Unknown',
            'resetTime' => $resetTime,
            'features' => []
        ];

        // Chat messages
        if (isset($quotas['chatMessages'])) {
            $chat = $quotas['chatMessages'];
            $summary['features']['chat'] = [
                'name' => 'Chat Messages',
                'used' => $chat['used'] ?? 0,
                'limit' => $chat['limit'] ?? 0,
                'percentage' => $chat['limit'] > 0 ? round(($chat['used'] / $chat['limit']) * 100, 1) : 0,
                'available' => ($chat['used'] ?? 0) < ($chat['limit'] ?? PHP_INT_MAX)
            ];
        }

        // Search queries
        if (isset($quotas['searchQueries'])) {
            $search = $quotas['searchQueries'];
            $summary['features']['search'] = [
                'name' => 'Search Queries',
                'used' => $search['used'] ?? 0,
                'limit' => $search['limit'] ?? 0,
                'percentage' => $search['limit'] > 0 ? round(($search['used'] / $search['limit']) * 100, 1) : 0,
                'available' => ($search['used'] ?? 0) < ($search['limit'] ?? PHP_INT_MAX)
            ];
        }

        // Recommendations
        if (isset($quotas['recommendations'])) {
            $rec = $quotas['recommendations'];
            $summary['features']['recommendations'] = [
                'name' => 'Recommendations',
                'used' => $rec['used'] ?? 0,
                'limit' => $rec['limit'] ?? 0,
                'percentage' => $rec['limit'] > 0 ? round(($rec['used'] / $rec['limit']) * 100, 1) : 0,
                'available' => ($rec['used'] ?? 0) < ($rec['limit'] ?? PHP_INT_MAX)
            ];
        }

        return $summary;
    }



    /**
     * Check license service health via AI SDK
     *
     * @return array Health status information
     */
    public function getLicenseServiceHealth(): array
    {
        try {
            $license_key = get_option(Constants::LICENSE_KEY_OPTION, '');
            if (empty($license_key)) {
                return [
                    'success' => false,
                    'message' => 'No license key configured'
                ];
            }

            // Use the AI SDK tenant service for subscription check
            $aiClient = $this->aiClientManager->getClient();
            $tenantStatus = $aiClient->tenant()->getStatus();

            // Check if subscription is valid
            $isValid = $tenantStatus['subscription']['valid'] ?? false;

            if ($isValid) {
                return [
                    'success' => true,
                    'health' => $tenantStatus
                ];
            } else {
                $validationError = $tenantStatus['subscription']['validation']['error'] ?? null;
                return [
                    'success' => false,
                    'message' => 'Subscription validation failed: ' . ($validationError ?? 'Unknown error')
                ];
            }
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Exception checking license service health: ' . $e->getMessage()
            ];
        }
    }
}
