<?php

declare(strict_types=1);

namespace Intucart\Shortcodes;

use Intucart\Services\Constants;

/**
 * Handles the [intufind_search] shortcode.
 * Renders a button that opens the search widget overlay.
 */
class SearchWidgetShortcode
{
    private const SHORTCODE_TAG = 'intufind_search';

    /**
     * Registers the shortcode.
     */
    public function initialize(): void
    {
        add_shortcode(self::SHORTCODE_TAG, [$this, 'renderShortcode']);
    }

    /**
     * Renders the shortcode output.
     *
     * @param array|string $atts Shortcode attributes.
     * @param string|null $content Shortcode content.
     *
     * @return string The HTML output for the shortcode.
     */
    public function renderShortcode($atts = [], ?string $content = null): string
    {
        // Ensure $atts is an array
        if (!is_array($atts)) {
            $atts = [];
        }

        // Check if search widget is enabled
        $widget_enabled = get_option(Constants::OPTION_SEARCH_WIDGET_ENABLED, '0') === '1';
        if (!$widget_enabled) {
            return '<!-- Search widget is not enabled -->';
        }

        // Parse shortcode attributes
        $atts = shortcode_atts([
            'text' => __('Search', 'intufind'),
            'icon_only' => 'false',
            'class' => '',
            'style' => '',
            'query' => '', // Optional: pre-fill search query
        ], $atts);

        // Search icon SVG
        $icon = '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="11" cy="11" r="8"/><path d="m21 21-4.35-4.35"/></svg>';

        // Build button content
        $show_icon_only = strtolower($atts['icon_only']) === 'true';
        $text_content = $show_icon_only
            ? $icon
            : '<span class="intufind-search-icon">' . $icon . '</span><span class="intufind-search-text">' . esc_html($atts['text']) . '</span>';

        // Build class list
        $classes = ['intufind-search-trigger'];
        if (!empty($atts['class'])) {
            $classes[] = esc_attr($atts['class']);
        }
        if ($show_icon_only) {
            $classes[] = 'intufind-search-trigger--icon-only';
        }

        // Build inline style if provided
        $style_attr = !empty($atts['style']) ? ' style="' . esc_attr($atts['style']) . '"' : '';

        // Build onclick handler with optional query
        $query = !empty($atts['query']) ? esc_js($atts['query']) : '';
        $onclick = $query
            ? "window.IntufindSearch?.open('{$query}')"
            : 'window.IntufindSearch?.open()';

        // Get keyboard shortcut for tooltip
        $shortcut = get_option(
            Constants::OPTION_SEARCH_WIDGET_KEYBOARD_SHORTCUT,
            Constants::DEFAULT_SEARCH_WIDGET_KEYBOARD_SHORTCUT
        );
        $shortcut_label = $this->formatShortcutLabel($shortcut);

        return sprintf(
            '<button type="button" class="%s" onclick="%s" aria-label="%s" title="%s"%s>%s</button>',
            esc_attr(implode(' ', $classes)),
            $onclick,
            esc_attr__('Open search', 'intufind'),
            esc_attr($shortcut_label),
            $style_attr,
            $text_content
        );
    }

    /**
     * Format keyboard shortcut for display.
     *
     * @param string $shortcut The shortcut string (e.g., 'ctrl+k')
     * @return string Formatted label (e.g., 'Ctrl+K' or '⌘K')
     */
    private function formatShortcutLabel(string $shortcut): string
    {
        // Detect if user is likely on Mac (this is a hint for the tooltip)
        // Actual detection happens in JavaScript
        $is_mac = isset($_SERVER['HTTP_USER_AGENT']) && 
                  stripos($_SERVER['HTTP_USER_AGENT'], 'mac') !== false;

        $shortcut = strtolower($shortcut);

        switch ($shortcut) {
            case 'ctrl+k':
                return $is_mac ? '⌘K' : 'Ctrl+K';
            case 'ctrl+/':
                return $is_mac ? '⌘/' : 'Ctrl+/';
            case 'ctrl+shift+f':
                return $is_mac ? '⌘⇧F' : 'Ctrl+Shift+F';
            default:
                return strtoupper($shortcut);
        }
    }
}

