<?php

namespace Intucart\Utils;

/**
 * ColorUtils class
 *
 * Provides reusable color manipulation and conversion utilities
 * for the IntUCart plugin codebase.
 */
class ColorUtils
{
    /**
     * Convert a hex color to rgba with specified alpha
     *
     * @param string $hex Hex color (with or without #)
     * @param float $alpha Alpha value (0-1)
     * @return string RGBA color string
     */
    public static function convertToRgba(string $hex, float $alpha): string
    {
        // Validate input
        if (!$hex || !is_string($hex)) {
            return 'rgba(255, 255, 255, ' . $alpha . ')';
        }

        // Remove # if present
        $hex = ltrim($hex, '#');

        // Validate hex format
        if (!preg_match('/^([0-9A-F]{3}|[0-9A-F]{6})$/i', $hex)) {
            return 'rgba(255, 255, 255, ' . $alpha . ')';
        }

        // Handle 3-digit hex by expanding to 6-digit
        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }

        // Convert hex to RGB
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        // Clamp alpha to valid range
        $alpha = max(0, min(1, $alpha));

        return "rgba($r, $g, $b, $alpha)";
    }

    /**
     * Adjust the brightness of a hex color
     *
     * @param string $hex Hex color (with or without #)
     * @param int $percent Percentage to adjust (-100 to 100)
     * @return string Adjusted hex color
     */
    public static function adjustBrightness(string $hex, int $percent): string
    {
        // Validate input
        if (!$hex || !is_string($hex)) {
            return '#000000';
        }

        // Remove # if present
        $hex = ltrim($hex, '#');

        // Validate hex format
        if (!preg_match('/^([0-9A-F]{3}|[0-9A-F]{6})$/i', $hex)) {
            return '#' . $hex; // Return original if invalid
        }

        // Handle 3-digit hex by expanding to 6 digits
        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }

        // Convert to RGB
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        // Apply multiplicative brightness adjustment
        $factor = 1 + ($percent / 100);
        $r = round($r * $factor);
        $g = round($g * $factor);
        $b = round($b * $factor);

        // Ensure values stay within 0-255 range
        $r = max(0, min(255, $r));
        $g = max(0, min(255, $g));
        $b = max(0, min(255, $b));

        // Convert back to hex
        return sprintf('#%02x%02x%02x', $r, $g, $b);
    }

    /**
     * Convert hex color to HSL
     *
     * @param string $hex Hex color (with or without #)
     * @return array Array with keys 'h', 's', 'l' (hue 0-360, saturation/lightness 0-100)
     */
    public static function hexToHsl(string $hex): array
    {
        // Validate and normalize hex
        if (!$hex || !is_string($hex)) {
            return ['h' => 0, 's' => 0, 'l' => 0];
        }

        $hex = ltrim($hex, '#');
        if (!preg_match('/^([0-9A-F]{3}|[0-9A-F]{6})$/i', $hex)) {
            return ['h' => 0, 's' => 0, 'l' => 0];
        }

        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }

        // Convert to RGB (0-1 range)
        $r = hexdec(substr($hex, 0, 2)) / 255;
        $g = hexdec(substr($hex, 2, 2)) / 255;
        $b = hexdec(substr($hex, 4, 2)) / 255;

        $max = max($r, $g, $b);
        $min = min($r, $g, $b);
        $diff = $max - $min;

        // Lightness
        $l = ($max + $min) / 2;

        if ($diff == 0) {
            $h = $s = 0; // achromatic
        } else {
            // Saturation
            $s = $l > 0.5 ? $diff / (2 - $max - $min) : $diff / ($max + $min);

            // Hue
            switch ($max) {
                case $r:
                    $h = (($g - $b) / $diff + ($g < $b ? 6 : 0)) / 6;
                    break;
                case $g:
                    $h = (($b - $r) / $diff + 2) / 6;
                    break;
                case $b:
                    $h = (($r - $g) / $diff + 4) / 6;
                    break;
                default:
                    $h = 0;
            }
        }

        return [
            'h' => round($h * 360),
            's' => round($s * 100),
            'l' => round($l * 100)
        ];
    }

    /**
     * Convert HSL to hex color
     *
     * @param int $h Hue (0-360)
     * @param int $s Saturation (0-100)
     * @param int $l Lightness (0-100)
     * @return string Hex color
     */
    public static function hslToHex(int $h, int $s, int $l): string
    {
        // Normalize values
        $h = max(0, min(360, $h)) / 360;
        $s = max(0, min(100, $s)) / 100;
        $l = max(0, min(100, $l)) / 100;

        if ($s == 0) {
            // achromatic
            $r = $g = $b = $l;
        } else {
            $hue2rgb = function ($p, $q, $t) {
                if ($t < 0) $t += 1;
                if ($t > 1) $t -= 1;
                if ($t < 1/6) return $p + ($q - $p) * 6 * $t;
                if ($t < 1/2) return $q;
                if ($t < 2/3) return $p + ($q - $p) * (2/3 - $t) * 6;
                return $p;
            };

            $q = $l < 0.5 ? $l * (1 + $s) : $l + $s - $l * $s;
            $p = 2 * $l - $q;
            $r = $hue2rgb($p, $q, $h + 1/3);
            $g = $hue2rgb($p, $q, $h);
            $b = $hue2rgb($p, $q, $h - 1/3);
        }

        return sprintf('#%02x%02x%02x', 
            round($r * 255), 
            round($g * 255), 
            round($b * 255)
        );
    }

    /**
     * Check if a color is considered "light" or "dark"
     *
     * @param string $hex Hex color (with or without #)
     * @return bool True if the color is light, false if dark
     */
    public static function isLight(string $hex): bool
    {
        // Validate and normalize hex
        if (!$hex || !is_string($hex)) {
            return false;
        }

        $hex = ltrim($hex, '#');
        if (!preg_match('/^([0-9A-F]{3}|[0-9A-F]{6})$/i', $hex)) {
            return false;
        }

        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }

        // Convert to RGB
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        // Calculate luminance using the standard formula
        $luminance = (0.299 * $r + 0.587 * $g + 0.114 * $b) / 255;

        // Return true if luminance is greater than 0.5
        return $luminance > 0.5;
    }

    /**
     * Get a contrasting text color (black or white) for a given background color
     *
     * @param string $hex Background hex color (with or without #)
     * @return string Either '#000000' or '#ffffff'
     */
    public static function getContrastingTextColor(string $hex): string
    {
        return self::isLight($hex) ? '#000000' : '#ffffff';
    }

    /**
     * Parse RGBA color string to extract hex color and alpha percentage
     *
     * @param string $rgba RGBA color string like 'rgba(255, 0, 0, 0.8)'
     * @param string $fallbackHex Fallback hex color if parsing fails
     * @param int $fallbackAlpha Fallback alpha percentage if parsing fails
     * @return array Array with 'hex' and 'alpha_percent' keys
     */
    public static function parseRgbaToHexAndAlpha(string $rgba, string $fallbackHex = '#000000', int $fallbackAlpha = 100): array
    {
        // Set default return values
        $result = [
            'hex' => $fallbackHex,
            'alpha_percent' => $fallbackAlpha
        ];

        // Parse RGBA string
        if (preg_match('/^rgba\((\d+),\s*(\d+),\s*(\d+),\s*([\d.]+)\)/', $rgba, $matches)) {
            $r = max(0, min(255, intval($matches[1])));
            $g = max(0, min(255, intval($matches[2])));
            $b = max(0, min(255, intval($matches[3])));
            $alpha = max(0, min(1, floatval($matches[4])));

            $result['hex'] = sprintf('#%02x%02x%02x', $r, $g, $b);
            $result['alpha_percent'] = round($alpha * 100);
        }

        return $result;
    }
}
