<?php

namespace Intucart\Utils;

/**
 * ElementorUtils class
 *
 * Provides reusable Elementor page builder utilities for content extraction
 * and JSON data cleaning. Uses clean JSON approach to preserve widget structure,
 * settings, and semantic meaning for enhanced search capabilities.
 */
class ElementorUtils
{
    /**
     * Track posts currently being rendered to prevent infinite recursion
     * @var array<string, bool>
     */
    private static array $renderingPosts = [];

    /**
     * Maximum recursion depth for JSON cleaning
     */
    private const MAX_JSON_DEPTH = 20;

    /**
     * Check if a post uses Elementor
     *
     * @param int $postId Post ID
     * @return bool True if post uses Elementor
     */
    public static function isElementorPost(int $postId): bool
    {
        return class_exists('\Elementor\Plugin') && get_post_meta($postId, '_elementor_edit_mode', true);
    }

    /**
     * Get Elementor data for a post
     *
     * @param int $postId Post ID
     * @return array|null Elementor data array or null if not found
     */
    public static function getElementorData(int $postId): ?array
    {
        if (!self::isElementorPost($postId)) {
            return null;
        }

        $elementorData = get_post_meta($postId, '_elementor_data', true);
        if (empty($elementorData)) {
            return null;
        }

        // Handle both JSON and base64 encoded data
        $data = self::decodeElementorData($elementorData);
        return is_array($data) ? $data : null;
    }

    /**
     * Decode Elementor data from either JSON or base64 encoded format
     *
     * @param mixed $elementorData Raw Elementor data from post meta
     * @return array|null Decoded data array or null if invalid
     */
    public static function decodeElementorData($elementorData): ?array
    {
        // If it's already an array, validate and return it
        if (is_array($elementorData)) {
            return self::isValidElementorStructure($elementorData) ? $elementorData : null;
        }

        // If it's not a string, we can't decode it
        if (!is_string($elementorData)) {
            return null;
        }

        // First try to decode as JSON directly
        $jsonData = json_decode($elementorData, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($jsonData)) {
            // Validate structure before returning
            if (self::isValidElementorStructure($jsonData)) {
                return $jsonData;
            }
        }

        // If JSON decode failed, try base64 decode first then JSON
        $base64Decoded = base64_decode($elementorData, true);
        if ($base64Decoded !== false) {
            $jsonData = json_decode($base64Decoded, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($jsonData)) {
                // Validate structure to ensure it's actually Elementor data
                if (self::isValidElementorStructure($jsonData)) {
                    return $jsonData;
                }
            }
        }

        // If both methods failed or structure validation failed, return null
        return null;
    }

    /**
     * Validate that decoded data has a valid Elementor structure
     * Elementor data is an array of elements with 'elType' or 'widgetType' properties
     *
     * @param array $data Decoded data to validate
     * @return bool True if data appears to be valid Elementor structure
     */
    private static function isValidElementorStructure(array $data): bool
    {
        // Empty array is valid (could be a new/empty page)
        if (empty($data)) {
            return true;
        }

        // Check if at least one top-level element has Elementor structure markers
        foreach ($data as $element) {
            if (is_array($element)) {
                // Elementor elements have 'elType' (section, column, widget) or 'widgetType'
                if (isset($element['elType']) || isset($element['widgetType'])) {
                    return true;
                }
                // Also check for 'elements' array which is common in Elementor structures
                if (isset($element['elements']) && is_array($element['elements'])) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Extract content from Elementor data using clean JSON approach
     *
     * @param int $postId Post ID
     * @return string JSON string of cleaned Elementor data
     */
    public static function extractContentAsJson(int $postId): string
    {
        $data = self::getElementorData($postId);
        if (!$data) {
            return '';
        }

        $cleanedData = self::cleanElementorJson($data);
        return json_encode($cleanedData, JSON_UNESCAPED_UNICODE);
    }

    /**
     * Check if Elementor data appears to be base64 encoded
     *
     * @param string $data Data to check
     * @return bool True if data appears to be base64 encoded
     */
    public static function isBase64EncodedElementorData(string $data): bool
    {
        // Quick checks for base64 characteristics
        if (empty($data) || strlen($data) < 4) {
            return false;
        }

        // Base64 should only contain these characters
        if (!preg_match('/^[A-Za-z0-9+\/]*={0,2}$/', $data)) {
            return false;
        }

        // Try to decode and see if it looks like JSON
        $decoded = base64_decode($data, true);
        if ($decoded === false) {
            return false;
        }

        // Check if decoded data looks like JSON (starts with [ or {)
        $trimmed = trim($decoded);
        return !empty($trimmed) && (
            $trimmed[0] === '[' ||
            $trimmed[0] === '{'
        );
    }

    /**
     * Clean Elementor JSON data by removing system properties while preserving content
     *
     * @param array $data Elementor data array
     * @param int $currentDepth Current recursion depth (for internal use)
     * @return array Cleaned data ready for search indexing
     */
    public static function cleanElementorJson(array $data, int $currentDepth = 0): array
    {
        // Prevent excessively deep structures that could cause stack overflow
        if ($currentDepth >= self::MAX_JSON_DEPTH) {
            return [];
        }

        $cleaned = [];

        foreach ($data as $element) {
            if (!is_array($element)) {
                continue;
            }

            $cleanElement = self::cleanElementorElement($element, $currentDepth + 1);
            if (!empty($cleanElement)) {
                $cleaned[] = $cleanElement;
            }
        }

        return $cleaned;
    }

    /**
     * Clean individual Elementor element
     *
     * @param array $element Element data
     * @param int $currentDepth Current recursion depth
     * @return array|null Cleaned element or null if should be excluded
     */
    public static function cleanElementorElement(array $element, int $currentDepth = 0): ?array
    {
        // Prevent excessively deep structures
        if ($currentDepth >= self::MAX_JSON_DEPTH) {
            return null;
        }

        // System properties to remove (keep content, settings, and structure)
        $systemProps = [
            'id',           // Internal Elementor IDs
            'isInner',      // Internal flags
            'isLocked',     // Internal flags
            'editSettings', // Internal editor settings
            '_column_size', // Internal layout
            '_inline_size', // Internal layout
        ];

        $cleaned = [];

        foreach ($element as $key => $value) {
            // Skip system properties
            if (in_array($key, $systemProps)) {
                continue;
            }

            // Handle nested elements recursively
            if ($key === 'elements' && is_array($value)) {
                $cleanedElements = [];
                foreach ($value as $childElement) {
                    if (is_array($childElement)) {
                        $cleanedChild = self::cleanElementorElement($childElement, $currentDepth + 1);
                        if (!empty($cleanedChild)) {
                            $cleanedElements[] = $cleanedChild;
                        }
                    }
                }
                if (!empty($cleanedElements)) {
                    $cleaned[$key] = $cleanedElements;
                }
            }
            // Handle settings - clean but preserve structure
            elseif ($key === 'settings' && is_array($value)) {
                $cleanedSettings = self::cleanElementorSettings($value);
                if (!empty($cleanedSettings)) {
                    $cleaned[$key] = $cleanedSettings;
                }
            }
            // Keep other important properties
            else {
                $cleaned[$key] = $value;
            }
        }

        return !empty($cleaned) ? $cleaned : null;
    }

    /**
     * Clean Elementor widget settings
     *
     * @param array $settings Settings array
     * @return array Cleaned settings
     */
    public static function cleanElementorSettings(array $settings): array
    {
        // Remove purely visual/system settings, keep content
        $visualOnlyProps = [
            '_element_id',
            '_css_classes',
            'motion_fx_motion_fx_scrolling',
            'motion_fx_motion_fx_mouse',
            '_background_background',
            '_border_border',
            '_padding',
            '_margin',
            'animation',
            'hide_desktop',
            'hide_tablet',
            'hide_mobile',
        ];

        $cleaned = [];

        foreach ($settings as $key => $value) {
            // Skip purely visual properties
            if (in_array($key, $visualOnlyProps)) {
                continue;
            }

            // Skip properties that start with underscore (mostly internal)
            if (strpos($key, '_') === 0) {
                continue;
            }

            // Keep content and meaningful settings
            $cleaned[$key] = $value;
        }

        return $cleaned;
    }

    /**
     * Render Elementor builder HTML for a post if possible.
     * Returns empty string if Elementor is unavailable or nothing renders.
     * Includes recursion guard to prevent infinite loops.
     */
    public static function renderBuilderHtml(int $postId): string
    {
        if (!class_exists('\\Elementor\\Plugin')) {
            return '';
        }

        // Recursion guard
        $key = "builder_{$postId}";
        if (isset(self::$renderingPosts[$key])) {
            return ''; // Prevent re-entry
        }
        self::$renderingPosts[$key] = true;

        try {
            $html = \Elementor\Plugin::instance()->frontend->get_builder_content_for_display($postId, true);
            $plain = trim(wp_strip_all_tags((string) $html));
            return ($plain !== '') ? (string) $html : '';
        } catch (\Throwable $e) {
            return '';
        } finally {
            unset(self::$renderingPosts[$key]);
        }
    }

    /**
     * Render an Elementor Theme Builder location (e.g., 'single') for a given post context.
     * Returns empty string if the location doesn't render or Elementor is unavailable.
     * Includes recursion guard and proper global state restoration via finally.
     */
    public static function renderThemeTemplate(string $location, \WP_Post $post): string
    {
        if (!function_exists('elementor_theme_do_location')) {
            return '';
        }

        // Recursion guard
        $key = "theme_{$location}_{$post->ID}";
        if (isset(self::$renderingPosts[$key])) {
            return ''; // Prevent re-entry
        }
        self::$renderingPosts[$key] = true;

        $previousGlobalPost = $GLOBALS['post'] ?? null;
        $output = '';
        $didRender = false;

        try {
            $GLOBALS['post'] = $post;
            setup_postdata($post);

            ob_start();
            try {
                $didRender = (bool) call_user_func('elementor_theme_do_location', $location);
            } catch (\Throwable $e) {
                // swallow rendering errors
            }
            $output = ob_get_clean();
        } finally {
            // Always restore globals and clean up recursion guard
            unset(self::$renderingPosts[$key]);
            $GLOBALS['post'] = $previousGlobalPost;
            if ($previousGlobalPost instanceof \WP_Post) {
                setup_postdata($previousGlobalPost);
            } else {
                wp_reset_postdata();
            }
        }

        if (!$didRender) {
            return '';
        }

        $plain = trim(wp_strip_all_tags((string) $output));
        return ($plain !== '') ? (string) $output : '';
    }

    /**
     * Render Elementor output for a post using best-effort fallback:
     * 1) If edited with Elementor, return rendered builder HTML
     * 2) Otherwise, try Theme Builder 'single' template for the post
     * Returns empty string if nothing renders.
     */
    public static function renderPostHtml(\WP_Post $post): string
    {
        if (!class_exists('\\Elementor\\Plugin')) {
            return '';
        }

        // Prefer direct builder rendering when the post is edited with Elementor
        if (self::isElementorPost((int) $post->ID)) {
            $html = self::renderBuilderHtml((int) $post->ID);
            if ($html !== '') {
                return $html;
            }
        }

        // Fallback to Theme Builder template (single)
        return self::renderThemeTemplate('single', $post);
    }
}
