# Intufind AI PHP SDK

Official PHP SDK for [Intufind](https://intufind.com) AI Cloud Services — AI-powered semantic search, chat, and recommendations for e-commerce.

[![PHP Version](https://img.shields.io/badge/php-%5E7.4%20%7C%20%5E8.0-blue)](https://php.net)
[![License](https://img.shields.io/badge/license-MIT-green)](LICENSE)

## Installation

```bash
composer require intufind/ai-sdk
```

## Quick Start

```php
<?php

use Intufind\AI\Client;

$client = new Client([
    'api_key' => 'your-api-key',
    'workspace_id' => 'your-workspace-id',
]);

// Search products semantically
$results = $client->products()->search([
    'text' => 'wireless noise cancelling headphones',
    'limit' => 10,
]);

foreach ($results->getResults() as $product) {
    echo $product->name . ' - $' . $product->price . "\n";
}
```

## Features

- **Semantic Search** — AI-powered product and content search
- **AI Chat** — Conversational shopping assistant with streaming support
- **Recommendations** — Trending, similar, and personalized product suggestions
- **Content Indexing** — Index products, posts, and custom content
- **Webhooks** — Real-time event notifications

## Configuration

```php
use Intufind\AI\Client;
use Intufind\AI\Config\Configuration;

$config = new Configuration([
    'api_key' => 'your-api-key',
    'workspace_id' => 'your-workspace-id',
    'api_endpoint' => 'https://api.intufind.com',      // Optional: custom endpoint
    'streaming_endpoint' => 'https://stream.intufind.com', // Optional: for chat streaming
    'timeout' => 30,  // Request timeout in seconds
    'debug' => false, // Enable debug logging
]);

$client = new Client($config);
```

## Usage Examples

### Products

```php
// Index a product
$client->products()->upsert([
    'id' => 'sku-123',
    'name' => 'Wireless Headphones',
    'content' => 'Premium wireless headphones with active noise cancellation...',
    'price' => 299.99,
    'categories' => ['electronics', 'audio'],
    'attributes' => [
        'color' => ['Black', 'Silver'],
        'brand' => ['AudioPro'],
    ],
]);

// Search products
$results = $client->products()->search([
    'text' => 'comfortable headphones for working from home',
    'limit' => 10,
    'filters' => [
        'categories' => ['electronics'],
    ],
]);

// Delete a product
$client->products()->delete('sku-123');
```

### Posts / Content

```php
// Index a blog post
$client->posts()->upsert([
    'id' => 'post-456',
    'title' => 'Best Headphones for Remote Work',
    'content' => 'Finding the perfect headphones for your home office...',
    'status' => 'publish',
    'categories' => ['guides', 'audio'],
]);

// Search posts
$results = $client->posts()->search([
    'text' => 'headphone buying guide',
    'limit' => 5,
]);
```

### AI Chat

```php
// Send a chat message
$response = $client->chat()->send([
    'message' => 'I need headphones for working from home, budget around $200',
    'threadId' => 'user-session-123',
]);

echo $response->getIntro();

foreach ($response->getProducts() as $product) {
    echo "- {$product->name}: \${$product->price}\n";
}

// Streaming chat (for real-time responses)
$client->chat()->stream([
    'message' => 'Tell me more about the first option',
    'threadId' => 'user-session-123',
], function ($chunk) {
    echo $chunk->getContent();
    flush();
});
```

### Recommendations

```php
// Get trending products
$trending = $client->recommendations()->getTrending(['limit' => 5]);

// Get similar products
$similar = $client->recommendations()->getSimilar('sku-123', ['limit' => 5]);

// Get personalized recommendations
$personalized = $client->recommendations()->getPersonalized([
    'userId' => 'user-456',
    'limit' => 10,
]);
```

### Webhooks

```php
// Register a webhook
$client->webhooks()->create([
    'url' => 'https://yoursite.com/webhook',
    'events' => ['product.indexed', 'chat.completed'],
]);

// List webhooks
$webhooks = $client->webhooks()->list();

// Test a webhook
$result = $client->webhooks()->test('webhook-id');
```

## Error Handling

```php
use Intufind\AI\Exceptions\ApiException;
use Intufind\AI\Exceptions\AuthenticationException;
use Intufind\AI\Exceptions\RateLimitException;
use Intufind\AI\Exceptions\ValidationException;

try {
    $results = $client->products()->search(['text' => 'headphones']);
} catch (AuthenticationException $e) {
    // Invalid API key or license
    error_log('Auth failed: ' . $e->getMessage());
} catch (RateLimitException $e) {
    // Too many requests
    $retryAfter = $e->getRetryAfter();
    error_log("Rate limited. Retry after {$retryAfter} seconds");
} catch (ValidationException $e) {
    // Invalid request parameters
    error_log('Validation error: ' . $e->getMessage());
} catch (ApiException $e) {
    // General API error
    error_log('API error: ' . $e->getMessage());
    error_log('Status: ' . $e->getStatusCode());
}
```

## Available Services

| Service | Access | Description |
|---------|--------|-------------|
| Products | `$client->products()` | Product indexing and search |
| Posts | `$client->posts()` | Content/blog post management |
| Chat | `$client->chat()` | AI conversational interface |
| Recommendations | `$client->recommendations()` | Product recommendations |
| Taxonomy | `$client->taxonomy()` | Category and tag management |
| Prompts | `$client->prompts()` | Custom AI prompt templates |
| Threads | `$client->threads()` | Chat thread management |
| Feedback | `$client->feedback()` | User feedback collection |
| Webhooks | `$client->webhooks()` | Webhook management |
| Tenant | `$client->tenant()` | Account and subscription info |
| Provisioning | `$client->provisioning()` | License activation |

## Requirements

- PHP 7.4 or 8.0+
- `ext-json`
- `ext-curl`
- Guzzle HTTP client (installed automatically)

## Development

```bash
# Install dependencies
composer install

# Run tests
composer test

# Static analysis
composer phpstan

# Code style check
composer cs-check

# Fix code style
composer cs-fix

# Pre-release validation (runs all checks)
composer pre-release
```

## Support

- **Documentation**: [docs.intufind.com](https://docs.intufind.com)
- **Email**: support@intufind.com
- **Issues**: [GitHub Issues](https://github.com/intufind/ai-sdk-php/issues)

## License

MIT License — see [LICENSE](LICENSE) for details.

