<?php

require_once __DIR__ . '/../vendor/autoload.php';

use Intufind\AI\Client;
use Intufind\AI\Config\Configuration;
use Intufind\AI\Exceptions\ApiException;

// Configuration
$config = new Configuration([
    'api_endpoint' => 'https://your-api-gateway-endpoint.com/Prod',
    'license_key' => 'your-license-key-here',
    'debug' => false, // Disable debug for cleaner output
]);

$client = new Client($config);

try {
    echo "🔍 Advanced Search Examples\n\n";

    // Test connection
    if (!$client->testConnection()) {
        throw new Exception("Cannot connect to API");
    }

    // Example 1: Basic semantic search
    echo "=== Example 1: Basic Semantic Search ===\n";
    $results = $client->products()->search([
        'text' => 'high quality wireless headphones with noise cancellation',
        'limit' => 5
    ]);

    echo "Found {$results->getTotal()} products:\n";
    foreach ($results->getResults() as $product) {
        $score = $product->score ? sprintf('%.3f', $product->score) : 'N/A';
        echo "- {$product->name} (Score: {$score})\n";
        if ($product->matchingChunks) {
            echo "  Matching text: " . substr($product->matchingChunks[0]['chunk_text'] ?? '', 0, 100) . "...\n";
        }
    }
    echo "\n";

    // Example 2: Search with filters
    echo "=== Example 2: Search with Filters ===\n";
    $results = $client->products()->search([
        'text' => 'electronics',
        'filters' => [
            'price_min' => 50,
            'price_max' => 300,
            'categories' => ['electronics'],
            'on_sale' => true
        ],
        'limit' => 5
    ]);

    echo "Found {$results->getTotal()} products on sale in electronics ($50-$300):\n";
    foreach ($results->getResults() as $product) {
        $price = $product->price ? '$' . number_format($product->price, 2) : 'N/A';
        echo "- {$product->name} ({$price})\n";
    }
    echo "\n";

    // Example 3: AI-enhanced search
    echo "=== Example 3: AI-Enhanced Search ===\n";
    $results = $client->products()->searchWithAgent([
        'text' => 'I need something for working out at home',
        'limit' => 5
    ]);

    echo "AI-enhanced search results:\n";
    if ($results->wasAgentUsed()) {
        echo "✅ AI agent was used for this search\n";
        if ($results->wasEnhanced()) {
            echo "✅ Query was enhanced: '{$results->getOriginalText()}'\n";
        }
        if ($results->getComplexity()) {
            echo "Query complexity: {$results->getComplexity()}\n";
        }
    }

    foreach ($results->getResults() as $product) {
        echo "- {$product->name}\n";
    }
    echo "\n";

    // Example 4: Advanced product filtering
    echo "=== Example 4: Advanced Product Filtering ===\n";
    
    // Search by category
    $categoryResults = $client->products()->getByCategory('electronics', [
        'limit' => 3
    ]);
    echo "Electronics products: {$categoryResults->getTotal()}\n";

    // Search by price range
    $priceResults = $client->products()->getByPriceRange(100, 500, [
        'limit' => 3
    ]);
    echo "Products in $100-$500 range: {$priceResults->getTotal()}\n";

    // Search for sale items
    $saleResults = $client->products()->getOnSale([
        'limit' => 3
    ]);
    echo "Products on sale: {$saleResults->getTotal()}\n";
    echo "\n";

    // Example 5: Content search
    echo "=== Example 5: Content Search ===\n";
    $contentResults = $client->posts()->search([
        'text' => 'artificial intelligence machine learning',
        'filters' => [
            'status' => 'publish',
            'post_type' => 'post'
        ],
        'limit' => 3
    ]);

    echo "Found {$contentResults->getTotal()} published posts about AI/ML:\n";
    foreach ($contentResults->getResults() as $post) {
        echo "- {$post->title}\n";
        if ($post->excerpt) {
            echo "  {$post->excerpt}\n";
        }
    }
    echo "\n";

    // Example 6: Advanced content filtering
    echo "=== Example 6: Advanced Content Filtering ===\n";
    
    // Posts by category
    $techPosts = $client->posts()->getByCategory('technology', [
        'limit' => 3
    ]);
    echo "Technology posts: {$techPosts->getTotal()}\n";

    // Recent posts
    $recentPosts = $client->posts()->getRecent(30, [
        'limit' => 3
    ]);
    echo "Posts from last 30 days: {$recentPosts->getTotal()}\n";

    // Posts by type
    $blogPosts = $client->posts()->getByType('post', [
        'limit' => 3
    ]);
    echo "Blog posts: {$blogPosts->getTotal()}\n";
    echo "\n";

    // Example 7: Complex OpenSearch queries
    echo "=== Example 7: Complex OpenSearch Queries ===\n";
    
    // Boolean query with multiple conditions
    $complexResults = $client->products()->search([
        'text' => 'headphones',
        'filters' => [
            'bool' => [
                'must' => [
                    ['terms' => ['categories' => ['electronics', 'audio']]],
                    ['range' => ['price' => ['gte' => 50, 'lte' => 500]]]
                ],
                'should' => [
                    ['term' => ['on_sale' => true]],
                    ['terms' => ['tags' => ['premium', 'bestseller']]]
                ],
                'must_not' => [
                    ['term' => ['status' => 'discontinued']]
                ]
            ]
        ],
        'limit' => 5
    ]);

    echo "Complex query results: {$complexResults->getTotal()} products\n";
    foreach ($complexResults->getResults() as $product) {
        echo "- {$product->name}\n";
    }
    echo "\n";

    // Example 8: Similarity search
    echo "=== Example 8: Similarity Search ===\n";
    
    // First, get a product to find similar items for
    $baseSearch = $client->products()->search([
        'text' => 'wireless headphones',
        'limit' => 1
    ]);

    if ($baseSearch->hasResults()) {
        $baseProduct = $baseSearch->getFirstResult();
        echo "Finding products similar to: {$baseProduct->name}\n";
        
        $similarProducts = $client->products()->findSimilar($baseProduct->id, [
            'limit' => 3
        ]);
        
        echo "Similar products ({$similarProducts->getTotal()}):\n";
        foreach ($similarProducts->getResults() as $product) {
            echo "- {$product->name}\n";
        }
    } else {
        echo "No base product found for similarity search\n";
    }
    echo "\n";

    // Example 9: Faceted Search for Filter UI
    echo "=== Example 9: Faceted Search for Filter UI ===\n";
    $facetedResults = $client->products()->searchWithFacets([
        'text' => 'electronics',
        'limit' => 5,
        'facetFields' => ['categories', 'brands', 'stock_status', 'price_ranges'],
        'facetSize' => 10
    ]);

    echo "Search results: {$facetedResults->getTotal()} products\n";
    
    if ($facetedResults->hasFacets()) {
        echo "\nAvailable Filters:\n";
        
        // Display category facets
        if ($categoryFacets = $facetedResults->getFacet('categories')) {
            echo "\n📁 Categories:\n";
            foreach (array_slice($categoryFacets, 0, 5) as $bucket) {
                echo "   - {$bucket['key']} ({$bucket['count']})\n";
            }
        }
        
        // Display brand facets
        if ($brandFacets = $facetedResults->getFacet('brands')) {
            echo "\n🏷️ Brands:\n";
            foreach (array_slice($brandFacets, 0, 5) as $bucket) {
                echo "   - {$bucket['key']} ({$bucket['count']})\n";
            }
        }
        
        // Get facet counts as simple array (useful for UI dropdowns)
        $stockCounts = $facetedResults->getFacetCounts('stock_status');
        if (!empty($stockCounts)) {
            echo "\n📦 Stock Status:\n";
            foreach ($stockCounts as $status => $count) {
                echo "   - {$status}: {$count}\n";
            }
        }
        
        // Price ranges (numeric ranges)
        if ($priceFacets = $facetedResults->getFacet('price_ranges')) {
            echo "\n💰 Price Ranges:\n";
            foreach ($priceFacets as $bucket) {
                $from = isset($bucket['from']) ? '$' . number_format($bucket['from'], 0) : '$0';
                $to = isset($bucket['to']) ? '$' . number_format($bucket['to'], 0) : '+';
                echo "   - {$from} - {$to}: {$bucket['count']} products\n";
            }
        }
    } else {
        echo "No facets returned\n";
    }
    echo "\n";

    // Example 10: Search result analysis
    echo "=== Example 10: Search Result Analysis ===\n";
    $analysisResults = $client->products()->searchWithAgent([
        'text' => 'best wireless earbuds for running',
        'limit' => 5
    ]);

    echo "Search Analysis:\n";
    echo "- Total results: {$analysisResults->getTotal()}\n";
    echo "- Agent used: " . ($analysisResults->wasAgentUsed() ? 'Yes' : 'No') . "\n";
    echo "- Query enhanced: " . ($analysisResults->wasEnhanced() ? 'Yes' : 'No') . "\n";
    echo "- Fallback used: " . ($analysisResults->wasFallbackUsed() ? 'Yes' : 'No') . "\n";
    
    if ($analysisResults->getClassification()) {
        $classification = $analysisResults->getClassification();
        echo "- Query complexity: {$classification['complexity']}\n";
        echo "- Complexity score: {$classification['score']}\n";
        if (isset($classification['reasons'])) {
            echo "- Complexity reasons: " . implode(', ', $classification['reasons']) . "\n";
        }
    }

    if ($analysisResults->getUsageMultiplier()) {
        echo "- Usage multiplier: {$analysisResults->getUsageMultiplier()}\n";
    }
    echo "\n";

    echo "✅ Advanced search examples completed!\n";
    echo "\n💡 Search Tips:\n";
    echo "- Use semantic search for natural language queries\n";
    echo "- Combine filters for precise results\n";
    echo "- Use AI-enhanced search for complex queries\n";
    echo "- Leverage similarity search for recommendations\n";
    echo "- Use faceted search to build dynamic filter UIs\n";
    echo "- Monitor search analytics for optimization\n";

} catch (ApiException $e) {
    echo "❌ API Error: " . $e->getMessage() . "\n";
    echo "Status Code: " . $e->getStatusCode() . "\n";
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}
