<?php

require_once __DIR__ . '/../vendor/autoload.php';

use Intufind\AI\Client;
use Intufind\AI\Config\Configuration;
use Intufind\AI\Exceptions\ApiException;
use Intufind\AI\Exceptions\AuthenticationException;

// Basic configuration
$config = new Configuration([
    'api_endpoint' => 'https://your-api-gateway-endpoint.com/Prod',
    'license_key' => 'your-license-key-here',
    'workspace_id' => 'your-workspace-id', // Optional
    'debug' => true, // Enable debug logging
]);

// Create the client
$client = new Client($config);

try {
    // Test the connection
    echo "Testing connection...\n";
    if ($client->testConnection()) {
        echo "✅ Connection successful!\n\n";
    } else {
        echo "❌ Connection failed!\n\n";
        exit(1);
    }

    // Validate license
    echo "Validating license...\n";
    if ($client->validateLicense()) {
        echo "✅ License is valid!\n";
        
        // Get tier information
        $tierInfo = $client->license()->getTierInfo();
        echo "Tier: " . ($tierInfo['tier'] ?? 'Unknown') . "\n\n";
    } else {
        echo "❌ License validation failed!\n\n";
    }

    // Example 1: Add a product
    echo "Adding a product...\n";
    $productResult = $client->products()->upsert([
        'id' => 'demo-product-' . time(),
        'name' => 'Demo Wireless Headphones',
        'content' => 'High-quality wireless headphones with noise cancellation and long battery life.',
        'price' => 199.99,
        'categories' => ['electronics', 'audio'],
        'tags' => ['wireless', 'noise-cancelling', 'bluetooth'],
        'attributes' => [
            'color' => ['Black', 'White'],
            'brand' => ['DemoTech'],
            'warranty' => ['2 years']
        ]
    ]);
    echo "✅ Product added successfully!\n";
    echo "Result: " . json_encode($productResult, JSON_PRETTY_PRINT) . "\n\n";

    // Example 2: Search products
    echo "Searching for products...\n";
    $searchResult = $client->products()->search([
        'text' => 'wireless headphones',
        'limit' => 5
    ]);
    
    echo "Found " . $searchResult->getTotal() . " products:\n";
    foreach ($searchResult->getResults() as $product) {
        echo "- " . $product->name . " ($" . $product->price . ")\n";
    }
    echo "\n";

    // Example 3: Add a blog post
    echo "Adding a blog post...\n";
    $postResult = $client->posts()->upsert([
        'id' => 'demo-post-' . time(),
        'title' => 'The Future of Wireless Audio Technology',
        'content' => 'Wireless audio technology continues to evolve with better sound quality, longer battery life, and improved connectivity...',
        'excerpt' => 'Exploring the latest trends in wireless audio technology.',
        'status' => 'publish',
        'categories' => ['technology', 'audio'],
        'tags' => ['wireless', 'bluetooth', 'audio-tech']
    ]);
    echo "✅ Post added successfully!\n";
    echo "Result: " . json_encode($postResult, JSON_PRETTY_PRINT) . "\n\n";

    // Example 4: Search posts
    echo "Searching for posts...\n";
    $postSearchResult = $client->posts()->search([
        'text' => 'wireless audio technology',
        'limit' => 3
    ]);
    
    echo "Found " . $postSearchResult->getTotal() . " posts:\n";
    foreach ($postSearchResult->getResults() as $post) {
        echo "- " . $post->title . "\n";
    }
    echo "\n";

    // Example 5: Chat with AI (if streaming endpoint is configured)
    if ($config->getStreamingEndpoint()) {
        echo "Starting AI chat...\n";
        $chatResponse = $client->chat()->send([
            'message' => 'Tell me about wireless headphones',
            'threadId' => 'demo-thread-' . time()
        ]);
        
        echo "AI Response: " . $chatResponse->getIntro() . "\n";
        if ($chatResponse->getMessage()->hasProducts()) {
            echo "Products found: " . count($chatResponse->getProducts()) . "\n";
        }
        echo "\n";
    } else {
        echo "Skipping chat example (streaming endpoint not configured)\n\n";
    }

    // Example 6: Get recommendations
    echo "Getting product recommendations...\n";
    $recommendations = $client->recommendations()->getTrending([
        'limit' => 3
    ]);
    
    if (isset($recommendations['recommendations'])) {
        echo "Trending products:\n";
        foreach ($recommendations['recommendations'] as $rec) {
            $product = $rec['product'];
            echo "- " . $product->name . " (Score: " . $rec['score'] . ")\n";
        }
    } else {
        echo "No recommendations available\n";
    }
    echo "\n";

    echo "✅ All examples completed successfully!\n";

} catch (AuthenticationException $e) {
    echo "❌ Authentication failed: " . $e->getMessage() . "\n";
    echo "Please check your license key.\n";
} catch (ApiException $e) {
    echo "❌ API Error: " . $e->getMessage() . "\n";
    echo "Status Code: " . $e->getStatusCode() . "\n";
    if ($e->getErrorDetails()) {
        echo "Details: " . json_encode($e->getErrorDetails(), JSON_PRETTY_PRINT) . "\n";
    }
} catch (Exception $e) {
    echo "❌ Unexpected error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}
