<?php

require_once __DIR__ . '/../vendor/autoload.php';

use Intufind\AI\Client;
use Intufind\AI\Config\Configuration;
use Intufind\AI\Exceptions\ApiException;

// Configuration
$config = new Configuration([
    'api_endpoint' => 'https://your-api-gateway-endpoint.com/Prod',
    'license_key' => 'your-license-key-here',
    'debug' => true,
]);

$client = new Client($config);

try {
    echo "🚀 Starting bulk operations example...\n\n";

    // Test connection
    if (!$client->testConnection()) {
        throw new Exception("Cannot connect to API");
    }

    // Example 1: Bulk product upsert
    echo "=== Example 1: Bulk Product Upsert ===\n";
    
    $products = [];
    for ($i = 1; $i <= 5; $i++) {
        $products[] = [
            'id' => 'bulk-product-' . $i . '-' . time(),
            'name' => "Bulk Product {$i}",
            'content' => "This is bulk product number {$i} for testing bulk operations.",
            'price' => 99.99 + ($i * 10),
            'categories' => ['bulk-test', 'electronics'],
            'tags' => ['bulk', 'test', "product{$i}"],
            'attributes' => [
                'color' => ['Red', 'Blue', 'Green'],
                'size' => ['Small', 'Medium', 'Large'],
                'bulk_batch' => [date('Y-m-d')]
            ]
        ];
    }

    echo "Upserting " . count($products) . " products...\n";
    $bulkResult = $client->products()->bulkUpsert($products);
    
    echo "✅ Bulk upsert completed!\n";
    echo "Processed: " . $bulkResult['processed_count'] . "\n";
    echo "Successful: " . $bulkResult['success_count'] . "\n";
    echo "Errors: " . $bulkResult['error_count'] . "\n";
    
    if (!empty($bulkResult['errors'])) {
        echo "Error details:\n";
        foreach ($bulkResult['errors'] as $error) {
            echo "- " . json_encode($error) . "\n";
        }
    }
    echo "\n";

    // Example 2: Bulk post upsert
    echo "=== Example 2: Bulk Post Upsert ===\n";
    
    $posts = [];
    for ($i = 1; $i <= 3; $i++) {
        $posts[] = [
            'id' => 'bulk-post-' . $i . '-' . time(),
            'title' => "Bulk Post {$i}: Technology Trends",
            'content' => "This is bulk post number {$i} discussing the latest technology trends and innovations in the industry.",
            'excerpt' => "Exploring technology trends in post {$i}",
            'status' => 'publish',
            'categories' => ['technology', 'bulk-test'],
            'tags' => ['bulk', 'tech', 'trends', "post{$i}"],
            'post_type' => 'post'
        ];
    }

    echo "Upserting " . count($posts) . " posts...\n";
    $postBulkResult = $client->posts()->bulkUpsert($posts);
    
    echo "✅ Bulk post upsert completed!\n";
    echo "Processed: " . $postBulkResult['processed_count'] . "\n";
    echo "Successful: " . $postBulkResult['success_count'] . "\n";
    echo "Errors: " . $postBulkResult['error_count'] . "\n";
    echo "\n";

    // Example 3: Search and verify bulk data
    echo "=== Example 3: Verify Bulk Data ===\n";
    
    // Search for bulk products
    echo "Searching for bulk products...\n";
    $productSearch = $client->products()->search([
        'text' => 'bulk test electronics',
        'filters' => [
            'categories' => ['bulk-test']
        ],
        'limit' => 10
    ]);
    
    echo "Found " . $productSearch->getTotal() . " bulk products:\n";
    foreach ($productSearch->getResults() as $product) {
        echo "- {$product->name} (\${$product->price})\n";
    }
    echo "\n";

    // Search for bulk posts
    echo "Searching for bulk posts...\n";
    $postSearch = $client->posts()->search([
        'text' => 'bulk technology trends',
        'filters' => [
            'categories' => ['bulk-test']
        ],
        'limit' => 10
    ]);
    
    echo "Found " . $postSearch->getTotal() . " bulk posts:\n";
    foreach ($postSearch->getResults() as $post) {
        echo "- {$post->title}\n";
    }
    echo "\n";

    // Example 4: Get IDs for cleanup
    echo "=== Example 4: Get IDs for Cleanup ===\n";
    
    // Get product IDs
    $productIds = $client->products()->getIds([
        'categories' => 'bulk-test',
        'limit' => 100
    ]);
    
    echo "Found " . count($productIds['ids']) . " product IDs for cleanup\n";
    
    // Get post IDs  
    $postIds = $client->posts()->getIds([
        'categories' => 'bulk-test',
        'limit' => 100
    ]);
    
    echo "Found " . count($postIds['ids']) . " post IDs for cleanup\n\n";

    // Example 5: Bulk delete (uncomment to actually delete)
    echo "=== Example 5: Bulk Delete (Commented Out) ===\n";
    echo "To actually delete the bulk test data, uncomment the deletion code below.\n";
    
    /*
    if (!empty($productIds['ids'])) {
        echo "Deleting " . count($productIds['ids']) . " products...\n";
        $deleteResult = $client->products()->bulkDelete($productIds['ids']);
        echo "Deleted " . $deleteResult['success_count'] . " products\n";
    }
    
    if (!empty($postIds['ids'])) {
        echo "Deleting " . count($postIds['ids']) . " posts...\n";
        $deleteResult = $client->posts()->bulkDelete($postIds['ids']);
        echo "Deleted " . $deleteResult['success_count'] . " posts\n";
    }
    */

    // Example 6: Delete by query
    echo "\n=== Example 6: Delete by Query Example ===\n";
    echo "This shows how to delete items by query (not executed):\n";
    
    $deleteByQueryExample = [
        'filters' => [
            [
                'bool' => [
                    'must' => [
                        ['term' => ['categories' => 'bulk-test']],
                        ['range' => ['created_at' => ['lt' => date('Y-m-d\TH:i:s\Z')]]]
                    ]
                ]
            ]
        ]
    ];
    
    echo "Example delete by query filter:\n";
    echo json_encode($deleteByQueryExample, JSON_PRETTY_PRINT) . "\n";
    
    // Uncomment to actually execute:
    /*
    $deleteByQueryResult = $client->products()->deleteByQuery($deleteByQueryExample['filters']);
    echo "Delete by query result:\n";
    echo json_encode($deleteByQueryResult, JSON_PRETTY_PRINT) . "\n";
    */

    echo "\n✅ Bulk operations example completed!\n";
    echo "\n💡 Tips for bulk operations:\n";
    echo "- Process items in batches of 50-100 for optimal performance\n";
    echo "- Check your tier limits before running large bulk operations\n";
    echo "- Use the getIds() method for efficient cleanup operations\n";
    echo "- Use deleteByQuery() for complex deletion criteria\n";
    echo "- Always handle errors gracefully in production code\n";

} catch (ApiException $e) {
    echo "❌ API Error: " . $e->getMessage() . "\n";
    echo "Status Code: " . $e->getStatusCode() . "\n";
    if ($e->getErrorDetails()) {
        echo "Details: " . json_encode($e->getErrorDetails(), JSON_PRETTY_PRINT) . "\n";
    }
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}
