<?php

require_once __DIR__ . '/../vendor/autoload.php';

use Intufind\AI\Client;

// Example: Using the Feedback Service
try {
    // Initialize the client with your configuration
    $client = new Client([
        'license_key' => 'your-license-key-here',
        'api_endpoint' => 'https://your-api-endpoint.com',
        // For local development:
        // 'api_endpoint' => 'http://localhost:3000',
    ]);

    echo "🔄 Feedback Service Examples\n";
    echo "===========================\n\n";

    // Example 1: Submit positive feedback for a chat response
    echo "1. Submitting positive feedback for chat response...\n";
    $feedbackResult = $client->feedback()->submit([
        'rating' => 'positive',
        'source' => 'chat_response',
        'thread_id' => 'thread_123',
        'content_id' => 'msg_1694123456789_a1b2c3d4',
        'content_type' => 'message',
        'comment' => 'This response was very helpful and accurate!',
        'category' => 'helpfulness',
        'query_text' => 'How do I reset my password?',
        'user_id' => 'user_456',
    ]);

    if ($feedbackResult['success']) {
        echo "✅ Feedback submitted successfully!\n";
        echo "   Feedback ID: {$feedbackResult['feedback_id']}\n";
        echo "   Message: {$feedbackResult['message']}\n\n";
    }

    // Example 2: Submit negative feedback for a recommendation
    echo "2. Submitting negative feedback for recommendation...\n";
    $negativeResult = $client->feedback()->submit([
        'rating' => 'negative',
        'source' => 'recommendation',
        'content_id' => 'product_789',
        'content_type' => 'product',
        'comment' => 'This recommendation was not relevant to my needs.',
        'category' => 'relevance',
        'session_id' => 'session_abc123',
    ]);

    if ($negativeResult['success']) {
        echo "✅ Negative feedback submitted!\n";
        echo "   Feedback ID: {$negativeResult['feedback_id']}\n\n";
    }

    // Example 3: Get feedback for a specific thread
    echo "3. Retrieving feedback for thread...\n";
    $threadFeedback = $client->feedback()->getThreadFeedback('thread_123');
    
    echo "📊 Found " . count($threadFeedback) . " feedback items for thread:\n";
    foreach ($threadFeedback as $feedback) {
        echo "   - {$feedback->rating} feedback";
        if ($feedback->comment) {
            echo " with comment: \"{$feedback->comment}\"";
        }
        echo " (created: {$feedback->created_at})\n";
    }
    echo "\n";

    // Example 4: Search feedback (admin only)
    echo "4. Searching feedback (admin functionality)...\n";
    $searchResults = $client->feedback()->search([
        'rating' => 'positive',
        'source' => 'chat_response',
        'limit' => 5,
        'sort_by' => 'created_at',
        'sort_order' => 'desc',
        'include_aggregations' => true,
    ]);

    echo "📈 Search results:\n";
    echo "   - Total feedback: {$searchResults['total_count']}\n";
    echo "   - Returned: " . count($searchResults['results']) . " items\n";
    foreach ($searchResults['results'] as $feedback) {
        echo "     * {$feedback->rating} - {$feedback->source}";
        if ($feedback->category) {
            echo " ({$feedback->category})";
        }
        echo "\n";
    }
    echo "\n";

    // Example 5: Get analytics
    echo "5. Getting feedback analytics...\n";
    $analytics = $client->feedback()->getAnalytics([
        'date_from' => date('Y-m-d', strtotime('-30 days')),
        'date_to' => date('Y-m-d'),
    ]);

    echo "📊 Analytics for last 30 days:\n";
    echo "   - Total feedback: {$analytics['total_feedback']}\n";
    echo "   - Positive: {$analytics['positive_feedback']}\n";
    echo "   - Negative: {$analytics['negative_feedback']}\n";
    echo "   - Positive percentage: {$analytics['positive_percentage']}%\n";
    
    if (isset($analytics['feedback_by_source'])) {
        echo "   - By source:\n";
        foreach ($analytics['feedback_by_source'] as $source => $stats) {
            echo "     * {$source}: {$stats['total']} total ({$stats['positive_percentage']}% positive)\n";
        }
    }
    echo "\n";

    // Example 6: Delete feedback (if you have the feedback ID)
    if (isset($feedbackResult['feedback_id'])) {
        echo "6. Deleting feedback...\n";
        $deleteResult = $client->feedback()->delete($feedbackResult['feedback_id']);
        
        if ($deleteResult['success']) {
            echo "✅ Feedback deleted successfully!\n";
        }
    }

} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}

echo "\n🎉 Feedback examples completed!\n";
