<?php

require_once __DIR__ . '/../vendor/autoload.php';

use Intufind\AI\Client;
use Intufind\AI\Config\Configuration;
use Intufind\AI\Exceptions\ApiException;

// Configuration
$config = new Configuration([
    'api_endpoint' => 'https://your-api-gateway-endpoint.com/Prod',
    'license_key' => 'your-license-key-here',
    'debug' => false,
]);

$client = new Client($config);

try {
    echo "🎯 Product Recommendations Examples\n\n";

    // Test connection
    if (!$client->testConnection()) {
        throw new Exception("Cannot connect to API");
    }

    // Example 1: Get trending products
    echo "=== Example 1: Trending Products ===\n";
    $trending = $client->recommendations()->getTrending([
        'limit' => 5
    ]);

    if (isset($trending['recommendations'])) {
        echo "Trending products:\n";
        foreach ($trending['recommendations'] as $rec) {
            $product = $rec['product'];
            $score = isset($rec['score']) ? sprintf('%.3f', $rec['score']) : 'N/A';
            echo "- {$product->name} (Score: {$score})\n";
            if ($product->price) {
                echo "  Price: \${$product->price}\n";
            }
        }
    } else {
        echo "No trending products available\n";
    }
    echo "\n";

    // Example 2: Get popular products
    echo "=== Example 2: Popular Products ===\n";
    $popular = $client->recommendations()->getPopular([
        'limit' => 5
    ]);

    if (isset($popular['recommendations'])) {
        echo "Popular products:\n";
        foreach ($popular['recommendations'] as $rec) {
            $product = $rec['product'];
            echo "- {$product->name}\n";
            if (!empty($product->categories)) {
                echo "  Categories: " . implode(', ', $product->categories) . "\n";
            }
        }
    } else {
        echo "No popular products available\n";
    }
    echo "\n";

    // Example 3: Product-based recommendations
    echo "=== Example 3: Product-based Recommendations ===\n";
    
    // First, find a product to base recommendations on
    $productSearch = $client->products()->search([
        'text' => 'headphones',
        'limit' => 1
    ]);

    if ($productSearch->hasResults()) {
        $baseProduct = $productSearch->getFirstResult();
        echo "Getting recommendations based on: {$baseProduct->name}\n";
        
        try {
            $productRecs = $client->recommendations()->getForProduct($baseProduct->id, [
                'limit' => 5
            ]);

            if (isset($productRecs['recommendations'])) {
                echo "Products similar to {$baseProduct->name}:\n";
                foreach ($productRecs['recommendations'] as $rec) {
                    $product = $rec['product'];
                    echo "- {$product->name}\n";
                }
            } else {
                echo "No product-based recommendations available\n";
            }
        } catch (Exception $e) {
            echo "Product recommendations not available: " . $e->getMessage() . "\n";
        }
    } else {
        echo "No base product found for recommendations\n";
    }
    echo "\n";

    // Example 4: Cart-based recommendations
    echo "=== Example 4: Cart-based Recommendations ===\n";
    
    // Simulate a shopping cart
    $cartItems = [];
    $cartSearch = $client->products()->search([
        'text' => 'electronics',
        'limit' => 3
    ]);

    foreach ($cartSearch->getResults() as $product) {
        $cartItems[] = $product->id;
    }

    if (!empty($cartItems)) {
        echo "Cart contains " . count($cartItems) . " items\n";
        
        try {
            $cartRecs = $client->recommendations()->getForCart($cartItems, [
                'limit' => 5
            ]);

            if (isset($cartRecs['recommendations'])) {
                echo "Recommendations based on cart contents:\n";
                foreach ($cartRecs['recommendations'] as $rec) {
                    $product = $rec['product'];
                    echo "- {$product->name}\n";
                    if (isset($rec['type'])) {
                        echo "  Type: {$rec['type']}\n";
                    }
                }
            } else {
                echo "No cart-based recommendations available\n";
            }
        } catch (Exception $e) {
            echo "Cart recommendations not available: " . $e->getMessage() . "\n";
        }
    } else {
        echo "No items in cart for recommendations\n";
    }
    echo "\n";

    // Example 5: Category-based recommendations
    echo "=== Example 5: Category-based Recommendations ===\n";
    
    try {
        $categoryRecs = $client->recommendations()->getForCategory(['electronics', 'audio'], [
            'limit' => 5
        ]);

        if (isset($categoryRecs['recommendations'])) {
            echo "Recommendations for electronics/audio categories:\n";
            foreach ($categoryRecs['recommendations'] as $rec) {
                $product = $rec['product'];
                echo "- {$product->name}\n";
            }
        } else {
            echo "No category-based recommendations available\n";
        }
    } catch (Exception $e) {
        echo "Category recommendations not available: " . $e->getMessage() . "\n";
    }
    echo "\n";

    // Example 6: Search query-based recommendations
    echo "=== Example 6: Query-based Recommendations ===\n";
    
    try {
        $queryRecs = $client->recommendations()->getForQuery('wireless gaming headset', [
            'limit' => 5
        ]);

        if (isset($queryRecs['recommendations'])) {
            echo "Recommendations for 'wireless gaming headset':\n";
            foreach ($queryRecs['recommendations'] as $rec) {
                $product = $rec['product'];
                echo "- {$product->name}\n";
                if ($product->price) {
                    echo "  Price: \${$product->price}\n";
                }
            }
        } else {
            echo "No query-based recommendations available\n";
        }
    } catch (Exception $e) {
        echo "Query recommendations not available: " . $e->getMessage() . "\n";
    }
    echo "\n";

    // Example 7: Cross-sell recommendations
    echo "=== Example 7: Cross-sell Recommendations ===\n";
    
    if ($productSearch->hasResults()) {
        $baseProduct = $productSearch->getFirstResult();
        
        try {
            $crossSell = $client->recommendations()->getCrossSell($baseProduct->id, [
                'limit' => 3
            ]);

            if (isset($crossSell['recommendations'])) {
                echo "Cross-sell products for {$baseProduct->name}:\n";
                foreach ($crossSell['recommendations'] as $rec) {
                    $product = $rec['product'];
                    echo "- {$product->name}\n";
                }
            } else {
                echo "No cross-sell recommendations available\n";
            }
        } catch (Exception $e) {
            echo "Cross-sell recommendations not available: " . $e->getMessage() . "\n";
        }
    }
    echo "\n";

    // Example 8: Upsell recommendations
    echo "=== Example 8: Upsell Recommendations ===\n";
    
    if ($productSearch->hasResults()) {
        $baseProduct = $productSearch->getFirstResult();
        
        try {
            $upsell = $client->recommendations()->getUpsell($baseProduct->id, [
                'limit' => 3
            ]);

            if (isset($upsell['recommendations'])) {
                echo "Upsell products for {$baseProduct->name}:\n";
                foreach ($upsell['recommendations'] as $rec) {
                    $product = $rec['product'];
                    echo "- {$product->name}\n";
                    if ($product->price && $baseProduct->price) {
                        $priceDiff = $product->price - $baseProduct->price;
                        echo "  Price difference: \${$priceDiff}\n";
                    }
                }
            } else {
                echo "No upsell recommendations available\n";
            }
        } catch (Exception $e) {
            echo "Upsell recommendations not available: " . $e->getMessage() . "\n";
        }
    }
    echo "\n";

    // Example 9: Price-filtered recommendations
    echo "=== Example 9: Price-filtered Recommendations ===\n";
    
    try {
        $priceFiltered = $client->recommendations()->getWithPriceFilter([
            'strategy' => 'popular',
            'limit' => 5
        ], 50.0, 200.0);

        if (isset($priceFiltered['recommendations'])) {
            echo "Popular products in $50-$200 range:\n";
            foreach ($priceFiltered['recommendations'] as $rec) {
                $product = $rec['product'];
                $price = $product->price ? '$' . number_format($product->price, 2) : 'N/A';
                echo "- {$product->name} ({$price})\n";
            }
        } else {
            echo "No price-filtered recommendations available\n";
        }
    } catch (Exception $e) {
        echo "Price-filtered recommendations not available: " . $e->getMessage() . "\n";
    }
    echo "\n";

    echo "✅ Recommendations examples completed!\n";
    echo "\n💡 Recommendation Tips:\n";
    echo "- Use different strategies for different use cases\n";
    echo "- Combine product and cart data for better results\n";
    echo "- Filter by price range to match customer budgets\n";
    echo "- Use cross-sell and upsell for revenue optimization\n";
    echo "- Monitor recommendation performance and adjust strategies\n";

} catch (ApiException $e) {
    echo "❌ API Error: " . $e->getMessage() . "\n";
    echo "Status Code: " . $e->getStatusCode() . "\n";
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}
