<?php

/**
 * Intucart AI PHP SDK Setup Guide
 * 
 * This file demonstrates how to properly set up and configure the SDK
 * for different environments and use cases.
 */

require_once __DIR__ . '/../vendor/autoload.php';

use Intufind\AI\Client;
use Intufind\AI\Config\Configuration;
use Monolog\Logger;
use Monolog\Handler\StreamHandler;

echo "🚀 Intucart AI PHP SDK Setup Guide\n\n";

// ========================================
// Example 1: Basic Setup
// ========================================
echo "=== Example 1: Basic Setup ===\n";

$basicConfig = new Configuration([
    'api_endpoint' => 'https://your-api-gateway-endpoint.com/Prod',
    'license_key' => 'your-license-key-here',
]);

$basicClient = new Client($basicConfig);
echo "✅ Basic client created successfully\n\n";

// ========================================
// Example 2: Advanced Configuration
// ========================================
echo "=== Example 2: Advanced Configuration ===\n";

// Create a custom logger
$logger = new Logger('intucart-ai');
$logger->pushHandler(new StreamHandler('php://stdout', Logger::DEBUG));

$advancedConfig = new Configuration([
    'api_endpoint' => 'https://your-api-gateway-endpoint.com/Prod',
    'streaming_endpoint' => 'https://streaming.intufind.com',
    'license_key' => 'your-license-key-here',
    'workspace_id' => 'your-workspace-id',
    'timeout' => 30,
    'connect_timeout' => 10,
    'retry_attempts' => 3,
    'debug' => true,
    'logger' => $logger,
    'user_agent' => 'MyApp/1.0 (Intucart-AI-PHP-SDK)',
    'http_client_options' => [
        'verify' => true, // SSL verification
        'headers' => [
            'X-Custom-Header' => 'MyValue'
        ]
    ]
]);

$advancedClient = new Client($advancedConfig);
echo "✅ Advanced client created with custom configuration\n\n";

// ========================================
// Example 3: Environment-based Configuration
// ========================================
echo "=== Example 3: Environment-based Configuration ===\n";

function createClientForEnvironment($env) {
    $configs = [
        'development' => [
            'api_endpoint' => 'https://dev-api.example.com/Prod',
            'streaming_endpoint' => 'https://dev-streaming.example.com',
            'license_key' => 'dev-license-key',
            'debug' => true,
            'timeout' => 60, // Longer timeout for dev
        ],
        'staging' => [
            'api_endpoint' => 'https://staging-api.example.com/Prod',
            'streaming_endpoint' => 'https://staging-streaming.example.com',
            'license_key' => 'staging-license-key',
            'debug' => false,
            'timeout' => 30,
        ],
        'production' => [
            'api_endpoint' => 'https://api.example.com/Prod',
            'streaming_endpoint' => 'https://streaming.example.com',
            'license_key' => 'production-license-key',
            'debug' => false,
            'timeout' => 30,
            'retry_attempts' => 5, // More retries in production
        ],
    ];

    if (!isset($configs[$env])) {
        throw new InvalidArgumentException("Unknown environment: {$env}");
    }

    return new Client(new Configuration($configs[$env]));
}

$devClient = createClientForEnvironment('development');
echo "✅ Development client created\n";

$prodClient = createClientForEnvironment('production');
echo "✅ Production client created\n\n";

// ========================================
// Example 4: Configuration from Environment Variables
// ========================================
echo "=== Example 4: Environment Variables Configuration ===\n";

function createClientFromEnv() {
    $config = [
        'api_endpoint' => $_ENV['INTUCART_API_ENDPOINT'] ?? getenv('INTUCART_API_ENDPOINT'),
        'streaming_endpoint' => $_ENV['INTUCART_STREAMING_ENDPOINT'] ?? getenv('INTUCART_STREAMING_ENDPOINT'),
        'license_key' => $_ENV['INTUCART_LICENSE_KEY'] ?? getenv('INTUCART_LICENSE_KEY'),
        'workspace_id' => $_ENV['INTUCART_WORKSPACE_ID'] ?? getenv('INTUCART_WORKSPACE_ID'),
        'debug' => filter_var($_ENV['INTUCART_DEBUG'] ?? getenv('INTUCART_DEBUG') ?? 'false', FILTER_VALIDATE_BOOLEAN),
    ];

    // Remove null values
    $config = array_filter($config, function($value) {
        return $value !== null && $value !== false && $value !== '';
    });

    if (empty($config['api_endpoint']) || empty($config['license_key'])) {
        throw new InvalidArgumentException('INTUCART_API_ENDPOINT and INTUCART_LICENSE_KEY environment variables are required');
    }

    return new Client(new Configuration($config));
}

// Example of setting environment variables (in practice, set these in your environment)
putenv('INTUCART_API_ENDPOINT=https://your-api-gateway-endpoint.com/Prod');
putenv('INTUCART_LICENSE_KEY=your-license-key-here');
putenv('INTUCART_DEBUG=true');

try {
    $envClient = createClientFromEnv();
    echo "✅ Client created from environment variables\n";
} catch (Exception $e) {
    echo "❌ Failed to create client from environment: " . $e->getMessage() . "\n";
}
echo "\n";

// ========================================
// Example 5: Configuration Validation
// ========================================
echo "=== Example 5: Configuration Validation ===\n";

function validateClientConfiguration($client) {
    $results = [];
    
    // Test basic connectivity
    $results['connectivity'] = $client->testConnection();
    
    // Validate license
    $results['license'] = $client->validateLicense();
    
    // Check if streaming is available
    $results['streaming'] = $client->health()->isStreamingAvailable();
    
    // Get tier information
    try {
        $tierInfo = $client->license()->getTierInfo();
        $results['tier'] = $tierInfo['tier'] ?? 'unknown';
        $results['tier_limits'] = $tierInfo['limits'] ?? [];
    } catch (Exception $e) {
        $results['tier'] = 'error: ' . $e->getMessage();
    }
    
    return $results;
}

$validation = validateClientConfiguration($basicClient);

echo "Configuration validation results:\n";
echo "- Connectivity: " . ($validation['connectivity'] ? '✅ OK' : '❌ Failed') . "\n";
echo "- License: " . ($validation['license'] ? '✅ Valid' : '❌ Invalid') . "\n";
echo "- Streaming: " . ($validation['streaming'] ? '✅ Available' : '❌ Not available') . "\n";
echo "- Tier: " . $validation['tier'] . "\n";

if (isset($validation['tier_limits'])) {
    echo "- Available features:\n";
    foreach ($validation['tier_limits'] as $feature => $limits) {
        if (is_array($limits) && isset($limits['enabled'])) {
            echo "  - {$feature}: " . ($limits['enabled'] ? '✅ Enabled' : '❌ Disabled') . "\n";
        }
    }
}
echo "\n";

// ========================================
// Example 6: Client Configuration Updates
// ========================================
echo "=== Example 6: Dynamic Configuration Updates ===\n";

// Start with basic client
$client = new Client($basicConfig);
echo "Original client created\n";

// Create a new client with debug enabled
$debugClient = $client->withDebug();
echo "Debug client created\n";

// Create a client with different workspace ID
$workspaceClient = $client->withWorkspaceId('different-workspace-id');
echo "Workspace-specific client created\n";

// Create a client with different license key
$newLicenseClient = $client->withLicenseKey('different-license-key');
echo "Client with new license key created\n";

// Create a client with custom configuration
$customClient = $client->withConfig([
    'timeout' => 60,
    'retry_attempts' => 5,
    'debug' => true
]);
echo "Custom configured client created\n\n";

// ========================================
// Example 7: Error Handling Setup
// ========================================
echo "=== Example 7: Error Handling Setup ===\n";

use Intufind\AI\Exceptions\ApiException;
use Intufind\AI\Exceptions\AuthenticationException;
use Intufind\AI\Exceptions\ValidationException;
use Intufind\AI\Exceptions\RateLimitException;
use Intufind\AI\Exceptions\NetworkException;

function demonstrateErrorHandling($client) {
    try {
        // This might fail for demonstration
        $result = $client->products()->search([
            'text' => 'test query',
            'limit' => 5
        ]);
        
        echo "✅ Search completed successfully\n";
        return $result;
        
    } catch (AuthenticationException $e) {
        echo "❌ Authentication Error: " . $e->getMessage() . "\n";
        echo "Check your license key configuration\n";
    } catch (ValidationException $e) {
        echo "❌ Validation Error: " . $e->getMessage() . "\n";
        echo "Check your request parameters\n";
    } catch (RateLimitException $e) {
        echo "❌ Rate Limit Error: " . $e->getMessage() . "\n";
        echo "Wait before making more requests\n";
    } catch (NetworkException $e) {
        echo "❌ Network Error: " . $e->getMessage() . "\n";
        echo "Check your internet connection and API endpoint\n";
    } catch (ApiException $e) {
        echo "❌ API Error: " . $e->getMessage() . "\n";
        echo "Status Code: " . $e->getStatusCode() . "\n";
        if ($e->getErrorDetails()) {
            echo "Details: " . json_encode($e->getErrorDetails(), JSON_PRETTY_PRINT) . "\n";
        }
    } catch (Exception $e) {
        echo "❌ Unexpected Error: " . $e->getMessage() . "\n";
    }
    
    return null;
}

demonstrateErrorHandling($basicClient);
echo "\n";

// ========================================
// Example 8: Best Practices Summary
// ========================================
echo "=== Example 8: Best Practices Summary ===\n";
echo "📋 Configuration Best Practices:\n";
echo "1. Store sensitive data (license keys) in environment variables\n";
echo "2. Use different configurations for different environments\n";
echo "3. Enable debug mode only in development\n";
echo "4. Set appropriate timeouts for your use case\n";
echo "5. Configure retry attempts based on your reliability needs\n";
echo "6. Use custom loggers for better monitoring\n";
echo "7. Validate your configuration on startup\n";
echo "8. Handle all exception types appropriately\n";
echo "9. Monitor your usage and tier limits\n";
echo "10. Keep your SDK version updated\n\n";

echo "🔧 Performance Tips:\n";
echo "1. Reuse client instances when possible\n";
echo "2. Use bulk operations for multiple items\n";
echo "3. Implement proper caching strategies\n";
echo "4. Monitor API response times\n";
echo "5. Use streaming for real-time features\n";
echo "6. Implement exponential backoff for retries\n";
echo "7. Set reasonable limits for search results\n";
echo "8. Use filters to narrow search scope\n\n";

echo "🔒 Security Recommendations:\n";
echo "1. Never hardcode license keys in your code\n";
echo "2. Use HTTPS endpoints only\n";
echo "3. Validate SSL certificates\n";
echo "4. Implement proper access controls\n";
echo "5. Log security-relevant events\n";
echo "6. Regularly rotate license keys\n";
echo "7. Monitor for suspicious usage patterns\n\n";

echo "✅ Setup guide completed!\n";
echo "You're now ready to use the Intucart AI PHP SDK in your application.\n";
