<?php

require_once __DIR__ . '/../vendor/autoload.php';

use Intufind\AI\Client;
use Intufind\AI\Config\Configuration;
use Intufind\AI\Entities\StreamingChunk;
use Intufind\AI\Exceptions\ApiException;

// Configuration with streaming endpoint
$config = new Configuration([
    'api_endpoint' => 'https://your-api-gateway-endpoint.com/Prod',
    'streaming_endpoint' => 'https://streaming.intufind.com',
    'license_key' => 'your-license-key-here',
    'debug' => true,
]);

$client = new Client($config);

try {
    echo "🚀 Starting streaming chat example...\n\n";

    // Test connection first
    if (!$client->testConnection()) {
        throw new Exception("Cannot connect to API");
    }

    $threadId = 'streaming-demo-' . time();
    echo "Thread ID: {$threadId}\n\n";

    // Example 1: Basic streaming chat
    echo "=== Example 1: Basic Streaming Chat ===\n";
    echo "User: Tell me about wireless headphones\n";
    echo "AI: ";

    $client->chat()->sendStreaming([
        'message' => 'Tell me about wireless headphones',
        'threadId' => $threadId
    ], function ($chunk) {
        if (isset($chunk['type'])) {
            switch ($chunk['type']) {
                case 'text_delta':
                    // Streaming text content - data is a direct string
                    echo is_string($chunk['data']) ? $chunk['data'] : '';
                    break;
                case 'product':
                    // Product streamed - show product name
                    echo "\n📦 Product: " . ($chunk['data']['name'] ?? 'Unknown') . "\n";
                    break;
                case 'post':
                    // Post streamed - show post title
                    echo "\n📄 Post: " . ($chunk['data']['title'] ?? 'Unknown') . "\n";
                    break;
                case 'post_delta':
                    // Post summary is being streamed - append to existing post
                    echo ($chunk['data']['delta'] ?? '');
                    break;
                case 'prompts':
                    // Suggested prompts
                    if (!empty($chunk['data'])) {
                        echo "\n💡 Suggestions:\n";
                        foreach ($chunk['data'] as $prompt) {
                            echo "  • " . ($prompt['prompt'] ?? $prompt['title'] ?? '') . "\n";
                        }
                    }
                    break;
                case 'domain_offer':
                    // Domain offer (e.g., live agent handoff)
                    echo "\n🤝 " . ($chunk['data']['message'] ?? 'An action is available') . "\n";
                    if (!empty($chunk['data']['ui']['actions'])) {
                        echo "Options:\n";
                        foreach ($chunk['data']['ui']['actions'] as $action) {
                            echo "  - " . ($action['label'] ?? $action['id']) . "\n";
                        }
                    }
                    break;
                case 'domain_offer_success':
                    // Domain offer accepted
                    $domain = $chunk['data']['domain'] ?? 'unknown';
                    if ($domain === 'handoff' && !empty($chunk['data']['agent'])) {
                        echo "\n✅ Connected to " . ($chunk['data']['agent']['name'] ?? 'a live agent') . "\n";
                    } else {
                        echo "\n✅ Action completed for domain: {$domain}\n";
                    }
                    break;
                case 'error':
                    echo "\n❌ Error: " . ($chunk['data']['error'] ?? 'Unknown error') . "\n";
                    break;
                case 'complete':
                    echo "\n✅ Stream completed\n\n";
                    break;
                case 'progress':
                    // Optional: show progress indicators
                    // echo "⏳ " . ($chunk['data']['message'] ?? '') . "\n";
                    break;
            }
        }
        
        // Flush output immediately
        flush();
    });

    // Example 2: Streaming with product context
    echo "=== Example 2: Streaming with Product Filter ===\n";
    echo "User: What headphones do you recommend for gaming?\n";
    echo "AI: ";

    $client->chat()->sendStreaming([
        'message' => 'What headphones do you recommend for gaming?',
        'threadId' => $threadId,
        'productFilter' => [
            'categories' => ['electronics', 'gaming'],
            'price_max' => 300
        ]
    ], function ($chunk) {
        if (isset($chunk['type'])) {
            switch ($chunk['type']) {
                case 'text_delta':
                    echo is_string($chunk['data']) ? $chunk['data'] : '';
                    break;
                case 'product':
                    echo "\n📦 Product: " . ($chunk['data']['name'] ?? 'Unknown') . "\n";
                    break;
                case 'post':
                    echo "\n📄 " . ($chunk['data']['title'] ?? '') . "\n";
                    break;
                case 'post_delta':
                    echo ($chunk['data']['delta'] ?? '');
                    break;
                case 'prompts':
                    if (!empty($chunk['data'])) {
                        echo "\n💡 Try asking:\n";
                        foreach ($chunk['data'] as $prompt) {
                            echo "  • " . ($prompt['prompt'] ?? '') . "\n";
                        }
                    }
                    break;
                case 'domain_offer':
                    echo "\n🤝 " . ($chunk['data']['message'] ?? '') . "\n";
                    break;
                case 'error':
                    echo "\n❌ Error: " . ($chunk['data']['error'] ?? 'Unknown error') . "\n";
                    break;
                case 'complete':
                    echo "\n✅ Stream completed\n\n";
                    break;
            }
        }
        flush();
    });

    // Example 3: Using the generator approach
    echo "=== Example 3: Generator-based Streaming ===\n";
    echo "User: Compare different types of headphones\n";
    echo "AI: ";

    try {
        $generator = $client->chat()->sendStreamingGenerator([
            'message' => 'Compare different types of headphones',
            'threadId' => $threadId
        ]);

        foreach ($generator as $chunk) {
            if (isset($chunk['type'])) {
                switch ($chunk['type']) {
                    case 'text_delta':
                        echo is_string($chunk['data']) ? $chunk['data'] : '';
                        break;
                    case 'product':
                        echo "\n📦 " . ($chunk['data']['name'] ?? '') . "\n";
                        break;
                    case 'post':
                        echo "\n📄 " . ($chunk['data']['title'] ?? '') . "\n";
                        break;
                    case 'post_delta':
                        echo ($chunk['data']['delta'] ?? '');
                        break;
                    case 'prompts':
                        // Suggested prompts available
                        break;
                    case 'domain_offer':
                        echo "\n🤝 " . ($chunk['data']['message'] ?? '') . "\n";
                        break;
                }
                flush();
            }
        }
        echo "\n✅ Generator stream completed\n\n";
    } catch (Exception $e) {
        echo "\n❌ Generator error: " . $e->getMessage() . "\n\n";
    }

    // Example 4: Using typed StreamingChunk objects (cleaner API)
    echo "=== Example 4: Typed Streaming (using StreamingChunk) ===\n";
    echo "User: I'd like to speak with a human please\n";
    echo "AI: ";

    $client->chat()->sendStreamingTyped([
        'message' => 'I\'d like to speak with a human please',
        'threadId' => $threadId,
        'liveAgentEnabled' => true
    ], function ($chunk) {
        // $chunk is a StreamingChunk object with typed helper methods
        if ($chunk->isTextDelta()) {
            echo $chunk->getTextDelta();
        } elseif ($chunk->isProduct()) {
            $product = $chunk->getProductData();
            echo "\n📦 Product: " . ($product['name'] ?? 'Unknown') . "\n";
        } elseif ($chunk->isPost()) {
            $post = $chunk->getPostData();
            echo "\n📄 Post: " . ($post['title'] ?? 'Unknown') . "\n";
        } elseif ($chunk->isPostDelta()) {
            // Streaming post summary update
            $delta = $chunk->getPostDelta();
            echo $delta['delta'] ?? '';
        } elseif ($chunk->isPrompts()) {
            $prompts = $chunk->getPrompts();
            if (!empty($prompts)) {
                echo "\n💡 Suggestions:\n";
                foreach ($prompts as $prompt) {
                    echo "  • " . ($prompt['prompt'] ?? '') . "\n";
                }
            }
        } elseif ($chunk->isDomainOffer()) {
            $offer = $chunk->getDomainOffer();
            echo "\n🤝 " . ($offer['message'] ?? '') . "\n";
            if (!empty($offer['ui']['actions'])) {
                echo "Options:\n";
                foreach ($offer['ui']['actions'] as $action) {
                    $variant = $action['variant'] ?? '';
                    $suffix = $variant === 'primary' ? ' (recommended)' : '';
                    echo "  - " . ($action['label'] ?? '') . "{$suffix}\n";
                }
            }
        } elseif ($chunk->isDomainOfferSuccess()) {
            $success = $chunk->getDomainOfferSuccess();
            if (($success['domain'] ?? '') === 'handoff' && !empty($success['agent'])) {
                echo "\n✅ Connected to " . ($success['agent']['name'] ?? 'a live agent') . "\n";
            } else {
                echo "\n✅ Action completed\n";
            }
        } elseif ($chunk->isError()) {
            echo "\n❌ Error: " . $chunk->getErrorMessage() . "\n";
        } elseif ($chunk->isComplete()) {
            echo "\n✅ Stream completed\n\n";
        }
        flush();
    });

    // Example 5: Interactive chat loop
    echo "=== Example 5: Interactive Chat (type 'quit' to exit) ===\n";
    
    while (true) {
        echo "\nYou: ";
        $userInput = trim(fgets(STDIN));
        
        if (strtolower($userInput) === 'quit') {
            echo "👋 Goodbye!\n";
            break;
        }
        
        if (empty($userInput)) {
            continue;
        }
        
        echo "AI: ";
        
        try {
            $client->chat()->sendStreaming([
                'message' => $userInput,
                'threadId' => $threadId
            ], function ($chunk) {
                if (isset($chunk['type'])) {
                    switch ($chunk['type']) {
                        case 'text_delta':
                            echo is_string($chunk['data']) ? $chunk['data'] : '';
                            break;
                        case 'product':
                            echo "\n📦 " . ($chunk['data']['name'] ?? '') . "\n";
                            break;
                        case 'post':
                            echo "\n📄 " . ($chunk['data']['title'] ?? '') . "\n";
                            break;
                        case 'post_delta':
                            echo ($chunk['data']['delta'] ?? '');
                            break;
                        case 'prompts':
                            if (!empty($chunk['data'])) {
                                echo "\n💡 Try asking:\n";
                                foreach ($chunk['data'] as $prompt) {
                                    echo "  • " . ($prompt['prompt'] ?? '') . "\n";
                                }
                            }
                            break;
                        case 'domain_offer':
                            echo "\n🤝 " . ($chunk['data']['message'] ?? '') . "\n";
                            if (!empty($chunk['data']['ui']['actions'])) {
                                foreach ($chunk['data']['ui']['actions'] as $action) {
                                    echo "  - " . ($action['label'] ?? $action['id']) . "\n";
                                }
                            }
                            break;
                        case 'domain_offer_success':
                            $domain = $chunk['data']['domain'] ?? '';
                            if ($domain === 'handoff' && !empty($chunk['data']['agent'])) {
                                echo "\n✅ Connected to " . ($chunk['data']['agent']['name'] ?? 'agent') . "\n";
                            } else {
                                echo "\n✅ Action completed\n";
                            }
                            break;
                        case 'error':
                            echo "\n❌ Error: " . ($chunk['data']['error'] ?? 'Unknown error') . "\n";
                            return;
                        case 'complete':
                            echo "\n";
                            return;
                    }
                }
                flush();
            });
        } catch (Exception $e) {
            echo "\n❌ Chat error: " . $e->getMessage() . "\n";
        }
    }

    // Get thread history at the end
    echo "\n=== Thread History ===\n";
    try {
        $history = $client->chat()->getThread($threadId);
        if (isset($history['messages'])) {
            foreach ($history['messages'] as $message) {
                echo ($message['role'] ?? 'unknown') . ": " . ($message['content'] ?? '') . "\n";
            }
        }
    } catch (Exception $e) {
        echo "Could not retrieve thread history: " . $e->getMessage() . "\n";
    }

} catch (ApiException $e) {
    echo "❌ API Error: " . $e->getMessage() . "\n";
    echo "Status Code: " . $e->getStatusCode() . "\n";
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}

echo "\n🏁 Streaming chat example completed.\n";
