<?php

declare(strict_types=1);

require_once __DIR__ . '/../vendor/autoload.php';

use Intufind\AI\Client;

/**
 * Webhook Usage Example
 * 
 * This example demonstrates how to:
 * 1. Create webhooks with different authentication methods
 * 2. List and filter webhooks
 * 3. Update webhook configurations
 * 4. Test webhook endpoints
 * 5. Get webhook statistics
 * 6. Use the webhook builder pattern
 */

// Initialize the client
$client = new Client([
    'api_key' => 'your-api-key',
    'workspace_id' => 'your-workspace-id',
    'api_endpoint' => 'http://127.0.0.1:3001', // Local development
]);

try {
    echo "=== Webhook Management Example ===\n\n";

    // 1. Create a basic webhook
    echo "1. Creating a basic webhook...\n";
    $basicWebhook = $client->webhooks()->create([
        'name' => 'Product Updates Webhook',
        'description' => 'Receives notifications when products are created or updated',
        'url' => 'https://your-app.com/webhooks/products',
        'method' => 'POST',
        'events' => ['PRODUCT_CREATED', 'PRODUCT_UPDATED'],
        'secret' => 'your-webhook-secret-key',
        'timeout_ms' => 30000,
    ]);
    
    echo "Created webhook: {$basicWebhook->name} (ID: {$basicWebhook->id})\n";
    echo "Status: {$basicWebhook->status}\n";
    echo "Events: " . implode(', ', $basicWebhook->events) . "\n\n";

    // 2. Create webhook with Bearer authentication using builder pattern
    echo "2. Creating webhook with Bearer auth using builder pattern...\n";
    $authWebhook = $client->webhooks()
        ->builder('Chat Notifications', 'https://your-app.com/webhooks/chat')
        ->description('Receives chat-related notifications')
        ->events(['CHAT_MESSAGE_SENT', 'CHAT_CONVERSATION_STARTED'])
        ->bearerAuth('your-bearer-token')
        ->timeout(45000)
        ->retryPolicy(5, 'exponential', 2000, 120000)
        ->headers([
            'X-Custom-Header' => 'webhook-notification',
            'Content-Type' => 'application/json'
        ])
        ->create();
    
    echo "Created authenticated webhook: {$authWebhook->name} (ID: {$authWebhook->id})\n";
    echo "Auth type: {$authWebhook->authType}\n\n";

    // 3. Create webhook with API key authentication
    echo "3. Creating webhook with API key authentication...\n";
    $apiKeyWebhook = $client->webhooks()
        ->builder('Search Events', 'https://your-app.com/webhooks/search')
        ->description('Receives search-related events')
        ->event('SEARCH_PERFORMED')
        ->event('SEARCH_FAILED')
        ->apiKeyAuth('your-api-key', 'X-API-Key')
        ->filters([
            'data.query' => ['product', 'category'], // Only fire for specific search queries
        ])
        ->create();
    
    echo "Created API key webhook: {$apiKeyWebhook->name} (ID: {$apiKeyWebhook->id})\n\n";

    // 4. List all webhooks
    echo "4. Listing all webhooks...\n";
    $allWebhooks = $client->webhooks()->list();
    echo "Total webhooks: {$allWebhooks->total}\n";
    
    foreach ($allWebhooks->results as $webhook) {
        echo "- {$webhook->name} ({$webhook->status}) - Events: " . implode(', ', $webhook->events) . "\n";
    }
    echo "\n";

    // 5. List webhooks with filters
    echo "5. Listing active webhooks only...\n";
    $activeWebhooks = $client->webhooks()->list([
        'status' => 'active',
        'limit' => 10,
        'sort_by' => 'created_at',
        'sort_order' => 'desc'
    ]);
    echo "Active webhooks: {$activeWebhooks->total}\n\n";

    // 6. Get a specific webhook
    echo "6. Getting webhook details...\n";
    $webhook = $client->webhooks()->getById($basicWebhook->id);
    if ($webhook) {
        echo "Webhook: {$webhook->name}\n";
        echo "URL: {$webhook->url}\n";
        echo "Success rate: " . number_format($webhook->getSuccessRate(), 1) . "%\n";
        echo "Total deliveries: {$webhook->totalDeliveries}\n";
        echo "Is active: " . ($webhook->isActive() ? 'Yes' : 'No') . "\n";
        echo "Has auth: " . ($webhook->hasAuth() ? 'Yes' : 'No') . "\n\n";
    }

    // 7. Update webhook
    echo "7. Updating webhook...\n";
    $updatedWebhook = $client->webhooks()->update($basicWebhook->id, [
        'description' => 'Updated: Receives notifications for all product events',
        'events' => ['PRODUCT_CREATED', 'PRODUCT_UPDATED', 'PRODUCT_DELETED'],
        'timeout_ms' => 60000,
    ]);
    echo "Updated webhook events: " . implode(', ', $updatedWebhook->events) . "\n";
    echo "Updated timeout: {$updatedWebhook->timeoutMs}ms\n\n";

    // 8. Test webhook
    echo "8. Testing webhook...\n";
    $testResult = $client->webhooks()->test($basicWebhook->id, [
        'test_data' => 'This is a test webhook payload',
        'timestamp' => date('c'),
    ]);
    
    echo "Test successful: " . ($testResult->isSuccessful() ? 'Yes' : 'No') . "\n";
    if ($testResult->responseCode) {
        echo "Response code: {$testResult->responseCode}\n";
    }
    echo "Response time: {$testResult->getFormattedResponseTime()}\n";
    if ($testResult->errorMessage) {
        echo "Error: {$testResult->errorMessage}\n";
    }
    echo "\n";

    // 9. Get webhook statistics
    echo "9. Getting webhook statistics...\n";
    $stats = $client->webhooks()->getStats($basicWebhook->id);
    if ($stats) {
        echo "Webhook Statistics:\n";
        echo "- Total deliveries: {$stats->totalDeliveries}\n";
        echo "- Successful: {$stats->successfulDeliveries}\n";
        echo "- Failed: {$stats->failedDeliveries}\n";
        echo "- Success rate: {$stats->getFormattedSuccessRate()}\n";
        echo "- Average response time: " . number_format($stats->avgResponseTimeMs, 0) . "ms\n";
        if ($stats->lastDeliveryAt) {
            echo "- Last delivery: {$stats->lastDeliveryAt}\n";
        }
        echo "\n";
    }

    // 10. Bulk update webhook status
    echo "10. Bulk updating webhook status...\n";
    $bulkResult = $client->webhooks()->bulkUpdateStatus(
        [$basicWebhook->id, $authWebhook->id],
        'inactive'
    );
    echo "Successful updates: " . count($bulkResult['successful']) . "\n";
    echo "Failed updates: " . count($bulkResult['failed']) . "\n\n";

    // 11. Show available events and auth types
    echo "11. Available webhook events:\n";
    $availableEvents = $client->webhooks()->getAvailableEvents();
    foreach ($availableEvents as $event) {
        echo "- {$event}\n";
    }
    echo "\n";

    echo "Available authentication types:\n";
    $authTypes = $client->webhooks()->getAvailableAuthTypes();
    foreach ($authTypes as $type => $description) {
        echo "- {$type}: {$description}\n";
    }
    echo "\n";

    // 12. Clean up - delete test webhooks
    echo "12. Cleaning up test webhooks...\n";
    $client->webhooks()->delete($basicWebhook->id);
    $client->webhooks()->delete($authWebhook->id);
    $client->webhooks()->delete($apiKeyWebhook->id);
    echo "Test webhooks deleted.\n\n";

    echo "=== Webhook example completed successfully! ===\n";

} catch (\Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . " (Line: " . $e->getLine() . ")\n";
    
    if (method_exists($e, 'getContext')) {
        echo "Context: " . json_encode($e->getContext(), JSON_PRETTY_PRINT) . "\n";
    }
}

/**
 * Example webhook endpoint (for testing purposes)
 * 
 * This is what your webhook endpoint might look like:
 */
function exampleWebhookEndpoint()
{
    // Verify webhook signature (if using secret)
    $secret = 'your-webhook-secret-key';
    $payload = file_get_contents('php://input');
    $signature = $_SERVER['HTTP_X_WEBHOOK_SIGNATURE'] ?? '';
    
    if ($signature) {
        $expectedSignature = 'sha256=' . hash_hmac('sha256', $payload, $secret);
        if (!hash_equals($expectedSignature, $signature)) {
            http_response_code(401);
            echo json_encode(['error' => 'Invalid signature']);
            return;
        }
    }
    
    // Parse webhook data
    $data = json_decode($payload, true);
    
    // Process the webhook based on event type
    switch ($data['event_type']) {
        case 'PRODUCT_CREATED':
            // Handle product creation
            error_log("New product created: " . $data['data']['name']);
            break;
            
        case 'CHAT_MESSAGE_SENT':
            // Handle chat message
            error_log("Chat message sent: " . $data['data']['message']);
            break;
            
        default:
            error_log("Unhandled webhook event: " . $data['event_type']);
            break;
    }
    
    // Return success response
    http_response_code(200);
    echo json_encode(['status' => 'success']);
}
