<?php

declare(strict_types=1);

namespace Intufind\AI;

use Intufind\AI\Config\Configuration;
use Intufind\AI\Http\HttpClient;
use Intufind\AI\Services\ProductService;
use Intufind\AI\Services\PostService;
use Intufind\AI\Services\ChatService;
use Intufind\AI\Services\ConfigService;
use Intufind\AI\Services\ProvisioningService;
use Intufind\AI\Services\RecommendationsService;
use Intufind\AI\Services\TaxonomyService;
use Intufind\AI\Services\PromptService;
use Intufind\AI\Services\ThreadService;
use Intufind\AI\Services\FeedbackService;
use Intufind\AI\Services\WebhookService;
use Intufind\AI\Services\TenantService;

class Client
{
    private Configuration $config;
    private HttpClient $httpClient;
    private ?ProductService $productService = null;
    private ?PostService $postService = null;
    private ?ChatService $chatService = null;
    private ?ConfigService $configService = null;
    private ?ProvisioningService $provisioningService = null;
    private ?RecommendationsService $recommendationsService = null;
    private ?TaxonomyService $taxonomyService = null;
    private ?PromptService $promptService = null;
    private ?ThreadService $threadService = null;
    private ?FeedbackService $feedbackService = null;
    private ?WebhookService $webhookService = null;
    private ?TenantService $tenantService = null;

    /**
     * @param Configuration|array<string, mixed> $config
     */
    public function __construct(Configuration|array $config)
    {
        if (is_array($config)) {
            $config = new Configuration($config);
        }

        if (!$config instanceof Configuration) {
            throw new \InvalidArgumentException('Config must be a Configuration instance or array');
        }

        $this->config = $config;
        $this->httpClient = new HttpClient($config);
    }

    public function products(): ProductService
    {
        if ($this->productService === null) {
            $this->productService = new ProductService($this->httpClient, $this->config);
        }
        return $this->productService;
    }

    public function posts(): PostService
    {
        if ($this->postService === null) {
            $this->postService = new PostService($this->httpClient, $this->config);
        }
        return $this->postService;
    }

    public function chat(): ChatService
    {
        if ($this->chatService === null) {
            $this->chatService = new ChatService($this->httpClient, $this->config);
        }
        return $this->chatService;
    }

    public function config(): ConfigService
    {
        if ($this->configService === null) {
            $this->configService = new ConfigService($this->httpClient, $this->config);
        }
        return $this->configService;
    }

    /**
     * Get the provisioning service for license activation/deactivation
     */
    public function provisioning(): ProvisioningService
    {
        if ($this->provisioningService === null) {
            $this->provisioningService = new ProvisioningService($this->httpClient, $this->config);
        }
        return $this->provisioningService;
    }

    public function recommendations(): RecommendationsService
    {
        if ($this->recommendationsService === null) {
            $this->recommendationsService = new RecommendationsService($this->httpClient, $this->config);
        }
        return $this->recommendationsService;
    }

    public function taxonomy(): TaxonomyService
    {
        if ($this->taxonomyService === null) {
            $this->taxonomyService = new TaxonomyService($this->httpClient, $this->config);
        }
        return $this->taxonomyService;
    }

    public function prompts(): PromptService
    {
        if ($this->promptService === null) {
            $this->promptService = new PromptService($this->httpClient, $this->config);
        }
        return $this->promptService;
    }

    public function threads(): ThreadService
    {
        if ($this->threadService === null) {
            $this->threadService = new ThreadService($this->httpClient, $this->config);
        }
        return $this->threadService;
    }

    public function feedback(): FeedbackService
    {
        if ($this->feedbackService === null) {
            $this->feedbackService = new FeedbackService($this->httpClient, $this->config);
        }
        return $this->feedbackService;
    }

    public function webhooks(): WebhookService
    {
        if ($this->webhookService === null) {
            $this->webhookService = new WebhookService($this->httpClient, $this->config);
        }
        return $this->webhookService;
    }

    public function tenant(): TenantService
    {
        if ($this->tenantService === null) {
            $this->tenantService = new TenantService($this->httpClient, $this->config);
        }
        return $this->tenantService;
    }

    public function getConfig(): Configuration
    {
        return $this->config;
    }

    public function getHttpClient(): HttpClient
    {
        return $this->httpClient;
    }

    /**
     * Test API connectivity by checking tenant status
     *
     * @return bool
     */
    public function testConnection(): bool
    {
        try {
            $status = $this->tenant()->getStatus();
            return $status['subscription']['valid'];
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Quick health check - returns tenant status and subscription info
     *
     * @return array
     */
    public function quickHealthCheck(): array
    {
        try {
            $status = $this->tenant()->getStatus();
            return [
                'healthy' => $status['subscription']['valid'],
                'tier' => $status['subscription']['tier'],
                'status' => $status['status'],
                'timestamp' => $status['timestamp'],
            ];
        } catch (\Exception $e) {
            return [
                'healthy' => false,
                'error' => $e->getMessage(),
                'timestamp' => date('c'),
            ];
        }
    }

    public function getVersionInfo(): array
    {
        return [
            'sdk_version' => '1.0.0',
            'php_version' => PHP_VERSION,
            'api_endpoint' => $this->config->getApiEndpoint(),
            'streaming_endpoint' => $this->config->getStreamingEndpoint(),
            'debug_mode' => $this->config->isDebug(),
        ];
    }

    public function withConfig(array $updates): self
    {
        $newConfig = $this->config->withUpdates($updates);
        return new self($newConfig);
    }

    public function withDebug(): self
    {
        return $this->withConfig(['debug' => true]);
    }

    public function withWorkspaceId(string $workspaceId): self
    {
        return $this->withConfig(['workspace_id' => $workspaceId]);
    }

    public function withApiKey(string $apiKey): self
    {
        return $this->withConfig(['api_key' => $apiKey]);
    }

    public function __get(string $name): mixed
    {
        switch ($name) {
            case 'products':
                return $this->products();
            case 'posts':
                return $this->posts();
            case 'chat':
                return $this->chat();
            case 'provisioning':
                return $this->provisioning();
            case 'recommendations':
                return $this->recommendations();
            case 'taxonomy':
                return $this->taxonomy();
            case 'prompts':
                return $this->prompts();
            case 'threads':
                return $this->threads();
            case 'feedback':
                return $this->feedback();
            case 'webhooks':
                return $this->webhooks();
            case 'tenant':
                return $this->tenant();
            default:
                throw new \BadMethodCallException("Property '{$name}' does not exist");
        }
    }
}
