<?php

declare(strict_types=1);

namespace Intufind\AI\Config;

use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

/**
 * Configuration class for the Intucart AI SDK
 *
 * AUTHENTICATION:
 *
 * - `api_key`: REQUIRED for all API calls (chat, search, sync, etc.)
 *   This is the cloud authentication credential returned after activation.
 *   Sent in Authorization: Bearer header.
 *
 * TYPICAL USAGE:
 *
 * // Activate EDD license to get an API key
 * $client = new Client(['api_endpoint' => 'https://api.intufind.com']);
 * $result = $client->provisioning()->activate('your-edd-license-key');
 *
 * // Use the returned API key for subsequent calls
 * $client = new Client([
 *     'api_endpoint' => 'https://api.intufind.com',
 *     'api_key' => $result['apiKey'],
 * ]);
 * $client->chat()->send(...);
 */
class Configuration
{
    /** @var string */
    private $apiEndpoint;

    /** @var string|null */
    private $streamingEndpoint;

    /** @var string API key for cloud authentication */
    private $apiKey;

    /** @var string|null */
    private $workspaceId;

    /** @var int */
    private $timeout;

    /** @var int */
    private $connectTimeout;

    /** @var int */
    private $retryAttempts;

    /** @var bool */
    private $debug;

    /** @var LoggerInterface */
    private $logger;

    /** @var array */
    private $httpClientOptions;

    /** @var string */
    private $userAgent;

    /**
     * @param array $config Configuration options
     */
    public function __construct(array $config = [])
    {
        $this->apiEndpoint = $config['api_endpoint'] ?? '';
        $this->streamingEndpoint = $config['streaming_endpoint'] ?? null;
        $this->apiKey = $config['api_key'] ?? '';
        $this->workspaceId = $config['workspace_id'] ?? null;
        $this->timeout = $config['timeout'] ?? 30;
        $this->connectTimeout = $config['connect_timeout'] ?? 10;
        $this->retryAttempts = $config['retry_attempts'] ?? 3;
        $this->debug = $config['debug'] ?? false;
        $this->logger = $config['logger'] ?? new NullLogger();
        $this->httpClientOptions = $config['http_client_options'] ?? [];
        $this->userAgent = $config['user_agent'] ?? 'Intucart-AI-PHP-SDK/1.0';

        $this->validate();
    }

    /**
     * Validate the configuration
     *
     * @throws \InvalidArgumentException
     */
    private function validate(): void
    {
        if (empty($this->apiEndpoint)) {
            throw new \InvalidArgumentException('API endpoint is required');
        }

        if (!filter_var($this->apiEndpoint, FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException('Invalid API endpoint URL');
        }

        if ($this->streamingEndpoint && !filter_var($this->streamingEndpoint, FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException('Invalid streaming endpoint URL');
        }

        if ($this->timeout <= 0) {
            throw new \InvalidArgumentException('Timeout must be positive');
        }

        if ($this->connectTimeout <= 0) {
            throw new \InvalidArgumentException('Connect timeout must be positive');
        }

        if ($this->retryAttempts < 0) {
            throw new \InvalidArgumentException('Retry attempts must be non-negative');
        }
    }

    public function getApiEndpoint(): string
    {
        return $this->apiEndpoint;
    }

    public function getStreamingEndpoint(): ?string
    {
        return $this->streamingEndpoint;
    }

    public function getApiKey(): string
    {
        return $this->apiKey;
    }

    /**
     * Get the authentication token for API requests
     *
     * @return string API key or empty string if not configured
     */
    public function getAuthToken(): string
    {
        return $this->apiKey;
    }

    /**
     * Check if the configuration has a valid API key for authentication
     *
     * @return bool
     */
    public function hasAuthentication(): bool
    {
        return !empty($this->apiKey);
    }

    public function getWorkspaceId(): ?string
    {
        return $this->workspaceId;
    }

    public function getTimeout(): int
    {
        return $this->timeout;
    }

    public function getConnectTimeout(): int
    {
        return $this->connectTimeout;
    }

    public function getRetryAttempts(): int
    {
        return $this->retryAttempts;
    }

    public function isDebug(): bool
    {
        return $this->debug;
    }

    public function getLogger(): LoggerInterface
    {
        return $this->logger;
    }

    public function getHttpClientOptions(): array
    {
        return $this->httpClientOptions;
    }

    public function getUserAgent(): string
    {
        return $this->userAgent;
    }

    /**
     * Get default headers for API requests
     *
     * @return array
     */
    public function getDefaultHeaders(): array
    {
        $headers = [
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'User-Agent' => $this->userAgent,
        ];

        if (!empty($this->apiKey)) {
            $headers['Authorization'] = 'Bearer ' . trim($this->apiKey);
        }

        if ($this->workspaceId) {
            $headers['X-Workspace-ID'] = $this->workspaceId;
        }

        return $headers;
    }

    /**
     * Convert a URL to a normalized workspace ID
     *
     * Examples:
     *   "https://mystore.com" → "mystore-com"
     *   "my-store.myshopify.com" → "my-store-myshopify-com"
     *   "example.com/path" → "example-com"
     *
     * @param string $url The URL or domain to convert
     * @return string The normalized workspace ID
     */
    public static function workspaceIdFromUrl(string $url): string
    {
        if (empty($url)) {
            return '';
        }

        // Try parsing as URL to extract hostname
        $parsed = parse_url(
            strpos($url, '://') !== false ? $url : 'https://' . $url
        );

        $hostname = $parsed['host'] ?? $url;

        // Remove path if present in hostname fallback
        $hostname = preg_replace('/\/.*$/', '', $hostname) ?? $hostname;

        // Replace dots and non-alphanumeric with hyphens, lowercase
        $workspaceId = preg_replace('/[^a-z0-9-]/i', '-', $hostname) ?? '';
        $workspaceId = preg_replace('/-+/', '-', $workspaceId) ?? $workspaceId; // Collapse multiple hyphens
        $workspaceId = trim($workspaceId, '-'); // Trim leading/trailing hyphens

        return strtolower($workspaceId);
    }

    /**
     * Create a new configuration instance with updated values
     *
     * @param array $updates
     * @return self
     */
    public function withUpdates(array $updates): self
    {
        $config = [
            'api_endpoint' => $this->apiEndpoint,
            'streaming_endpoint' => $this->streamingEndpoint,
            'api_key' => $this->apiKey,
            'workspace_id' => $this->workspaceId,
            'timeout' => $this->timeout,
            'connect_timeout' => $this->connectTimeout,
            'retry_attempts' => $this->retryAttempts,
            'debug' => $this->debug,
            'logger' => $this->logger,
            'http_client_options' => $this->httpClientOptions,
            'user_agent' => $this->userAgent,
        ];

        return new self(array_merge($config, $updates));
    }
}
