<?php

declare(strict_types=1);

namespace Intufind\AI\Entities;

/**
 * Structured chat message containing AI response and optional data
 */
class ChatMessage
{
    public ?string $intro;
    public ?array $products;
    public ?array $posts;
    public ?array $prompts;
    // Order-specific fields (when order service agent responds)
    public ?array $orders;
    public ?string $order_summary;
    // Domain offer (server-driven UI for offers like handoff)
    public ?DomainOffer $domainOffer;

    public static function fromArray(array $data): self
    {
        $message = new self();
        $message->intro = $data['intro'] ?? null;
        $message->products = $data['products'] ?? null;
        $message->posts = $data['posts'] ?? null;
        $message->prompts = $data['prompts'] ?? null;
        $message->orders = $data['orders'] ?? null;
        $message->order_summary = $data['order_summary'] ?? null;
        $message->domainOffer = null;
        return $message;
    }

    public function getIntro(): ?string
    {
        return $this->intro;
    }

    public function getProducts(): ?array
    {
        return $this->products;
    }

    public function getPosts(): ?array
    {
        return $this->posts;
    }

    public function getPrompts(): ?array
    {
        return $this->prompts;
    }

    public function getOrders(): ?array
    {
        return $this->orders;
    }

    public function getOrderSummary(): ?string
    {
        return $this->order_summary;
    }

    public function hasProducts(): bool
    {
        return !empty($this->products);
    }

    public function hasPosts(): bool
    {
        return !empty($this->posts);
    }

    public function hasPrompts(): bool
    {
        return !empty($this->prompts);
    }

    public function hasOrders(): bool
    {
        return !empty($this->orders);
    }

    public function hasDomainOffer(): bool
    {
        return $this->domainOffer !== null;
    }

    public function getDomainOffer(): ?DomainOffer
    {
        return $this->domainOffer;
    }

    /**
     * Set domain offer from streaming chunk data
     */
    public function setDomainOffer(array $data): void
    {
        $this->domainOffer = DomainOffer::fromArray($data);
    }
}
