<?php

declare(strict_types=1);

namespace Intufind\AI\Entities;

class ChatResponse
{
    public ChatMessage $message;
    public ?string $threadId = null;
    public ?string $lastAgent = null;
    public ?int $turnCount = null;
    public ?bool $finished = null;

    /** @var array<string, mixed> Metadata from cloud response (usage, tier limits, etc.) */
    private array $metadata = [];

    public static function fromArray(array $data): self
    {
        $response = new self();
        $messageData = $data['message'] ?? [];

        // Handle both structured object and legacy string format
        if (is_string($messageData)) {
            $response->message = new ChatMessage();
            $response->message->intro = $messageData;
        } else {
            $response->message = ChatMessage::fromArray($messageData);
        }

        // Extract additional response fields
        $response->threadId = $data['threadId'] ?? null;
        $response->lastAgent = $data['lastAgent'] ?? null;
        $response->turnCount = isset($data['turnCount']) ? (int) $data['turnCount'] : null;
        $response->finished = isset($data['finished']) ? (bool) $data['finished'] : null;

        // Store metadata fields (usage, tier limits, etc.)
        $metadataFields = ['usage', 'tierLimits', 'rateLimits', 'quotas', 'resetTimes'];
        foreach ($metadataFields as $field) {
            if (isset($data[$field])) {
                $response->metadata[$field] = $data[$field];
            }
        }

        return $response;
    }

    public function getMessage(): ChatMessage
    {
        return $this->message;
    }

    public function getThreadId(): ?string
    {
        return $this->threadId;
    }

    public function getLastAgent(): ?string
    {
        return $this->lastAgent;
    }

    public function getTurnCount(): ?int
    {
        return $this->turnCount;
    }

    public function isFinished(): ?bool
    {
        return $this->finished;
    }

    /**
     * Get a metadata value by key
     *
     * @param string $key Metadata key (e.g., 'usage', 'tierLimits')
     * @param mixed $default Default value if key not found
     * @return mixed
     */
    public function getMetadataValue(string $key, $default = null)
    {
        return $this->metadata[$key] ?? $default;
    }

    /**
     * Get all metadata
     *
     * @return array<string, mixed>
     */
    public function getMetadata(): array
    {
        return $this->metadata;
    }

    /**
     * Convenience method to get the intro text directly
     */
    public function getIntro(): ?string
    {
        return $this->message->getIntro();
    }

    /**
     * Convenience method to get products directly
     */
    public function getProducts(): ?array
    {
        return $this->message->getProducts();
    }

    /**
     * Convenience method to get posts directly
     */
    public function getPosts(): ?array
    {
        return $this->message->getPosts();
    }

    /**
     * Convenience method to get prompts directly
     */
    public function getPrompts(): ?array
    {
        return $this->message->getPrompts();
    }
}
