<?php

declare(strict_types=1);

namespace Intufind\AI\Entities;

/**
 * Color scheme entity containing light and dark mode color palettes
 */
class ColorScheme
{
    /** @var array Light mode colors */
    public $light;

    /** @var array Dark mode colors */
    public $dark;

    /** @var string AI reasoning for color choices */
    public $reasoning;

    /**
     * Create a ColorScheme from API response data
     *
     * @param array $data
     * @return self
     */
    public static function fromArray(array $data): self
    {
        $colorScheme = new self();

        $colorScheme->light = $data['light'] ?? [];
        $colorScheme->dark = $data['dark'] ?? [];
        $colorScheme->reasoning = $data['reasoning'] ?? '';

        return $colorScheme;
    }

    /**
     * Get light mode colors
     *
     * @return array
     */
    public function getLight(): array
    {
        return $this->light;
    }

    /**
     * Get dark mode colors
     *
     * @return array
     */
    public function getDark(): array
    {
        return $this->dark;
    }

    /**
     * Get AI reasoning for color choices
     *
     * @return string
     */
    public function getReasoning(): string
    {
        return $this->reasoning;
    }

    /**
     * Get a specific light mode color by key
     *
     * @param string $key Color key (e.g., 'primary', 'accent')
     * @return string|null Hex color code or null if not found
     */
    public function getLightColor(string $key): ?string
    {
        return $this->light[$key] ?? null;
    }

    /**
     * Get a specific dark mode color by key
     *
     * @param string $key Color key (e.g., 'primary', 'accent')
     * @return string|null Hex color code or null if not found
     */
    public function getDarkColor(string $key): ?string
    {
        return $this->dark[$key] ?? null;
    }

    /**
     * Convert to array
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            'light' => $this->light,
            'dark' => $this->dark,
            'reasoning' => $this->reasoning,
        ];
    }
}
