<?php

declare(strict_types=1);

namespace Intufind\AI\Entities;

/**
 * Post entity
 */
class Post
{
    /** @var string */
    public $id;

    /** @var string */
    public $title;

    /** @var string|null */
    public $content;

    /** @var string|null */
    public $excerpt;

    /** @var string|null */
    public $searchableText;

    /** @var string|null */
    public $postType;

    /** @var string|null */
    public $url;

    /** @var string|null */
    public $status;

    /** @var string|null */
    public $authorId;

    /** @var string|null */
    public $authorName;

    /** @var int|null */
    public $wordCount;

    /** @var int|null */
    public $readingTime;

    /** @var array */
    public $categories = [];

    /** @var array */
    public $tags = [];

    /** @var string|null */
    public $metaTitle;

    /** @var string|null */
    public $metaDescription;

    /** @var string|null */
    public $featuredImage;

    /** @var string|null */
    public $createdAt;

    /** @var string|null */
    public $updatedAt;

    /** @var string|null */
    public $publishedAt;

    /** @var array|null */
    public $attributes;

    /** @var array|null */
    public $metadata;

    /** @var string|null Document source (wordpress, shopify, manual, api) */
    public $source;

    /** @var float|null Search score */
    public $score;

    /** @var array|null Matching text chunks from search */
    public $matchingChunks;

    /**
     * Create a Post from API response data
     *
     * @param array $data
     * @return self
     */
    public static function fromArray(array $data): self
    {
        $post = new self();

        $post->id = $data['id'] ?? '';
        $post->title = $data['title'] ?? '';
        $post->content = $data['content'] ?? null;
        $post->excerpt = $data['excerpt'] ?? null;
        $post->searchableText = $data['searchableText'] ?? null;
        $post->postType = $data['post_type'] ?? null;
        $post->url = $data['url'] ?? null;
        $post->status = $data['status'] ?? null;
        $post->authorId = $data['author_id'] ?? null;
        $post->authorName = $data['author_name'] ?? null;
        $post->wordCount = isset($data['word_count']) ? (int) $data['word_count'] : null;
        $post->readingTime = isset($data['reading_time']) ? (int) $data['reading_time'] : null;
        $post->categories = $data['categories'] ?? [];
        $post->tags = $data['tags'] ?? [];
        $post->metaTitle = $data['meta_title'] ?? null;
        $post->metaDescription = $data['meta_description'] ?? null;
        $post->featuredImage = $data['featured_image'] ?? null;
        $post->createdAt = $data['created_at'] ?? null;
        $post->updatedAt = $data['updated_at'] ?? null;
        $post->publishedAt = $data['published_at'] ?? null;
        $post->attributes = $data['attributes'] ?? null;
        $post->metadata = $data['metadata'] ?? null;
        $post->source = $data['source'] ?? null;
        $post->score = isset($data['score']) ? (float) $data['score'] : null;
        $post->matchingChunks = $data['matching_chunks'] ?? null;

        return $post;
    }

    /**
     * Convert the post to an array for API requests
     *
     * @return array
     */
    public function toArray(): array
    {
        $data = array_filter([
            'id' => $this->id,
            'title' => $this->title,
            'content' => $this->content,
            'excerpt' => $this->excerpt,
            'searchableText' => $this->searchableText,
            'post_type' => $this->postType,
            'url' => $this->url,
            'status' => $this->status,
            'author_id' => $this->authorId,
            'author_name' => $this->authorName,
            'categories' => $this->categories,
            'tags' => $this->tags,
            'featured_image' => $this->featuredImage,
            'published_at' => $this->publishedAt,
            'attributes' => $this->attributes,
            'metadata' => $this->metadata,
            'source' => $this->source,
        ], function ($value) {
            return $value !== null;
        });

        return $data;
    }

    /**
     * Check if the post is published
     *
     * @return bool
     */
    public function isPublished(): bool
    {
        return $this->status === 'publish';
    }

    /**
     * Check if the post is a draft
     *
     * @return bool
     */
    public function isDraft(): bool
    {
        return $this->status === 'draft';
    }

    /**
     * Get estimated reading time in minutes
     *
     * @return int
     */
    public function getReadingTime(): int
    {
        if ($this->readingTime !== null) {
            return $this->readingTime;
        }

        // Estimate based on word count (average 200 words per minute)
        if ($this->wordCount !== null) {
            return max(1, (int) ceil($this->wordCount / 200));
        }

        // Estimate based on content length
        if ($this->content) {
            $wordCount = str_word_count(strip_tags($this->content));
            return max(1, (int) ceil($wordCount / 200));
        }

        return 1;
    }

    /**
     * Get a specific attribute value
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function getAttribute(string $key, $default = null)
    {
        return $this->attributes[$key] ?? $default;
    }

    /**
     * Get a specific metadata value
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function getMetadata(string $key, $default = null)
    {
        return $this->metadata[$key] ?? $default;
    }

    /**
     * Get the post's SEO title (meta_title or title)
     *
     * @return string
     */
    public function getSeoTitle(): string
    {
        return $this->metaTitle ?: $this->title;
    }

    /**
     * Get the post's SEO description (meta_description or excerpt)
     *
     * @return string|null
     */
    public function getSeoDescription(): ?string
    {
        return $this->metaDescription ?: $this->excerpt;
    }
}
